package ru.yandex.mts_api_proxy.config;

import java.time.Duration;

import javax.annotation.Nonnull;

import org.apache.http.HttpHost;

import ru.yandex.http.config.AbstractHttpHostConfigBuilder;
import ru.yandex.http.util.HttpHostParser;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.config.IniConfig;

public abstract class AbstractMtsApiClientConfigBuilder
        <T extends AbstractMtsApiClientConfigBuilder<T>>
        extends AbstractHttpHostConfigBuilder<T>
        implements MtsApiClientConfig {
    @Nonnull
    private static final String CONSUMER_KEY = "consumer-key";
    @Nonnull
    private static final String CONSUMER_SECRET = "consumer-secret";
    @Nonnull
    private static final String CLIENT_ID = "client-id";
    @Nonnull
    private static final String GOZORA_HOST = "mts-api-host";
    @Nonnull
    private static final String TOKEN_RENEWAL_INTERVAL = "token-renewal-interval";
    @Nonnull
    private String consumerKey;
    @Nonnull
    private String consumerSecret;
    @Nonnull
    private String clientId;
    @Nonnull
    private HttpHost mtsApiHost;
    @Nonnull
    private Duration tokenRenewalInterval;

    protected AbstractMtsApiClientConfigBuilder(final MtsApiClientConfig config) {
        super(config);
        consumerKey = config.consumerKey();
        consumerSecret = config.consumerSecret();
        clientId = config.clientId();
        mtsApiHost = config.mtsApiHost();
        tokenRenewalInterval = config.tokenRenewalInterval();
    }

    protected AbstractMtsApiClientConfigBuilder(
            final IniConfig config,
            final MtsApiClientConfig defaults)
            throws ConfigException {
        super(config, defaults);

        consumerKey = config.getString(CONSUMER_KEY);
        consumerSecret = config.getString(CONSUMER_SECRET);
        clientId = config.getString(CLIENT_ID);
        mtsApiHost = config.get(GOZORA_HOST, HttpHostParser.INSTANCE);
        tokenRenewalInterval = Duration.ofMillis(config.getLongDuration(TOKEN_RENEWAL_INTERVAL,
                defaults.tokenRenewalInterval().toMillis()));
    }

    @Override
    @Nonnull
    public String consumerKey() {
        return consumerKey;
    }

    @Nonnull
    public T consumerKey(String consumerKey) {
        this.consumerKey = consumerKey;
        return self();
    }

    @Override
    @Nonnull
    public String consumerSecret() {
        return consumerSecret;
    }

    @Nonnull
    public T consumerSecret(String consumerSecret) {
        this.consumerSecret = consumerSecret;
        return self();
    }

    @Override
    @Nonnull
    public String clientId() {
        return clientId;
    }

    @Nonnull
    public T clientId(String clientId) {
        this.clientId = clientId;
        return self();
    }

    @Override
    @Nonnull
    public HttpHost mtsApiHost() {
        return mtsApiHost;
    }

    @Nonnull
    public T clientId(HttpHost mtsApiHost) {
        this.mtsApiHost = mtsApiHost;
        return self();
    }

    @Override
    @Nonnull
    public Duration tokenRenewalInterval() {
        return tokenRenewalInterval;
    }

    @Nonnull
    public T clientId(Duration tokenRenewalInterval) {
        this.tokenRenewalInterval = tokenRenewalInterval;
        return self();
    }
}

