#!/bin/bash
set -xo pipefail

export LOG_DIR="/logs"
export DATA_DIR="/data"
export CORE_DIR="/cores"

export LOGROTATE_DAYS=1

export ZOO_GEOS=("sas iva myt")
export POD_ID=$YP_POD_ID
export QUEUE_CLIENT_PORT=2222
export QUEUE_HTTP_PORT=2223
export QUEUE_VOTE_PORT=2224
export QUEUE_DATA_PORT=2225
export QUEUE_NANNY_SERVICE_ID=(sarlacc)
export QUEUE_ENDPOINT_SETS_IDS=(sarlacc)
export QUEUE_DEFAULT_DC_PODS_COUNT=3

export SARLACC_PORT=${BSCONFIG_IPORT:-80}
export LUCENE_PORT=${BSCONFIG_IPORT_PLUS_1:-81}
export LUCENE_SEARCHER_PORT=${BSCONFIG_IPORT_PLUS_2:-82}
export LUCENE_INDEXER_PORT=${BSCONFIG_IPORT_PLUS_3:-83}
export LUCENE_DUMP_PORT=${BSCONFIG_IPORT_PLUS_4:-84}
export PRODUCER_PORT=${BSCONFIG_IPORT_PLUS_8:-88}
export CONSUMER_PORT=${BSCONFIG_IPORT_PLUS_9:-89}

[ -z "$YP_TOKEN" -a -e "./yp/token" ] && export YP_TOKEN=$(cat ./yp/token)
[ -n "$MEM_LIMIT" ] && export MEMORY_LIMIT=$(echo $MEM_LIMIT | gawk '{print int($1*0.9)}')
[ -n "$CPU_LIMIT" ] && export CPU_CORES=$(echo $CPU_LIMIT | grep -Eo '[0-9\.]+' | awk '{print int($1)}')

list_backends() {
    pods_count=$1
    shift
    nanny_service_id=$1
    shift
    geo=$1
    shift
    pod_set_id=${nanny_service_id//_/-}
    dc_pods=()
    for endpoint_set_id in $@; do
        dc_pods=( $(yp select-objects --address $geo pod --selector '/status/dns/persistent_fqdn' --filter "[/meta/pod_set_id] = \"$endpoint_set_id\"" --config '{token="$YP_TOKEN";}' | awk '$2 ~ /".*"/ {print $2}' | sed -e 's/"//g' | sort) )
    done
    if [ ${#dc_pods[*]} -eq "0" -o -z "${dc_pods[*]}" ]; then
        dc_pods=( $(yp select-objects --address $geo pod --selector '/status/dns/persistent_fqdn' --filter "[/meta/pod_set_id] = \"$pod_set_id\"" --config '{token="$YP_TOKEN";}' | awk '$2 ~ /".*"/ {print $2}' | sed -e 's/"//g' | sort) )
    fi
    if [ ${#dc_pods[*]} -eq "0" -o -z "${dc_pods[*]}" ]; then
        for i in `seq $pods_count`; do
            echo "${pod_set_id}-${i}.${geo}.yp-c.yandex.net"
        done
    else
        for pod in ${dc_pods[*]}; do
            echo "$pod"
        done
    fi
}

list_zooloosers() {
    default_count=$1
    shift
    queue_nanny_service_id=$1
    shift
    zoo_geos="${ZOO_GEOS[$1]}"
    shift
    for geo in ${zoo_geos[@]}; do
        for pod in $(list_backends $default_count $queue_nanny_service_id $geo $@); do
            echo -n "$pod:$QUEUE_CLIENT_PORT/$QUEUE_HTTP_PORT|"
        done
    done | sed 's/|$//'
}

generate_searchmap_for_service() {
    queue_name=$1
    shift
    shards=$1
    shift
    shards_per_host=$1
    shift
    search_port=$1
    shift
    indexer_port=$1
    shift
    zooloosers=$1
    shift
    first=0
    max_shard=$(( shards - 1 ))
    for host in $@; do
        last=$(( first + shards_per_host - 1 ))
        [ "$last" -gt "$max_shard" ] && last=$max_shard
        echo "$queue_name host:$host,tag:$host,shards:${first}-${last},search_port:$search_port,json_indexer_port:$indexer_port,zk:$zooloosers"
        first=$(( (first + shards_per_host) % shards ))
    done
}

generate_searchmap_for_geo() {
    queue_name=$1
    shift
    geo=$1
    shift
    shards_total=$1
    shift
    shards_per_dc=$1
    shift
    search_port=$1
    shift
    indexer_port=$1
    shift
    zoo_clusters_cnt="${#ZOO_GEOS[@]}"
    shards_per_host=$(( shards_total / shards_per_dc ))
    first_shard=0
    max_shard=$(( shards_total - 1 ))
    zoo_shard_step=$(( shards_per_host / zoo_clusters_cnt))
    for host in $@; do
        zoo_first_shard=$first_shard
        for i in $(seq 0 $(( zoo_clusters_cnt-1 )) ); do
            zoo_geos="${ZOO_GEOS[$i]}"
            zoo_max_shard=$(( zoo_first_shard + zoo_shard_step - 1 ))
            zooloosers=$(list_zooloosers ${QUEUE_DEFAULT_DC_PODS_COUNT:-3} "${QUEUE_NANNY_SERVICE_ID[$i]}" $i "${QUEUE_ENDPOINT_SETS_IDS[$i]}")
            last_shard=$(( zoo_first_shard + zoo_shard_step - 1 ))
            [ "$last_shard" -gt "$max_shard" ] && last_shard=$max_shard
            echo "$queue_name host:$host,tag:$host,shards:${zoo_first_shard}-${last_shard},search_port:$search_port,json_indexer_port:$indexer_port,zk:$zooloosers"
            zoo_first_shard=$(( (zoo_first_shard + zoo_shard_step) % shards_total ))
        done
        first_shard=$(( (first_shard + shards_per_host) % shards_total ))
    done
}
