package ru.yandex.mail.so2.skeleton;

import java.io.IOException;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.LongAdder;
import java.util.function.Consumer;
import java.util.function.Function;

import ru.yandex.collection.Pattern;
import ru.yandex.concurrent.ExecutorServiceCloser;
import ru.yandex.concurrent.LifoWaitBlockingQueue;
import ru.yandex.concurrent.LoggingRejectedExecutionHandler;
import ru.yandex.concurrent.NamedThreadFactory;
import ru.yandex.http.proxy.HttpProxy;
import ru.yandex.http.server.async.DelegatedHttpAsyncRequestHandler;
import ru.yandex.mail.so.factors.extractors.ExtractorStat;
import ru.yandex.mail.so.factors.extractors.ExtractorStater;
import ru.yandex.mail.so.factors.extractors.SoFactorsExtractorModules;
import ru.yandex.mail.so.factors.extractors.SoFactorsExtractorsRegistry;
import ru.yandex.mail.so.factors.samples.SamplesLoader;
import ru.yandex.mail.so.factors.types.SoFactorTypesRegistry;
import ru.yandex.mail.so2.skeleton.config.ImmutableSo2SkeletonConfig;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.stater.LongAdderStater;
import ru.yandex.stater.ThreadPoolStater;

public class So2SkeletonServer<T extends ImmutableSo2SkeletonConfig>
    extends HttpProxy<T>
{
    protected final Map<String, ExtractorStater> extractorStaters =
        new HashMap<>();
    protected final SamplesLoader samplesLoader = new SamplesLoader();
    protected final SoFactorsExtractorModules extractors;
    protected final ThreadPoolExecutor threadPool;

    public So2SkeletonServer(final T config)
        throws ConfigException, IOException
    {
        super(config);

        SoFactorsExtractorsRegistry extractorsRegistry =
            createExtractorsRegistry();
        closeChain.add(extractorsRegistry);

        LongAdderStater violationsStater =
            new LongAdderStater("factors-access-violations_dmmm");
        registerStater(violationsStater);

        LongAdderStater luaErrorsStater =
            new LongAdderStater("lua-errors-stater_dmmm");
        registerStater(luaErrorsStater);

        extractors = new SoFactorsExtractorModules(
            config.extractModulesConfig(),
            violationsStater.adder(),
            extractorsRegistry,
            createExtractorStaterFactory(),
            luaErrorsStater.adder(),
            getThreadGroup(),
            this,
            this,
            logger,
            samplesLoader,
            config.metricsTimeFrame());
        closeChain.add(extractors);

        LongAdder rejectedTasksCounter = new LongAdder();
        threadPool = new ThreadPoolExecutor(
            config.workers(),
            config.workers(),
            1,
            TimeUnit.MINUTES,
            new LifoWaitBlockingQueue<>(config.connections()),
            new NamedThreadFactory(
                getThreadGroup(),
                getName() + "-W-",
                true),
            new LoggingRejectedExecutionHandler(logger, rejectedTasksCounter));
        closeChain.add(new ExecutorServiceCloser(threadPool));
        registerStater(
            new ThreadPoolStater(
                threadPool,
                rejectedTasksCounter,
                "thread-pool-"));

        register(
            new Pattern<>("/add-spam-samples", false),
            new DelegatedHttpAsyncRequestHandler<>(
                new AddSamplesHandler(samplesLoader),
                this,
                threadPool));
    }

    @Override
    public void start() throws IOException {
        threadPool.prestartAllCoreThreads();
        super.start();
    }

    public SoFactorsExtractorModules extractors() {
        return extractors;
    }

    public ThreadPoolExecutor threadPool() {
        return threadPool;
    }

    protected SoFactorsExtractorsRegistry createExtractorsRegistry()
        throws ConfigException
    {
        return new SoFactorsExtractorsRegistry(
            this,
            new SoFactorTypesRegistry());
    }

    protected Function<String, Consumer<ExtractorStat>>
        createExtractorStaterFactory()
    {
        return new ExtractorStaterFactory();
    }

    public class ExtractorStaterFactory
        implements Function<String, Consumer<ExtractorStat>>
    {
        @Override
        public Consumer<ExtractorStat> apply(final String nodeName) {
            ExtractorStater stater = extractorStaters.get(nodeName);
            if (stater == null) {
                String category = "extractors";
                String title = "Extract nodes";
                stater = new ExtractorStater(
                    category,
                    title,
                    nodeName,
                    config.staters().preparedStaters().asterisk().prefix()
                    + "-total_ammm");
                registerStater(stater);
                extractorStaters.put(nodeName, stater);
            }
            return stater.consumer();
        }
    }
}

