package ru.yandex.mail.so2;

import java.io.IOException;
import java.net.InetAddress;
import java.nio.charset.StandardCharsets;
import java.util.Arrays;
import java.util.Locale;
import java.util.Timer;
import java.util.TimerTask;
import java.util.function.BooleanSupplier;

import com.google.protobuf.ByteString;
import com.google.protobuf.util.JsonFormat;
import org.apache.http.HttpException;
import org.apache.http.HttpRequest;
import org.apache.http.HttpStatus;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.entity.ContentType;
import org.apache.http.nio.entity.NStringEntity;
import org.apache.http.nio.protocol.HttpAsyncExchange;
import org.apache.http.nio.protocol.HttpAsyncRequestConsumer;
import org.apache.http.nio.protocol.HttpAsyncRequestHandler;
import org.apache.http.protocol.HttpContext;
import org.apache.http.protocol.HttpCoreContext;
import org.joda.time.DateTimeZone;
import org.joda.time.format.DateTimeFormat;
import org.joda.time.format.DateTimeFormatter;

import ru.yandex.base64.Base64Encoder;
import ru.yandex.charset.Decoder;
import ru.yandex.function.AtomicBooleanSupplier;
import ru.yandex.function.ByteArrayProcessable;
import ru.yandex.function.FalseSupplier;
import ru.yandex.function.Processable;
import ru.yandex.http.proxy.AbstractProxySessionCallback;
import ru.yandex.http.proxy.BasicProxySession;
import ru.yandex.http.proxy.HttpResponseSendingCallback;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.BasicAsyncResponseConsumerFactory;
import ru.yandex.http.util.nio.ByteArrayProcessableAsyncConsumer;
import ru.yandex.http.util.nio.NByteArrayEntityGenerator;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.http.util.server.LoggingServerConnection;
import ru.yandex.io.DecodableByteArrayOutputStream;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonTypeExtractor;
import ru.yandex.mail.so.api.v1.ConnectInfo;
import ru.yandex.mail.so.api.v1.SmtpEnvelope;
import ru.yandex.mail.so.api.v1.SoRequest;
import ru.yandex.mail.so.factors.BasicSoFunctionInputs;
import ru.yandex.mail.so.factors.LoggingFactorsAccessViolationHandler;
import ru.yandex.mail.so.factors.http.HttpSoFactorsExtractorContext;
import ru.yandex.mail.so.factors.types.BinarySoFactorType;
import ru.yandex.mail.so.factors.types.SmtpEnvelopeSoFactorType;
import ru.yandex.parser.mail.envelope.SmtpEnvelopeHolder;
import ru.yandex.parser.mail.errors.ErrorsLogger;
import ru.yandex.parser.uri.PctEncodedString;
import ru.yandex.parser.uri.QueryParameter;
import ru.yandex.parser.uri.ScanningCgiParams;
import ru.yandex.parser.uri.UriParser;
import ru.yandex.util.string.StringUtils;
import ru.yandex.util.timesource.TimeSource;

public class So2HttpHandler
    implements HttpAsyncRequestHandler<ByteArrayProcessable>
{
    private static final DateTimeFormatter FAKE_RECEIVED_DATE_FORMATTER =
        DateTimeFormat.forPattern("EEE, dd MMM yyyy HH:mm:ss Z(zzz)")
            .withLocale(Locale.ENGLISH)
            .withZone(DateTimeZone.forID("Europe/Moscow"));
    private final Timer timer = new Timer("ExecutionLimiter", true);
    private final So2HttpServer server;

    public So2HttpHandler(final So2HttpServer server) {
        this.server = server;
    }

    @Override
    public HttpAsyncRequestConsumer<ByteArrayProcessable> processRequest(
        final HttpRequest request,
        final HttpContext context)
    {
        return new ByteArrayProcessableAsyncConsumer();
    }

    @Override
    public void handle(
        final ByteArrayProcessable body,
        final HttpAsyncExchange exchange,
        final HttpContext context)
        throws HttpException, IOException
    {
        LoggingServerConnection conn =
            (LoggingServerConnection)
                context.getAttribute(HttpCoreContext.HTTP_CONNECTION);
        long requestStartTime;
        if (conn == null) {
            requestStartTime = TimeSource.INSTANCE.currentTimeMillis();
        } else {
            requestStartTime = conn.requestStartTime();
        }
        ProxySession session =
            new BasicProxySession(server, exchange, context);

        String format = session.params().getString("format", null);
        Callback dataCallback;
        FutureCallback<String> finalCallback;
        JsonType jsonType;
        if (session.params().getBoolean("only-so2", false)) {
            dataCallback = null;
            finalCallback = new OnlySo2Callback(session);
            jsonType =
                JsonTypeExtractor.HUMAN_READABLE.extract(session.params());
        } else {
            dataCallback =
                new Callback(session, server, new Data(body, format));
            finalCallback = dataCallback;
            long timeLeft =
                requestStartTime
                + server.config().executionTimeout()
                - TimeSource.INSTANCE.currentTimeMillis();
            session.logger().fine("Only " + timeLeft + " ms left for so2");
            if (timeLeft <= 50L) {
                // Not much we can do here, pass data to sp-daemon immediately
                finalCallback.completed(null);
                return;
            }
            timer.schedule(
                new TimerTask() {
                    @Override
                    public void run() {
                        finalCallback.completed(null);
                    }
                },
                server.config().executionTimeout());
            jsonType = JsonType.NORMAL;
        }

        String extractorName =
            session.params().getString("extractor-name", "main");
        ErrorsLogger errorsLogger = new ErrorsLogger(session.logger());
        SmtpEnvelopeHolder envelope;
        ByteArrayProcessable rawMail;
        if (format == null) {
            envelope = new SmtpEnvelopeHolder(errorsLogger, session.params());
            rawMail = body;
        } else {
            SoRequest request;
            long start = TimeSource.INSTANCE.currentTimeMillis();
            if (format.equals("protobuf")) {
                request = SoRequest.parseFrom(body.content());
            } else if (format.equals("protobuf-json")) {
                Decoder decoder = new Decoder(StandardCharsets.UTF_8);
                body.processWith(decoder);
                SoRequest.Builder requestBuilder = SoRequest.newBuilder();
                JsonFormat.parser().merge(
                    decoder.toString(),
                    requestBuilder);
                request = requestBuilder.build();
            } else {
                throw new BadRequestException("Unknown format: " + format);
            }
            long end = TimeSource.INSTANCE.currentTimeMillis();
            session.logger().info("Request parsed in " + (end - start) + " ms");
            if (dataCallback != null && format.equals("protobuf-json")) {
                DecodableByteArrayOutputStream out =
                    new DecodableByteArrayOutputStream(body.length() * 3 / 4);
                start = TimeSource.INSTANCE.currentTimeMillis();
                request.writeTo(out);
                end = TimeSource.INSTANCE.currentTimeMillis();
                session.logger().info("Recoded in " + (end - start) + " ms");
                dataCallback.data = new Data(out, "protobuf");
            }
            SmtpEnvelope nativeEnvelope = request.getSmtpEnvelope();
            envelope = new SmtpEnvelopeHolder(nativeEnvelope);
            ConnectInfo connectInfo = nativeEnvelope.getConnectInfo();
            StringBuilder sb = new StringBuilder("Received: from ");
            sb.append(connectInfo.getRemoteDomain());
            sb.append(' ');
            sb.append('(');
            String remoteHost = connectInfo.getRemoteHost();
            if (remoteHost.isEmpty()) {
                sb.append("unknown");
            } else {
                sb.append(remoteHost);
                InetAddress addr = envelope.ip();
                if (addr != null) {
                    sb.append(' ');
                    sb.append('[');
                    sb.append(addr.getHostAddress());
                    sb.append(']');
                }
            }
            sb.append(") by cso-yandex.ru; ");
            long timestamp;
            if (nativeEnvelope.hasEmailTimestamp()) {
                timestamp =
                    nativeEnvelope.getEmailTimestamp().getValue() / 1000;
            } else {
                timestamp = TimeSource.INSTANCE.currentTimeMillis();
            }
            FAKE_RECEIVED_DATE_FORMATTER.printTo(sb, timestamp);
            sb.append('\r');
            sb.append('\n');
            String sessionId = session.params().getString("session_id", null);
            if (sessionId != null) {
                sb.append("x-yandex-queueid: ");
                sb.append(sessionId);
                sb.append('\r');
                sb.append('\n');
            }
            byte[] fakeHeader =
                new String(sb).getBytes(StandardCharsets.UTF_8);
            ByteString rawMailBytes = request.getRawEmail();
            int rawMailSize = rawMailBytes.size();
            byte[] data =
                Arrays.copyOf(fakeHeader, fakeHeader.length + rawMailSize);
            rawMailBytes.copyTo(data, fakeHeader.length);
            rawMail = new ByteArrayProcessable(data);
        }
        LoggingFactorsAccessViolationHandler accessViolationHandler =
            new LoggingFactorsAccessViolationHandler(
                server.extractors().violationsCounter(),
                session.logger());
        BooleanSupplier cancellationFlag;
        if (dataCallback == null) {
            cancellationFlag = FalseSupplier.INSTANCE;
        } else {
            cancellationFlag = dataCallback.completed;
        }
        HttpSoFactorsExtractorContext extractorContext =
            new HttpSoFactorsExtractorContext(
                session,
                accessViolationHandler,
                errorsLogger,
                server.threadPool(),
                cancellationFlag);
        server.extractors().extract(
            extractorName,
            extractorContext,
            new BasicSoFunctionInputs(
                accessViolationHandler,
                BinarySoFactorType.RAW_MAIL.createFactor(rawMail),
                SmtpEnvelopeSoFactorType.SMTP_ENVELOPE.createFactor(envelope)),
            jsonType,
            finalCallback);
    }

    private static class Callback
        extends AbstractProxySessionCallback<String>
    {
        private final AtomicBooleanSupplier completed =
            new AtomicBooleanSupplier();
        private final So2HttpServer server;
        private volatile Data data;
        private final String initialFormat;

        Callback(
            final ProxySession session,
            final So2HttpServer server,
            final Data data)
            throws HttpException
        {
            super(session);
            this.server = server;
            this.data = data;
            initialFormat = data.format;
        }

        @Override
        public void cancelled() {
            if (completed.compareAndSet(false, true)) {
                super.cancelled();
            }
        }

        @Override
        public void completed(final String result) {
            if (completed.compareAndSet(false, true)) {
                session.logger().info(
                    "SO2 completed, delegating data to spdaemon");
                Data data = this.data;
                String uri = session.request().getRequestLine().getUri();
                if (initialFormat != null
                    && !initialFormat.equals(data.format))
                {
                    UriParser uriParser = new UriParser(uri);
                    uri =
                        uriParser.addOrReplaceCgiParameter(
                            new QueryParameter(
                                "format",
                                new PctEncodedString(data.format, null)));
                    String oldOutputFormat =
                        new ScanningCgiParams(uriParser.queryParser())
                            .getString("output-format", null);
                    if (oldOutputFormat == null) {
                        uri += "&output-format=" + initialFormat;
                    }
                }
                AsyncClient client =
                    server.spdaemonClient().adjust(session.context());
                BasicAsyncRequestProducerGenerator producerGenerator =
                    new BasicAsyncRequestProducerGenerator(
                        uri,
                        new NByteArrayEntityGenerator(
                            data.data,
                            ContentType.APPLICATION_OCTET_STREAM));
                if (result == null) {
                    session.logger().warning("Null SO2 result");
                } else {
                    Base64Encoder base64 = new Base64Encoder();
                    base64.process(StringUtils.getUtf8Bytes(result));
                    producerGenerator.addHeader(
                        So2HttpServer.SO2_DATA_HEADER,
                        base64.toString());
                }
                client.execute(
                    server.spdaemonHost(),
                    producerGenerator,
                    BasicAsyncResponseConsumerFactory.OK,
                    session.listener().createContextGeneratorFor(client),
                    new HttpResponseSendingCallback(session));
            }
        }

        @Override
        public void failed(final Exception e) {
            if (completed.compareAndSet(false, true)) {
                super.failed(e);
            }
        }
    }

    private static class OnlySo2Callback
        extends AbstractProxySessionCallback<String>
    {
        OnlySo2Callback(final ProxySession session) throws HttpException {
            super(session);
        }

        @Override
        public void completed(final String result) {
            session.response(
                HttpStatus.SC_OK,
                new NStringEntity(
                    result,
                    ContentType.APPLICATION_JSON.withCharset(
                        session.acceptedCharset())));
        }
    }

    private static class Data {
        private final Processable<byte[]> data;
        private final String format;

        Data(final Processable<byte[]> data, final String format) {
            this.data = data;
            this.format = format;
        }
    }
}

