package ru.yandex.mail.so2;

import java.io.IOException;
import java.net.URISyntaxException;
import java.util.function.Consumer;
import java.util.function.Function;

import org.apache.http.HttpException;
import org.apache.http.HttpHost;

import ru.yandex.collection.Pattern;
import ru.yandex.http.config.ImmutableHttpHostConfig;
import ru.yandex.http.server.async.DelegatedHttpAsyncRequestHandler;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.json.parser.JsonException;
import ru.yandex.mail.so.factors.extractors.ExtractorStat;
import ru.yandex.mail.so.factors.extractors.ExtractorStater;
import ru.yandex.mail.so2.config.ImmutableSo2Config;
import ru.yandex.mail.so2.skeleton.So2SkeletonServer;
import ru.yandex.parser.config.ConfigException;

public class So2HttpServer extends So2SkeletonServer<ImmutableSo2Config> {
    public static final String SO2_DATA_HEADER = "X-SO2-Data";

    private final AsyncClient spdaemonClient;
    private final HttpHost spdaemonHost;

    public So2HttpServer(final ImmutableSo2Config config)
        throws ConfigException,
            HttpException,
            IOException,
            JsonException,
            URISyntaxException
    {
        super(config);
        ImmutableHttpHostConfig spdaemonConfig = config.spdaemonConfig();
        spdaemonClient = client("SpDaemon", spdaemonConfig);
        spdaemonHost = spdaemonConfig.host();

        register(
            new Pattern<>("", true),
            new DelegatedHttpAsyncRequestHandler<>(
                new So2HttpHandler(this),
                this,
                threadPool));

        samplesLoader.loadFromFile(config.spamSamplesFile(), logger);
    }

    @Override
    public void start() throws IOException {
        threadPool.prestartAllCoreThreads();
        super.start();
    }

    public AsyncClient spdaemonClient() {
        return spdaemonClient;
    }

    public HttpHost spdaemonHost() {
        return spdaemonHost;
    }

    @Override
    protected Function<String, Consumer<ExtractorStat>>
        createExtractorStaterFactory()
    {
        return new ExtractorStaterFactory();
    }

    private class ExtractorStaterFactory
        implements Function<String, Consumer<ExtractorStat>>
    {
        @Override
        public Consumer<ExtractorStat> apply(final String nodeName) {
            ExtractorStater stater = extractorStaters.get(nodeName);
            if (stater == null) {
                String category;
                String title;
                if (nodeName.contains("ugc")) {
                    category = "ugc";
                    title = "UGC extract";
                } else if (nodeName.contains("dssm")) {
                    category = "distance";
                    title = "DSSM";
                } else if (nodeName.contains("unperson")
                    || nodeName.contains("deobfuscate"))
                {
                    category = "unperson";
                    title = "Unperson and deobfuscate nodes";
                } else if (nodeName.contains("fast_text")) {
                    category = "distance";
                    title = "WMD Distance";
                } else if (nodeName.contains("tikaite")
                    || nodeName.contains("senders")
                    || nodeName.contains("blackbox"))
                {
                    category = "main-extract-nodes";
                    title = "Main extract nodes";
                } else {
                    category = "other-extract-nodes";
                    title = "Other extract nodes";
                }
                stater = new ExtractorStater(
                    category,
                    title,
                    nodeName,
                    config.staters().preparedStaters().asterisk().prefix()
                    + "-total_ammm");
                registerStater(stater);
                extractorStaters.put(nodeName, stater);
            }
            return stater.consumer();
        }
    }
}

