#include <mail/so/api/so_api.pb.h>
#include <mail/so/corp/agent_dialog.h>
#include <mail/so/corp/assassin.h>
#include <mail/so/corp/libdaemon.h>
#include <mail/so/corp/spam.h>
#include <mail/so/corp/trial.h>
#include <mail/so/libs/scheduler/scheduler.h>
#include <mail/so/spamstop/tools/simple_shingler/tsr.h>
#include <mail/so/spamstop/tools/so-clients/activity/tactivityshinglerenv.h>
#include <mail/so/spamstop/tools/so-common/so_answer.h>

#include <library/cpp/blackbox2/blackbox2.h>
#include <library/cpp/getopt/last_getopt.h>
#include <library/cpp/http/misc/parsed_request.h>
#include <library/cpp/http/server/response.h>
#include <library/cpp/monlib/metrics/metric_registry.h>
#include <library/cpp/monlib/encode/spack/spack_v1.h>
#include <library/cpp/string_utils/base64/base64.h>

#include <util/datetime/base.h>
#include <util/generic/iterator_range.h>
#include <util/generic/scope.h>
#include <util/generic/variant.h>
#include <util/stream/tee.h>
#include <util/string/join.h>
#include <util/string/strip.h>
#include <util/string/subst.h>
#include <util/thread/pool.h>

#include <contrib/libs/protobuf/src/google/protobuf/util/json_util.h>

#include "sp-daemon-server.h"

static TSimpleSharedPtr<const NHtmlSanMisc::TAnswer> ExtractSo2Context(
    const THandleContext& ctx,
    const TLog& logger) {
    for (const auto& header: ctx.headers) {
        if (AsciiEqualsIgnoreCase(header.first, "X-SO2-Data")) {
            logger << TLOG_INFO << "SO2 data found";
            try {
                const TString decoded = Base64StrictDecode(header.second);
                logger << TLOG_INFO << "SO2 base64 data decoded";
                if (NJson::TJsonValue value; ReadJsonTree(decoded, &value)) {
                    logger << TLOG_INFO << "SO2 json data parsed";
                    return MakeSimpleShared<NHtmlSanMisc::TAnswer>(NHtmlSanMisc::TAnswer::Parse(std::move(value)));
                } else {
                    logger << TLOG_ERR << "Failed to parse json";
                }
            } catch (...) {
                logger << TLOG_ERR << "Failed to parse SO2 data: " << CurrentExceptionMessageWithBt();
                return nullptr;
            }
        }
    }
    logger << TLOG_ERR << "SO2 data not found";
    return nullptr;
}

static TSimpleSharedPtr<const TActivityShingleRequestVector> ParseActivityInfo(
    TSimpleSharedPtr<const NHtmlSanMisc::TAnswer> so2Context)
{
    if (so2Context && so2Context->GetActivityInfo().Defined()) {
        TActivityShingleRequestVector shlist;
        for (const auto &scheme : so2Context->GetActivityInfo()->GetArraySafe()) {
            if (!scheme.Has("scheme") || !scheme.Has("find")) {
                continue;
            }
            const auto& schemeName = scheme["scheme"].GetStringSafe();

            const bool isActScheme = schemeName == "activity";

            for (const auto &found : scheme["find"].GetArraySafe()) {
                const ui64 uid = found["uid"].GetUIntegerRobust();

                auto reqIt = std::find_if(shlist.begin(), shlist.end(), [uid](const TActivityShingleRequest &d) {
                    return uid == d.Uid();
                });

                if (shlist.end() == reqIt) {
                    reqIt = &shlist.emplace_back(ToString(uid));
                }

                if (isActScheme)
                    reqIt->parseFromActJson(found);
                else
                    reqIt->parseFromComplJson(found);
            }
        }
        return MakeSimpleShared<const TActivityShingleRequestVector>(std::move(shlist));
    } else {
        return nullptr;
    }
}


class TUnistatHandler : public THandlerFunctor {
    const TString UnistatPrefix;
    const size_t MaxThreads;
    const TGlobalContext& GlobalContext;
public:
    explicit TUnistatHandler(const TGlobalContext& globalContext, TString unistatPrefix, size_t maxThreads) noexcept
            : UnistatPrefix(std::move(unistatPrefix))
            , MaxThreads(maxThreads)
            , GlobalContext(globalContext) {
    }

    void Reply(THandleContext&& handleContext, Y_DECLARE_UNUSED void* tsr) override {
        const TString staticStats = [this] {
            NJsonWriter::TBuf json;
            json.BeginList();
            {
                json.BeginList();
                json.WriteString(TStringBuilder{} << UnistatPrefix << "_max-threads_ammm");
                json.WriteInt(MaxThreads);
                json.EndList();
            }
            {
                json.BeginList();
                json.WriteString(TStringBuilder{} << UnistatPrefix << "_threads_ammm");
                json.WriteInt(AtomicGet(GlobalContext.nThreads));
                json.EndList();
            }

            json.EndList();

            return std::move(json.Str());
        }();
        TStringBuf staticStatsView(staticStats);
        staticStatsView.ChopSuffix("]");

        const TString newUnistat = TUnistat::Instance().CreateJsonDump(0, false);
        TStringBuf newUnistatView(newUnistat);
        newUnistatView.SkipPrefix("[");

        THttpResponse(HTTP_OK).SetContent(TStringBuilder{} << staticStatsView << ',' << newUnistatView, "application/json").OutTo(handleContext.output);
    }
};

class TSolomonHandler : public THandlerFunctor {
    void Reply(THandleContext&& handleContext, Y_DECLARE_UNUSED void* tsr) override {
        TStringStream s;
        {
            auto encoder = NMonitoring::EncoderSpackV1(&s, NMonitoring::ETimePrecision::SECONDS, NMonitoring::ECompression::ZSTD);
            NMonitoring::TMetricRegistry::Instance()->Accept(TInstant::Zero(), encoder.Get());
        }
        THttpResponse(HTTP_OK).SetContent(s.Str(), "application/x-solomon-spack").OutTo(handleContext.output);
    }
};

class TReopenGlobalLogHandler : public THandlerFunctor {
    TGlobalContext& GlobalContext;
    THandlerServer& Server;
public:
    explicit TReopenGlobalLogHandler(TGlobalContext& globalContext, THandlerServer& server) noexcept
    : GlobalContext(globalContext)
    , Server(server){}

    void Reply(THandleContext&& handleContext, Y_DECLARE_UNUSED void* tsr) override {
        Server.ReopenLog();
        ReopenLog();
        GlobalContext.Loggers.ReopenLogs();
        handleContext.output << THttpResponse().SetContent("Ok");
    }
};

class TReopenRulesHandler : public THandlerFunctor {
    const TSoConfig& Config;
    TGlobalContext& GlobalContext;
public:
    explicit TReopenRulesHandler(const TSoConfig& config, TGlobalContext& globalContext)
    : Config(config)
    , GlobalContext(globalContext) {}

    void Reply(THandleContext &&handleContext, Y_DECLARE_UNUSED void *tsr) override {
        GlobalContext.RulesHolder = MakeTrueAtomicShared<TRulesHolder>(false,
                                                                Config.dnRules,
                                                                GlobalContext.Loggers,
                                                                Config.PcreSettings,
                                                                Config.HsRulesCache,
                                                                Config.RulesDictPath);
        handleContext.output << THttpResponse().SetContent("Ok");
    }
};

class TReopenModelsHandler : public THandlerFunctor {
    const TSoConfig& Config;
    TGlobalContext& GlobalContext;
public:
    explicit TReopenModelsHandler(const TSoConfig& config, TGlobalContext& globalContext)
    : Config(config)
    , GlobalContext(globalContext) {}

    void Reply(THandleContext &&handleContext, Y_DECLARE_UNUSED void *tsr) override {
        GlobalContext.Models = MakeTrueAtomicShared<TAppliersMap>(LoadModels(Config));
        handleContext.output << THttpResponse().SetContent("Ok");
    }
};

struct TAdditionalInfoFromCgi {
    constexpr static std::array<TStringBuf, 3> HEADERS_KEYS{TStringBuf("X-Yandex-Source"),
                                                            TStringBuf("X-Yandex-Karma"),
                                                            TStringBuf("X-Yandex-Karma-Status")};
    explicit TAdditionalInfoFromCgi(const TCgiParameters& cgiParameters) : Headers(Reserve(HEADERS_KEYS.size())) {
        for(const TStringBuf& key: HEADERS_KEYS) {
            if(const TString* value = MapFindPtr(cgiParameters, key)) {
                Headers.emplace_back(key, *value);
            }
        }

        if(const TString* value = MapFindPtr(cgiParameters, "source")) {
            Headers.emplace_back("X-Yandex-Source", *value);
        }
    }

    TVector<std::pair<TString, TString>> Headers;
};


struct TAntispamHandler : public THandlerFunctor {
    const NUnistat::IHolePtr totalTimings = TUnistat::Instance().DrillHistogramHole("total_check_timings", "hgram", NUnistat::TPriority{0}, xrange(0, 3500, 100));
    const TSoConfig& Config;
    TGlobalContext& GlobalContext;

    explicit TAntispamHandler(const TSoConfig& config, TGlobalContext& globalContext) noexcept
            : Config(config)
            , GlobalContext(globalContext) {
    }

    void Reply(THandleContext&& handleContext, void* tsr) override {
        using namespace mail::so::api::v1;

        const auto start = Now();
        Y_DEFER {
          totalTimings->PushSignal((Now() - start).MilliSeconds());
        };

        const TString qid = handleContext.cgiParameters.Get("session_id");
        if(!qid) {
            throw THttpError(HTTP_BAD_REQUEST) << "session id is absent or empty";
        }

        const TLog logger = [qid](){
            TLog logger = SysLogger();
            logger.SetFormatter([qid](ELogPriority priority, TStringBuf data) {
                return TimePidQidFormatter(priority, data, qid);
            });
            return logger;
        }();

        logger << TLOG_INFO << "Request processing started";

        const auto data = handleContext.ExtractData();

        const TString format = handleContext.cgiParameters.Get("format");
        TString outputFormat = handleContext.cgiParameters.Get("output-format");
        if (outputFormat.empty()) {
            outputFormat = format;
        }

        const SoRequest request = [&format, &data]() {
            SoRequest request;
            if (format == "protobuf-json") {
                if(const google::protobuf::util::Status status = google::protobuf::util::JsonStringToMessage(TString(data.AsCharPtr(), data.Size()), &request);
                   !status.ok()) {
                    throw THttpError(HTTP_BAD_REQUEST) << "cannot parse binary protobuf " << status.ToString();
                }
                return request;
            }

            if(format == "protobuf") {
                if(!request.ParseFromArray(data.Data(), data.Size())) {
                    throw THttpError(HTTP_BAD_REQUEST) << "cannot parse binary protobuf";
                }
                return request;
            }

            throw THttpError(HTTP_BAD_REQUEST) << "invalid format value: expected protobuf, protobuf-json, got: " << format;
        }();
        SoRequest shortRequest(request);
        shortRequest.clear_raw_email();
        NProtoBuf::string logMessage;
        NProtoBuf::util::JsonOptions options;
        options.preserve_proto_field_names = true;
        NProtoBuf::util::MessageToJsonString(shortRequest, &logMessage, options);
        logger << TLOG_INFO << "Request parsed: " << logMessage;

        const SmtpEnvelope& envelope = request.smtp_envelope();

        if(envelope.recipients().empty()) {
            throw THttpError(HTTP_BAD_REQUEST) << "zero recipients";
        }

        TString mailfrom;
        if (envelope.has_mail_from()) {
            mailfrom = envelope.mail_from().address().email();
            if (envelope.mail_from().has_uid()) {
                if (mailfrom) {
                    mailfrom += TStringBuf(" id=");
                } else {
                    mailfrom = TStringBuf("id=");
                }
                mailfrom += ToString(envelope.mail_from().uid().value());
            }
        } else if (envelope.email_type() == EMAIL_TYPE_REGULAR) {
            throw THttpError(HTTP_BAD_REQUEST) << "mailfrom must be specified for regular emails";
        }

        const ConnectInfo& connectInfo = envelope.connect_info();
        const TString& helo = connectInfo.remote_domain();

        const size_t originalSize = request.raw_email().size();

        auto& executionContext = *reinterpret_cast<TMultiContext::TExecutionContext*>(tsr);

        {
            TVector<TString> rcpttos(Reserve(envelope.recipients().size()));

            for (const EmailInfo &rawRcpt: envelope.recipients()) {
                TString rcpt = rawRcpt.address().email();
                if(rawRcpt.has_uid()) {
                    rcpt += " uid=" + ToString(rawRcpt.uid().value());
                }
                if(rawRcpt.has_suid()) {
                    rcpt += " id=" + ToString(rawRcpt.suid().value());
                }
                if(rawRcpt.has_karma()) {
                    rcpt += " karma=" + ToString(rawRcpt.karma().value());
                }
                if(rawRcpt.has_karma_status()) {
                    rcpt += " karma_status=" + ToString(rawRcpt.karma_status().value());
                }
                rcpttos.emplace_back(std::move(rcpt));
                if (rcpttos.size() >= Config.TotalRcptsToProceed)
                    goto RCPTTOS_ENOUGH;
            }

            RCPTTOS_ENOUGH:;

            const auto rcpttoStr = JoinSeq(";", rcpttos);

            const TRemoteHostData remoteHostData(request);

            const TMailConsumer mailConsumer = [&remoteHostData,
                    &request,
                    config = &this->Config,
                    &helo,
                    originalSize]() {
                TMailConsumer mailConsumer(config->MailTruncateSize, config->MailProceedSize, config->HeaderMaxSize);
                if (remoteHostData.RemoteHost && config->add_rcvd) {
                    mailConsumer << TFakeRcvdMaker(remoteHostData.RemoteHost, helo, remoteHostData.RemoteIp);
                }
                if (originalSize) {
                    mailConsumer << "X-Original-Size: " << originalSize << "\r\n";
                }
                if (remoteHostData.FRNR) {
                    mailConsumer << "IY-FRNR: 1\r\n";
                }

                for (const TString &header: request.additional_headers()) {
                    mailConsumer << header;
                    if (header.EndsWith('\n')) {
                        mailConsumer << '\n';
                    }
                }

                mailConsumer << request.raw_email();
                return mailConsumer;
            }();

            const TSimpleSharedPtr<const NHtmlSanMisc::TAnswer> so2Context = ExtractSo2Context(handleContext, logger);

            const TSimpleSharedPtr<const TActivityShingleRequestVector> activityInfo =
                ParseActivityInfo(so2Context);

            const bool dry = request.dry_run();
            TSessionCache cache;

            TVector<TAgentDialogArtifact> artifacts;
            if (Config.fWebMail) {
                TAgentDialogArtifact artifact(remoteHostData, mailConsumer, cache, dry);
                artifact.sIP = remoteHostData.RemoteIp;
                artifact.qID = qid;
                artifact.sHeloHost = helo;
                artifact.sSender = mailfrom;
                artifact.sRecip = rcpttoStr;
                artifact.sOriginalRecip = rcpttoStr;
                artifact.So2Context = so2Context;
                artifact.ActivityInfo = activityInfo;
                artifact.originalSize = originalSize;
                artifact.CheckRecipCount = rcpttos.size();

                artifacts.emplace_back(std::move(artifact));
            } else {
                constexpr size_t MAX_RCPTTOS = 5;

                TAgentDialogArtifact* lastArtifact{};
                for (size_t i = 0; i < std::min(rcpttos.size(), MAX_RCPTTOS); i++) {
                    TAgentDialogArtifact artifact(remoteHostData, mailConsumer, cache, dry);
                    artifact.sIP = remoteHostData.RemoteIp;
                    artifact.qID = qid;
                    artifact.sHeloHost = helo;
                    artifact.sSender = mailfrom;
                    artifact.sRecip = rcpttos[i];
                    artifact.sOriginalRecip = rcpttoStr;
                    artifact.So2Context = so2Context;
                    artifact.ActivityInfo = activityInfo;
                    artifact.originalSize = originalSize;
                    artifact.CheckRecipCount = 1;

                    lastArtifact = &artifacts.emplace_back(std::move(artifact));
                }

                if (lastArtifact && rcpttos.size() > MAX_RCPTTOS) {
                    lastArtifact->CheckRecipCount = rcpttos.size() - MAX_RCPTTOS + 1;
                    lastArtifact->sRecip.append(';');
                    lastArtifact->sRecip.append(JoinRange(";", std::next(rcpttos.cbegin(), MAX_RCPTTOS), rcpttos.cend()));
                }
            }

            try {
                const TMaybe<TCheckedMessage> resolutionMB = CheckMessage(
                        executionContext,
                        artifacts,
                        dry,
                        TString{data.AsCharPtr(), data.size()},
                        handleContext.cgiParameters,
                        logger);

                if(!resolutionMB) {
                    ythrow yexception() << "empty resolution; rcpts: " << rcpttos.size();
                }

                const TCheckedMessage& resolution = *resolutionMB;

                if(!dry && resolution.ShouldLogForKaspersky) {
                    NJsonWriter::TBuf json;
                    json << TKasperskyLogFormatter(TStringBuf(data.AsCharPtr(), data.Size()), remoteHostData.RemoteHost, helo, mailfrom, rcpttos);
                    GlobalContext.Loggers.KasperskyLogger << json.Str() << '\n';
                }
                SoResponse response = resolution.AsSoResponse(request.get_delivery_log(), logger);
                NProtoBuf::string responseBody;
                if (outputFormat == "protobuf-json") {
                    NProtoBuf::util::JsonOptions options;
                    options.preserve_proto_field_names = true;
                    NProtoBuf::util::MessageToJsonString(response, &responseBody, options);
                } else { // format == "protobuf"
                    Y_PROTOBUF_SUPPRESS_NODISCARD response.SerializeToString(&responseBody);
                }

                NProtoBuf::string logMessage;
                NProtoBuf::util::JsonOptions options;
                options.preserve_proto_field_names = true;
                NProtoBuf::util::MessageToJsonString(response, &logMessage, options);
                logger << TLOG_INFO << "Complete resolution: " << logMessage;

                handleContext.output << THttpResponse(HTTP_OK) << responseBody;
            } catch (...) {
                logger << TLOG_ERR << "Exception occured: " << CurrentExceptionMessageWithBt();
                NProtoBuf::string logMessage;
                NProtoBuf::util::JsonOptions options;
                options.preserve_proto_field_names = true;
                NProtoBuf::util::MessageToJsonString(request, &logMessage, options);
                logger << TLOG_ERR << "Complete request: " << logMessage;
                throw;
            }
        }
    }

    std::variant<TMaybe<TCheckedMessage>, EFastResolution> Reply(THandleContext& handleContext, TLog logger, void* tsr) const {
        logger << TLOG_INFO << "Request processing started";
        const auto start = Now();
        Y_DEFER {
                    totalTimings->PushSignal((Now() - start).MilliSeconds());
                };

        auto& executionContext = *reinterpret_cast<TMultiContext::TExecutionContext*>(tsr);

        const TAdditionalInfoFromCgi additionalInfoFromCgi(handleContext.cgiParameters);
        {
            TFastChecker fastChecker(GlobalContext);
            TString connect;
            TString ip;
            TString qid;
            if (const TString * v = MapFindPtr(handleContext.cgiParameters, "CONNECT")) {
                connect = *v;
                ip = NAgentDialog::NConnect::ExtractIp(connect);
                qid = NAgentDialog::NConnect::ExtractQID(connect);
            } else if (const TString * v = MapFindPtr(handleContext.cgiParameters, "client_ip")) {
                ip = *v;
                connect = TString("unknown [") + *v + ']';
            }

            TString helo;
            TString mailfrom;
            TStringBuf frm;
            size_t originalSize{};

            fastChecker.CheckConnect(connect, qid, TString{ip}, Config.fWebMail);

            pDaemonLogger->splog(TLOG_INFO, "starts with %s", connect.c_str()); // mail log back capability

            if (fastChecker.iFastCode == EFastResolution::ACCEPT) {
                return *fastChecker.iFastCode;
            }

            for(const TStringBuf field: {"HELO", "host"}) {
                if (const TString *v = MapFindPtr(handleContext.cgiParameters, field)) {
                    helo = NAgentDialog::NHelo::Parse(*v);
                    break;
                }
            }

            if (const TString * v = MapFindPtr(handleContext.cgiParameters, "MAILFROM")) {
                mailfrom = *v;

                frm = NAgentDialog::NMailFrom::ExtractFrm(mailfrom);

                if(auto size = NAgentDialog::NMailFrom::ExtractSize(mailfrom)) {
                    originalSize = *size;
                } else {
                    ythrow TWithBackTrace<yexception>() << "cannot parse original size from " << mailfrom << ';' << handleContext.cgiParameters.QuotedPrint();
                }

                fastChecker.CheckMailFrom(mailfrom, qid, ip, frm, Config.fWebMail);

                if (fastChecker.iFastCode == EFastResolution::ACCEPT) {
                    return *fastChecker.iFastCode;
                }
            } else if (const TString * v = MapFindPtr(handleContext.cgiParameters, "from")) {
                mailfrom = *v;
                if (const TString * v = MapFindPtr(handleContext.cgiParameters, "uid")) {
                    mailfrom += " id=" + *v;
                }
            }

            TVector<TString> rcpttos;
            for(const TStringBuf field: {"to", "RCPTTO"}) {
                for (const auto&[_, rcptto] : MakeIteratorRange(handleContext.cgiParameters.equal_range(field))) {
                    for (auto t : StringSplitter(rcptto).Split(';').SkipEmpty()) {
                        const TStringBuf tok = StripString(t.Token());
                        rcpttos.emplace_back(tok);
                        fastChecker.CheckRcptto(qid, ip, NAgentDialog::NRcptto::ExtractSuid(tok), Config.fWebMail);
                        if (rcpttos.size() >= Config.TotalRcptsToProceed)
                            goto RCPTTOS_ENOUGH;
                    }
                }
            }
            RCPTTOS_ENOUGH:;

            const auto rcpttoStr = JoinSeq(";", rcpttos);

            fastChecker.DoShinglerCheck(logger, mailfrom, rcpttoStr, qid, Config);
            if (fastChecker.iFastCode.Defined()) {
                return *fastChecker.iFastCode;
            }

            const auto data = handleContext.ExtractData();
            const TStringBuf text(data.AsCharPtr(), data.Size());

            const TRemoteHostData remoteHostData(connect);

            const TMailConsumer mailConsumer = [&remoteHostData,
                    &additionalInfoFromCgi,
                    text,
                    config = &this->Config,
                    &helo,
                    &connect,
                    originalSize]() {
                TMailConsumer mailConsumer(config->MailTruncateSize, config->MailProceedSize, config->HeaderMaxSize);
                if(connect && config->add_rcvd) {
                    mailConsumer << TFakeRcvdMaker(remoteHostData.RemoteHost, helo, remoteHostData.RemoteIp);
                }
                if(originalSize) {
                    mailConsumer << "X-Original-Size: " << originalSize << "\r\n";
                }
                if(remoteHostData.FRNR) {
                    mailConsumer << "IY-FRNR: 1\r\n";
                }
                for(const auto& [header, value]: additionalInfoFromCgi.Headers) {
                    mailConsumer << header << ": " << value << "\r\n";
                }
                mailConsumer << text;
                return mailConsumer;
            }();

            const TSimpleSharedPtr<const NHtmlSanMisc::TAnswer> so2Context = ExtractSo2Context(handleContext, logger);

            const TSimpleSharedPtr<const TActivityShingleRequestVector> activityInfo =
                ParseActivityInfo(so2Context);

            if(so2Context) {
                qid = so2Context->GetQueueId();
                logger.SetFormatter([qid](ELogPriority priority, TStringBuf data) {
                    return TimePidQidFormatter(priority, data, qid);
                });
            }

            const bool dry = [cgiParameters = &handleContext.cgiParameters]() -> bool {
                if(auto p = MapFindPtr(*cgiParameters, "dry")) {
                    return FromString(*p);
                }
                return false;
            }();

            TSessionCache cache;
            TVector<TAgentDialogArtifact> artifacts;
            if (Config.fWebMail) {
                TAgentDialogArtifact artifact(remoteHostData, mailConsumer, cache, dry);
                artifact.sIP = ip;
                artifact.qID = qid;
                artifact.sHeloHost = helo;
                artifact.sSender = mailfrom;
                artifact.sRecip = rcpttoStr;
                artifact.sOriginalRecip = rcpttoStr;
                artifact.So2Context = so2Context;
                artifact.ActivityInfo = activityInfo;
                artifact.originalSize = originalSize;

                artifacts.emplace_back(std::move(artifact));
            } else {
                constexpr size_t MAX_RCPTTOS = 5;

                TAgentDialogArtifact* lastArtifact{};
                for (size_t i = 0; i < std::min(rcpttos.size(), MAX_RCPTTOS); i++) {
                    TAgentDialogArtifact artifact(remoteHostData, mailConsumer, cache, dry);
                    artifact.sIP = ip;
                    artifact.qID = qid;
                    artifact.sHeloHost = helo;
                    artifact.sSender = mailfrom;
                    artifact.sRecip = rcpttos[i];
                    artifact.sOriginalRecip = rcpttoStr;
                    artifact.So2Context = so2Context;
                    artifact.ActivityInfo = activityInfo;
                    artifact.originalSize = originalSize;

                    lastArtifact = &artifacts.emplace_back(std::move(artifact));
                }

                if (lastArtifact && rcpttos.size() > MAX_RCPTTOS) {
                    lastArtifact->sRecip.append(';');
                    lastArtifact->sRecip.append(JoinRange(";", std::next(rcpttos.cbegin(), MAX_RCPTTOS), rcpttos.cend()));
                }
            }

            TMaybe<TCheckedMessage> resolution{
                CheckMessage(
                    executionContext,
                    artifacts,
                    dry,
                    mailConsumer.GetFullText(),
                    handleContext.cgiParameters,
                    logger)};

            if(!dry && resolution && resolution->ShouldLogForKaspersky) {
                NJsonWriter::TBuf json;
                json << TKasperskyLogFormatter(text, connect, helo, mailfrom, rcpttos);
                GlobalContext.Loggers.KasperskyLogger << json.Str() << '\n';
            }

            return resolution;
        }
    }
public:
    struct TJson : THandlerFunctor {
        struct TVisitor {
            void operator()(EFastResolution fastResolution) {
                {
                    THttpResponse response(HTTP_OK);
                    response.SetContentType("application/json");
                    HandleContext.output << response;
                }
                NJsonWriter::TBuf buf(NJsonWriter::HEM_DONT_ESCAPE_HTML, &HandleContext.output);

                buf.BeginObject();
                buf.WriteKey("resolution")
                   .WriteString(ToString(fastResolution));
                buf.EndObject();

                HandleContext.output << Flush;
            }
            void operator()(const TMaybe<TCheckedMessage>& resolution) {
                if(!resolution) {
                    ythrow TWithBackTrace<yexception>() << "zero rcpts";
                }

                {
                    THttpResponse response(HTTP_OK);
                    response.SetContentType("application/json");
                    HandleContext.output << response;
                }
                NJsonWriter::TBuf buf(NJsonWriter::HEM_DONT_ESCAPE_HTML, &HandleContext.output);

                buf.BeginObject();
                buf.WriteKey("resolution")
                   .WriteString(IsHam(resolution->spClass) ? "HAM" : "SPAM");
                buf.EndObject();

                HandleContext.output << Flush;
            }

            explicit TVisitor(THandleContext& handleContext) noexcept : HandleContext(handleContext) {}
            THandleContext& HandleContext;
        };

        void Reply(THandleContext&& handleContext, void* tsr) override {
            std::visit(TVisitor(handleContext), Master->Reply(handleContext, SysLogger(), tsr));
        }

        explicit TJson(TAtomicSharedPtr<TAntispamHandler> master) noexcept
        : Master(std::move(master)) {}

        TAtomicSharedPtr<TAntispamHandler> Master;
    };

private:
    TMaybe<TCheckedMessage> CheckMessage(
        TMultiContext::TExecutionContext& executionContext,
        TVector<TAgentDialogArtifact> artifacts,
        bool dry,
        const TString& requestBody,
        const TCgiParameters cgiParameters,
        const TLog& logger) const
    {
        TVector<TCheckedMessage> resolutions(Reserve(artifacts.size()));
        bool mailish = false;
        bool hasResolution = false;
        for (TAgentDialogArtifact& artifact : artifacts) {
            TMaybe<TCheckedMessage> checked =
                run_assassin(
                    Config,
                    GlobalContext,
                    artifact,
                    logger,
                    dry ? nullptr : &std::get<0>(executionContext)->Resource);
            if (checked) {
                mailish |= checked->Mailish;
                if (checked->spClass != TSpClass::UNKNOWN) {
                    hasResolution = true;
                    artifact.PrevSpClass = checked->spClass;
                }
                resolutions.emplace_back(std::move(*checked));
            }
        }

        if (hasResolution && !mailish && GlobalContext.Pools->OcrRequester) {
            NCurl::TSimpleArtifacts ocrArtifacts;
            TMaybe<NCurl::TError> error = GlobalContext.Pools->OcrRequester->Perform(
                ocrArtifacts,
                TRequestClient::TRequest{}
                    .SetRequest("ocr?extractor-name=ocr&" + cgiParameters.Print())
                    .SetData(requestBody));
            TString ocrText;
            if (error) {
                logger << (TLOG_ERR) << __FILE__ << ':' << __LINE__ << " ocr request failed:" << *error;
            } else {
                try {
                    NJson::TJsonValue parsedJson = NJson::ReadJsonTree(&ocrArtifacts.body, true);
                    if (auto it = parsedJson.GetValueByPath("ocr_text")) {
                        ocrText = it->GetStringSafe();
                    }
                } catch (...) {
                    logger << (TLOG_ERR) << __FILE__ << ':' << __LINE__ << " failed to parse json <" << ocrArtifacts.body.Str() << ">: " << CurrentExceptionMessage();
                }
            }

            if (ocrText) {
                resolutions.clear();
                for (TAgentDialogArtifact& artifact: artifacts) {
                    artifact.OcrText = ocrText;
                    TMaybe<TCheckedMessage> checked =
                        run_assassin(
                            Config,
                            GlobalContext,
                            artifact,
                            logger,
                            dry ? nullptr : &std::get<0>(executionContext)->Resource);
                    if(checked) {
                        resolutions.emplace_back(std::move(*checked));
                    }
                }
            }
        }

        TMaybe<TCheckedMessage> resolution;

        for (TCheckedMessage& checkedMessage: resolutions) {
            if(!resolution.Defined()) {
                resolution = std::move(checkedMessage);
            } else {
                resolution->CombineWith(std::move(checkedMessage));
            }
        }
        if (resolution.Defined() && resolution->TempFail) {
            throw THttpError(HTTP_SERVICE_UNAVAILABLE) << "temporary fail requested by rules";
        }
        return resolution;
    }
};

void TSpDaemonServer::Start() {
    if (GlobalContext.Pools) {
        if (GlobalContext.Pools->RblProducerRequester) {
            Scheduler.Add([this]() {
                GlobalContext.Pools->RblProducerRequester->Perform(Server.ContextManager.Get<0>().MakeIpsList());
            }, Config.ClientsConfigs.RblProducerPeriod);
        }

        if (GlobalContext.Pools->UserWeightsNgRequester) {
            GlobalContext.UpUserWeights();
            Scheduler.Add([this]() {
                GlobalContext.UpUserWeights();
            }, Config.UserWeightsFetchingPeriod);
        }
    }
    
    Y_VERIFY(Server.Start());
}

void TSpDaemonServer::Wait() {
    Server.Wait();
}

TSpDaemonServer::TSpDaemonServer(const TSoConfig& config)
: Config(config)
, GlobalContext(config)
, Server(config.ServerOptions)
, Scheduler(GlobalContext.ThreadPool) {

    const TAtomicSharedPtr<TAntispamHandler> antispamHandler = MakeAtomicShared<TAntispamHandler>(config, GlobalContext);
    auto pingHandler = MakeAtomicShared<TPingHandler>();
    Server
            .On("/ping", pingHandler)
            .On("/unistat", MakeAtomicShared<TUnistatHandler>(GlobalContext, config.UnistatPrefix, config.solverThreads))
            .On("/reopen_log", MakeAtomicShared<TReopenGlobalLogHandler>(GlobalContext, Server))
            .On("/reopen_rules", MakeAtomicShared<TReopenRulesHandler>(config, GlobalContext))
            .On("/reopen_models", MakeAtomicShared<TReopenModelsHandler>(config, GlobalContext))
            .On("/stop", MakeAtomicShared<TStopServerHandler>(&Server, pingHandler))
            .On("/v2/antispam", MakeAtomicShared<TAntispamHandler::TJson>(antispamHandler))
            .On("/v3/antispam", antispamHandler)
            .On("/solomon", MakeAtomicShared<TSolomonHandler>())
            ;
}

TSpDaemonServer::~TSpDaemonServer() {
    pDaemonLogger->Closelog();
}
