package ru.yandex.sp_kitsune;


import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import com.google.protobuf.util.JsonFormat;
import org.apache.http.HttpResponse;

import ru.yandex.charset.Decoder;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.CharsetUtils;
import ru.yandex.http.util.nio.BasicAsyncResponseProducerGenerator;
import ru.yandex.io.DecodableByteArrayOutputStream;
import ru.yandex.kitsune.KitsuneHttpServer;
import ru.yandex.kitsune.config.ImmutableKitsuneConfig;
import ru.yandex.mail.so.api.v1.SoResponse;
import ru.yandex.parser.string.ValuesStorage;

public class SpKitsuneHttpServer extends KitsuneHttpServer {

    private final Map<String, TailStater> tailsStaters;

    public SpKitsuneHttpServer(@Nonnull ImmutableKitsuneConfig kitsuneConfig) throws IOException {
        super(kitsuneConfig);

        tailsStaters = new HashMap<>(kitsuneConfig.tails().size());

        for (String tailName : kitsuneConfig.tails().keySet()) {
            final TailStater tailStater = new TailStater(tailName + '_');
            tailsStaters.put(tailName, tailStater);
            registerStater(tailStater);
        }
    }

    @SuppressWarnings("CatchAndPrintStackTrace")
    @Override
    public void compare(@Nonnull ProxySession session,
                        @Nonnull BasicAsyncResponseProducerGenerator head,
                        @Nonnull List<Map.Entry<String, HttpResponse>> tails) {

        final String format;
        final SoResponse headResponse;
        try {
            format = parseFormat(session.params());
            headResponse = parseResponse(format, head.get().generateResponse());
        } catch (FormatNotSpecifiedException | IOException | UnknownFormatException e) {
            session.logger().log(Level.WARNING, "failed to compare(head)", e);
            return;
        }

        for (var entry : tails) {
            final TailStater tailStater = tailsStaters.get(entry.getKey());
            final String tailName = entry.getKey();
            final HttpResponse response = entry.getValue();
            if (response == null) {
                tailStater.requestFailed();
                continue;
            }

            final SoResponse tailResponse;
            try {
                tailResponse = parseResponse(format, response);
            } catch (Exception e) {
                session.logger().log(Level.WARNING, "failed to compare(" + entry.getKey() + ")", e);
                tailStater.parsingFailed();
                continue;
            }

            if (tailResponse.getResolution() != headResponse.getResolution()) {
                tailStater.resolutionMismatched();
                session.logger().warning("resolution mismatch for " + tailName + ", " +
                        tailResponse.getResolution() + " vs " + headResponse.getResolution());
                continue;
            }

            tailStater.ok();
        }
    }

    @Nonnull
    public Map<String, TailStater> tailsStaters() {
        return tailsStaters;
    }

    @Nonnull
    static String parseFormat(@Nonnull ValuesStorage<?> params) throws FormatNotSpecifiedException {
        String format = params.getString("output-format", null);
        if (format != null) {
            return format;
        }
        format = params.getString("format", null);
        if (format != null) {
            return format;
        }
        throw new FormatNotSpecifiedException();
    }

    @Nonnull
    static SoResponse parseResponse(@Nonnull String format, @Nonnull HttpResponse response) throws IOException,
            UnknownFormatException {
        return parseResponse(format, CharsetUtils.toDecodable(response.getEntity()));
    }

    @Nonnull
    static SoResponse parseResponse(@Nullable String format, @Nonnull DecodableByteArrayOutputStream body) throws IOException,
            UnknownFormatException {
        if ("protobuf".equals(format)) {
            return SoResponse.parseFrom(body.toByteArray());
        } else if ("protobuf-json".equals(format)) {
            final Decoder decoder = new Decoder(StandardCharsets.UTF_8);
            body.processWith(decoder);
            final SoResponse.Builder requestBuilder = SoResponse.newBuilder();
            JsonFormat.parser().merge(
                    decoder.toString(),
                    requestBuilder);
            return requestBuilder.build();
        } else {
            throw new UnknownFormatException(format);
        }
    }
}
