package ru.yandex.sp_kitsune;

import java.util.function.BinaryOperator;

import javax.annotation.Nonnull;

import ru.yandex.concurrent.ConcurrentAccumulator;
import ru.yandex.stater.Stater;
import ru.yandex.stater.StatsConsumer;

public class TailStater implements Stater {
    private final String prefix;
    private final ConcurrentAccumulator<TailStat> accumulator =
            new ConcurrentAccumulator<>(
                    TailStatAccumulator.INSTANCE,
                    TailStat.INITIAL);

    public TailStater(@Nonnull String prefix) {
        this.prefix = prefix;
    }

    public void requestFailed() {
        accumulator.accept(TailStat.REQUEST_FAILED);
    }

    public void resolutionMismatched() {
        accumulator.accept(TailStat.RESOLUTION_MISMATCHED);
    }

    public void parsingFailed() {
        accumulator.accept(TailStat.PARSING_FAILED);
    }

    public void ok() {
        accumulator.accept(TailStat.OK);
    }

    @Nonnull
    public TailStat stat() {
        return accumulator.get();
    }

    @Override
    public <E extends Exception> void stats(StatsConsumer<? extends E> statsConsumer) throws E {
        final TailStat stat = accumulator.get();
        statsConsumer.stat(
                prefix + "total_dmmm",
                stat.total);
        statsConsumer.stat(
                prefix + "failed_parsing_dmmm",
                stat.failedParsing);
        statsConsumer.stat(
                prefix + "failed_request_dmmm",
                stat.failedRequest);
        statsConsumer.stat(
                prefix + "mismatched_resolution_dmmm",
                stat.mismatchedResolution);
    }

    public static class TailStat {
        public static final TailStat INITIAL =
                new TailStat(0, 0, 0, 0);
        public static final TailStat OK =
                new TailStat(1, 0, 0, 0);
        public static final TailStat PARSING_FAILED =
                new TailStat(1, 1, 0, 0);
        public static final TailStat REQUEST_FAILED =
                new TailStat(1, 0, 1, 0);
        public static final TailStat RESOLUTION_MISMATCHED =
                new TailStat(1, 0, 0, 1);

        private final long total;
        private final long failedParsing;
        private final long failedRequest;
        private final long mismatchedResolution;

        public TailStat(
                final long total,
                final long failedParsing,
                final long failedRequest,
                final long mismatchedResolution) {
            this.total = total;
            this.failedParsing = failedParsing;
            this.failedRequest = failedRequest;
            this.mismatchedResolution = mismatchedResolution;
        }

        public long total() {
            return total;
        }

        public long failedParsing() {
            return failedParsing;
        }

        public long failedRequest() {
            return failedRequest;
        }

        public long mismatchedResolution() {
            return mismatchedResolution;
        }
    }

    private enum TailStatAccumulator implements BinaryOperator<TailStat> {
        INSTANCE;

        @Override
        public TailStat apply(
                final TailStat lhs,
                final TailStat rhs) {
            return new TailStat(
                    lhs.total + rhs.total,
                    lhs.failedParsing + rhs.failedParsing,
                    lhs.failedRequest + rhs.failedRequest,
                    lhs.mismatchedResolution + rhs.mismatchedResolution);
        }
    }
}
