package ru.yandex.mail.so.logger;

import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.logging.Level;
import java.util.stream.Collectors;

import org.apache.http.HttpEntityEnclosingRequest;
import org.apache.http.HttpException;
import org.apache.http.HttpHeaders;
import org.apache.http.HttpRequest;
import org.apache.http.HttpStatus;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.StringEntity;
import org.apache.http.nio.protocol.HttpAsyncExchange;
import org.apache.http.nio.protocol.HttpAsyncRequestHandler;
import org.apache.http.protocol.HttpContext;

import ru.yandex.data.compressor.CompressorException;
import ru.yandex.data.compressor.DataCompressor;
import ru.yandex.http.proxy.AbstractProxySessionCallback;
import ru.yandex.http.proxy.BasicProxySession;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.BadResponseException;
import ru.yandex.http.util.MultiFutureCallback;
import ru.yandex.http.util.ServerException;
import ru.yandex.http.util.YandexHeaders;
import ru.yandex.http.util.YandexHttpStatus;
import ru.yandex.http.util.nio.AsyncStringConsumer;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.http.util.request.RequestHandlerMapper;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumerFactory;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.JsonType;
import ru.yandex.mail.so.logger.config.ImmutableLogRecordsHandlerConfig;
import ru.yandex.mail.so.logger.config.LogStorageConfig;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.searchmap.User;
import ru.yandex.search.prefix.LongPrefix;
import ru.yandex.search.proxy.universal.PlainUniversalSearchProxyRequestContext;
import ru.yandex.search.proxy.universal.UniversalSearchProxyRequestContext;
import ru.yandex.stater.RequestInfo;
import ru.yandex.stater.RequestsStater;
import ru.yandex.util.timesource.TimeSource;

public abstract class AbstractLogRecordsHandler extends RequestsStater
    implements HttpAsyncRequestHandler<String>
{
    protected final SpLogger spLogger;
    protected final LogStorage<LogRecordContext> logStorage;
    protected final ImmutableLogRecordsHandlerConfig config;
    protected final DataCompressor decompressor;

    @SuppressWarnings("unused")
    private final String path;
    private final RulesStatDatabase<BasicRoutedLogRecordProducer> dbClient;

    protected AbstractLogRecordsHandler(
        final ImmutableLogRecordsHandlerConfig config,
        final SpLogger spLogger,
        final String path)
        throws ConfigException
    {
        super(config);
        this.config = config;
        this.spLogger = spLogger;
        this.path = path;
        try {
            this.decompressor = DataCompressor.valueOf(config.decompression().toUpperCase(Locale.ROOT));
        } catch (IllegalArgumentException e) {
            throw new ConfigException("Decompression method <" + config.decompression() + "> for path <" + path
                + "> is unknown. Available methods: "
                + Arrays.stream(DataCompressor.values()).map(x -> x.name().toLowerCase(Locale.ROOT))
                    .collect(Collectors.joining(", ")) + '.');
        }
        //String nameSuffix = path.replaceAll("/", "-");
        logStorage = spLogger.logStorage(config.logStorage() == null ? NullLogStorage.NULL : config.logStorage());
        if (logStorage == null && config.logStorage() != null
                && spLogger.config().logStoragesConfig().storageConfigs().get(config.logStorage()).type()
                    != LogStorageType.NULL)
        {
            throw new ConfigException("Log storage with name <" + config.logStorage() + ">  for path <" + path
                + "> is not configured");
        }
        dbClient = spLogger.rulesStatDatabase(
            config.rulesStatDatabase() == null ? NullRulesStatDatabase.NULL : config.rulesStatDatabase());
        spLogger.logger().info("AbstractLogRecordsHandler: dbClient=" + dbClient + ", rulesStatDatabase="
            + config.rulesStatDatabase() + ", rulesStatDatabases size="
            + spLogger.config().rulesStatDatabasesConfig().rulesStatDatabases().size());
        if (dbClient == null && config.rulesStatDatabase() != null
                && spLogger.config().rulesStatDatabasesConfig().rulesStatDatabases().size() > 0
                && spLogger.config().rulesStatDatabasesConfig().rulesStatDatabases()
                    .get(config.rulesStatDatabase()).type() != RulesStatDatabaseType.NULL)
        {
            throw new ConfigException("Rules stat's database with name <" + config.rulesStatDatabase()
                + "> for path <" + path + "> is not configured");
        }
    }

    public ImmutableLogRecordsHandlerConfig config() {
        return config;
    }

    public LogStorage<LogRecordContext> logStorage() {
        return logStorage;
    }

    @SuppressWarnings("unused")
    public DataCompressor decompressor() {
        return decompressor;
    }

    public void saveLogRecord(
        final LogRecordContext context,
        final ProxySession session,
        final FutureCallback<Void> callback)
    {
        logStorage.save(context, session, callback);
    }

    public void saveToAuxiliaryStorages(
        final BasicRoutedLogRecordProducer logRecordProducer,
        final ProxySession session,
        final MultiFutureCallback<Void> multiCallback)
    {
        for (String auxiliaryStorage : config.auxiliaryStorages()) {
            spLogger.auxiliaryStorages().get(auxiliaryStorage)
                .save(logRecordProducer, session, multiCallback.newCallback());
        }
    }

    public void saveRulesStat(
        final BasicRoutedLogRecordProducer logRecordProducer,
        final ProxySession session,
        final FutureCallback<Void> callback)
    {
        dbClient.save(logRecordProducer, session, callback);
    }

    public abstract Map<String, Long> getSearchIndexRequest(final ProxySession session) throws BadRequestException;

    public abstract void searchData(
        final LogRecordsContext<?> context,
        final Map<String, Long> queries,    // Map: query -> prefix (may be null)
        final FutureCallback<JsonObject> callback);

    public static void unprefixedSearchData(
        final LogRecordsContext<LogRecordContext> context,
        final String query,
        final FutureCallback<JsonObject> callback)
    {
        context.session().logger().info("unprefixedSearchData: query=" + query);
        context.spLogger().unprefixedParallelRequest(context.session(), query, context.service(), callback);
    }

    public static void prefixedSearchData(
        final LogRecordsContext<LogRecordContext> context,
        final String query,
        final long prefix,
        final FutureCallback<JsonObject> callback)
    {
        final SpLogger spLogger = context.spLogger();
        final User user = new User(context.service(), new LongPrefix(prefix));
        spLogger.logger().info("requestIndexData: query=" + query + ", prefix=" + prefix + ", user=" + user);
        AsyncClient client = spLogger.searchClient().adjust(context.session().context());
        UniversalSearchProxyRequestContext requestContext =
            new PlainUniversalSearchProxyRequestContext(
                user,
                null,
                true,
                client,
                context.session().logger());
        spLogger.sequentialRequest(
            context.session(),
            requestContext,
            new BasicAsyncRequestProducerGenerator(query),
            null,
            true,
            JsonAsyncTypesafeDomConsumerFactory.OK,
            context.session().listener().createContextGeneratorFor(client),
            callback);
    }

    public void stat(final int status, final long startTime) {
        accept(new RequestInfo(TimeSource.INSTANCE.currentTimeMillis(), status, startTime, startTime, 0L, 0L));
    }

    @Override
    public AsyncStringConsumer processRequest(final HttpRequest request, final HttpContext context)
        throws BadRequestException
    {
        if (RequestHandlerMapper.POST.equals(request.getRequestLine().getMethod())
                && !(request instanceof HttpEntityEnclosingRequest))
        {
            throw new BadRequestException("Payload expected");
        }
        return new AsyncStringConsumer();
    }

    @Override
    @SuppressWarnings("FutureReturnValueIgnored")
    public void handle(final String payload, final HttpAsyncExchange exchange, final HttpContext context)
        throws HttpException
    {
        long startTime = TimeSource.INSTANCE.currentTimeMillis();
        ProxySession session = new BasicProxySession(spLogger, exchange, context);
        String method = exchange.getRequest().getRequestLine().getMethod();
        String uri = exchange.getRequest().getRequestLine().getUri();
        session.logger().info("AbstractLogRecordsHandler: type=" + config.type() + ", route=" + config.route()
            + ", storageType=" + logStorage.type() + ", method=" + method + ", uri=" + uri);
        if (RequestHandlerMapper.GET.equals(method)) {
            session.logger().info("Search query=" + exchange.getRequest().getRequestLine().getUri());
            LogRecordsContext<LogRecordContext> logRecordsContext =
                config.type().createLogRecordsContext(spLogger, session, logStorage.type(), config.route());
            Map<String, Long> queries = getSearchIndexRequest(session);
            session.params().add(YandexHeaders.SERVICE, logRecordsContext.service());
            searchData(
                logRecordsContext,
                queries,
                new GetIndexDataCallback(
                    logRecordsContext,
                    new AbstractProxySessionCallback<>(logRecordsContext.session()) {
                        @Override
                        public void completed(final List<LogRecordsBatch<LogRecordContext, LogStorageConfig>> batches) {
                            if (batches == null || batches.isEmpty()) {
                                session.response(HttpStatus.SC_NOT_FOUND);
                            } else {
                                List<String> logRecords = new ArrayList<>();
                                for (LogRecordsBatch<?, ?> batch : batches) {
                                    if (batch != null) {
                                        for (String logRecord : batch.logRecords()) {
                                            if (logRecord != null && !logRecord.isEmpty()) {
                                                logRecords.add(logRecord);
                                            }
                                        }
                                    }
                                }
                                //String body = JsonType.NORMAL.toString(logRecords);
                                String body = String.join("\n", logRecords);
                                //session.logger().info("GetIndexDataCallback.AbstractProxySessionCallback: result='"
                                //    + body + "'");
                                // session.response(HttpStatus.SC_OK, new StringEntity(body, ContentType.APPLICATION_JSON));
                                session.response(HttpStatus.SC_OK, new StringEntity(body, StandardCharsets.UTF_8));
                            }
                            stat(YandexHttpStatus.SC_OK, startTime);
                        }

                        @Override
                        public void cancelled() {
                            stat(YandexHttpStatus.SC_CLIENT_CLOSED_REQUEST, startTime);
                            super.cancelled();
                        }

                        @Override
                        public void failed(final Exception e) {
                            if (e instanceof BadResponseException) {
                                stat(((BadResponseException) e).statusCode(), startTime);
                            } else {
                                stat(YandexHttpStatus.SC_REMOTE_CLOSED_REQUEST, startTime);
                            }
                            super.failed(e);
                        }
                    }));
        } else if (RequestHandlerMapper.POST.equals(method)) {
            BasicRoutedLogRecordProducer logRecordProducer;
            try {
                logRecordProducer = new BasicRoutedLogRecordProducer(
                    payload,
                    config.route(),
                    ContentType.APPLICATION_FORM_URLENCODED,
                    ContentType.parse(exchange.getRequest().getHeaders(HttpHeaders.CONTENT_TYPE)[0].getValue()),
                    decompressor);
            } catch (CompressorException e) {
                stat(YandexHttpStatus.SC_NOT_ACCEPTABLE, startTime);
                throw new ServerException(HttpStatus.SC_NOT_ACCEPTABLE, "Uncompressing of payload failed: " + e, e);
            }
            MultiFutureCallback<Void> multiCallback =
                new MultiFutureCallback<>(
                    new AbstractProxySessionCallback<>(session) {
                        @Override
                        public void completed(final List<Void> unused) {
                            stat(YandexHttpStatus.SC_OK, startTime);
                            session.logger().info("AbstractLogRecordsHandler.handle: finished");
                            session.response(HttpStatus.SC_OK);
                        }

                        @Override
                        public void cancelled() {
                            super.cancelled();
                            stat(YandexHttpStatus.SC_CLIENT_CLOSED_REQUEST, startTime);
                        }

                        @Override
                        public void failed(final Exception e) {
                            super.failed(e);
                            if (e instanceof BadResponseException) {
                                stat(((BadResponseException) e).statusCode(), startTime);
                            } else {
                                stat(YandexHttpStatus.SC_REMOTE_CLOSED_REQUEST, startTime);
                            }
                        }
            });
            LogRecordContext logRecordContext = (LogRecordContext)
                config.type().createLogRecordContext(
                    spLogger,
                    session,
                    logStorage.type(),
                    logRecordProducer.logRecord(),
                    config.route());
            logRecordContext.setTTL(config.storeTTL());
            saveLogRecord(logRecordContext, session, multiCallback.newCallback());
            saveToAuxiliaryStorages(logRecordProducer, session, multiCallback);
            saveRulesStat(logRecordProducer, session, multiCallback.newCallback());
            multiCallback.done();
        } else {
            stat(YandexHttpStatus.SC_METHOD_NOT_ALLOWED, startTime);
            throw new ServerException(HttpStatus.SC_METHOD_NOT_ALLOWED, "Method Not Allowed");
        }
    }

    public class GetIndexDataCallback implements FutureCallback<JsonObject>
    {
        private final LogRecordsContext<LogRecordContext> context;
        private final FutureCallback<List<LogRecordsBatch<LogRecordContext, LogStorageConfig>>> callback;

        public GetIndexDataCallback(
            final LogRecordsContext<LogRecordContext> context,
            final FutureCallback<List<LogRecordsBatch<LogRecordContext, LogStorageConfig>>> callback)
        {
            this.callback = callback;
            this.context = context;
        }

        @Override
        public void completed(final JsonObject result) {
            if (result == null || result.isEmpty()) {
                context.session().logger().info("GetIndexDataCallback: received null");
                callback.completed(null);
                return;
            }
            try {
                context.session().logger().info("GetIndexDataCallback: processing of result="
                    + JsonType.NORMAL.toString(result) + ", size=" + result.asList().size());
                context.init(result);
                LogRecordsContext<LogRecordContext> recordsStorageContext =
                    logStorage.type().createLogRecordsStorageContext(context, result);
                logStorage.get(recordsStorageContext, callback);
            } catch (JsonException e) {
                context.session().logger().log(Level.SEVERE, "GetIndexDataCallback failed to parse response", e);
                callback.completed(null);
            }
        }

        @Override
        public void cancelled() {
            context.session().logger().warning(
                "GetIndexDataCallback request cancelled: " + context.session().listener().details());
            callback.cancelled();
        }

        @Override
        public void failed(final Exception e) {
            context.session().logger().log(Level.WARNING, "GetIndexDataCallback failed: "
                + context.session().listener().details() + " because of exception: " + e.toString(), e);
            callback.failed(e);
        }
    }
}
