package ru.yandex.mail.so.logger;

import java.io.IOException;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.logging.Logger;

import org.apache.http.concurrent.FutureCallback;

import ru.yandex.concurrent.NamedThreadFactory;
import ru.yandex.concurrent.SingleNamedThreadFactory;
import ru.yandex.concurrent.TimeFrameQueue;
import ru.yandex.http.proxy.HttpProxy;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.RequestErrorType;
import ru.yandex.http.util.ServerException;
import ru.yandex.http.util.YandexHttpStatus;
import ru.yandex.mail.so.logger.config.BatchSaverConfig;
import ru.yandex.stater.CountAggregatorFactory;
import ru.yandex.stater.DuplexStaterFactory;
import ru.yandex.stater.IntegralSumAggregatorFactory;
import ru.yandex.stater.NamedStatsAggregatorFactory;
import ru.yandex.stater.PassiveStaterAdapter;

public interface BatchLogSaver<P extends RoutedLogRecordProducer, C extends BatchSaverConfig>
    extends BatchHandler<P, C>
{
    String NULL = "null";
    int DEFAULT_BATCH_SIZE = 100;

    /**
     * Get logger associated to this object
     *
     * @return logger associated to this object
     */
    Logger logger();

    /**
     * Async save log record into batch (or in some degenerate case - directly into some storage).
     *
     * @param logRecordProducer producer of log record that to be saved to DB
     * @param session session in which the recording will be performed
     * @param callback callback for calling at the end of processing
     */
    void save(final P logRecordProducer, final ProxySession session, final FutureCallback<Void> callback);

    /**
     * Get signal for batches sizes.
     *
     * @return signal for batch size
     */
    TimeFrameQueue<Long> batchSize();

    /**
     * Get signal for batches capacity.
     *
     * @return signal for batches capacity
     */
    TimeFrameQueue<Long> batchCapacity();

    /**
     * Accumulates batch size for according signal.
     *
     * @param size batch size
     */
    default void batchSize(final long size) {
        batchSize().accept(size);
    }

    /**
     * Accumulates batch capacity for according signal.
     *
     * @param size batch capacity size
     */
    default void batchCapacity(final long size) {
        batchCapacity().accept(size);
    }

    /**
     * Creates the new batch saver for given HttpProxy object
     *
     * @param httpProxy current HttpProxy object
     * @return the new batch saver for given HttpProxy object
     */
    default BatchSaver<P, C> createBatchSaver(final HttpProxy<?> httpProxy) {
        ThreadGroup threadGroup = new ThreadGroup(httpProxy.getThreadGroup(), name() + "-AsyncWorker");
        return createBatchSaver(threadGroup, name());
    }

    /**
     * Creates the new batch saver for given ThreadGroup object and name of the worker
     * @param threadGroup given ThreadGroup object
     * @param name name of the worker
     * @return created batch saver
     */
    default BatchSaver<P, C> createBatchSaver(final ThreadGroup threadGroup, final String name) {
        return new BatchSaver<>(
            this,
            logger(),
            name(),
            new SingleNamedThreadFactory(threadGroup, name + "-BatchSaver", true));
    }

    /**
     * Creates thread pool for saving batches.
     *
     * @param threadGroup threads pool config object
     * @param workers number of threads in the pool
     * @return created thread pool executor object
     */
    default ThreadPoolExecutor createThreadPoolExecutor(final ThreadGroup threadGroup, final int workers) {
        return new ThreadPoolExecutor(
            workers,
            workers,
            1,
            TimeUnit.MINUTES,
            new LinkedBlockingQueue<>(),
            new NamedThreadFactory(new ThreadGroup(threadGroup, name() + "-AsyncExecutor"), true),
            new ThreadPoolExecutor.DiscardPolicy());
    }

    /**
     * Starts storage engine
     *
     * @throws IOException exceptions, which may be thrown, converted to IOException
     */
    default void start() throws IOException {
        logger().info("Start " + name() + "-LogSaver");
        batchSaver().start();
    }

    /**
     * Default implementation of closing this log saver
     *
     * @throws IOException exception that may be thrown during closing process
     */
    @Override
    default void close() throws IOException {
        batchSaver().close();
        logger().info(name() + "Storage: start to close, batchSaver's size = " + batchSaver().batchesCount());
        synchronized (batchSaver()) {
            while (!batchSaver().closed()) {
                try {
                    logger().info(name() + "Storage: still not closed, batchSaver's size = "
                        + batchSaver().batchesCount());
                    batchSaver().wait(config().savingOperationTimeout());
                } catch (InterruptedException e) {
                    break;
                }
            }
            logger().info(name() + "Storage: closed!");
            batchSaver().notifyAll();
        }
    }

    /**
     * Gets type of the error - it's retriable or not.
     *
     * @param e input error (exception)
     * @return type of the error - it's retriable or not
     */
    static boolean retriableFailure(final Exception e) {
        RequestErrorType t = RequestErrorType.ERROR_CLASSIFIER.apply(e);
        if (t == RequestErrorType.IO || t == RequestErrorType.HTTP) {
            return true;
        }
        if (e instanceof ServerException) {
            int code = ((ServerException) e).statusCode();
            switch (code) {
                case YandexHttpStatus.SC_TOO_MANY_REQUESTS:
                case YandexHttpStatus.SC_BUSY:
                    return true;
                default:
                    break;
            }
        }
        return false;
    }

    default void registerBatchHandlerStaters(
        final HttpProxy<?> httpProxy,
        final String namePrefix)
    {
        httpProxy.registerStater(
            new PassiveStaterAdapter<>(
                batchSize(),
                new DuplexStaterFactory<>(
                    new NamedStatsAggregatorFactory<>(
                        namePrefix + "batch-size_ammm",
                        IntegralSumAggregatorFactory.INSTANCE),
                    new NamedStatsAggregatorFactory<>(
                        namePrefix + "batch-save-requests_ammm",
                        CountAggregatorFactory.INSTANCE))));
        httpProxy.registerStater(
            new PassiveStaterAdapter<>(
                batchCapacity(),
                new NamedStatsAggregatorFactory<>(
                    namePrefix + "batch-capacity_ammm",
                    IntegralSumAggregatorFactory.INSTANCE)));
    }
}
