package ru.yandex.mail.so.logger;

import java.util.concurrent.ThreadFactory;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.Consumer;
import java.util.logging.Level;
import java.util.logging.Logger;

import ru.yandex.function.GenericAutoCloseable;
import ru.yandex.mail.so.logger.config.BatchSaverConfig;

class BatchSaver<P extends RoutedLogRecordProducer, C extends BatchSaverConfig>
    implements GenericAutoCloseable<RuntimeException>, Runnable
{
    private final BatchHandler<P, C> batchHandler;
    private final Logger logger;
    private final String saverName;
    private final Thread thread;
    private final Consumer<? super Exception> errorHandler;
    private volatile Boolean actuallyClosed = null;
    private volatile boolean mustBeClosed = false;
    public final AtomicLong batchesCount;

    public BatchSaver(
        final BatchHandler<P, C> batchHandler,
        final Logger logger,
        final String saverName,
        final ThreadFactory threadFactory)
    {
        this.batchHandler = batchHandler;
        this.logger = logger;
        this.saverName = saverName;
        this.errorHandler = e -> this.logger.log(Level.WARNING, saverName + " batch saver failed: " + e, e);
        batchesCount = new AtomicLong(0L);
        thread = threadFactory.newThread(this);
    }

    public void start() {
        thread.start();
        synchronized (this) {
            actuallyClosed = false;
            logger.info(saverName + " BatchSaver: STARTED");
            notifyAll();
        }
    }

    @Override
    public void close() {
        mustBeClosed = true;
    }

    @Override
    public void run() {
        while (!mustBeClosed) {
            //logger.info(saverType + " BatchSaver: start the new batch cycle of processing");
            synchronized (batchHandler) {
                //logger.info(saverType + " BatchSaver: start to wait until batch is not empty");
                while (!mustBeClosed && (!batchHandler.currentBatch().isReady() && (batchHandler.batchIsEmpty()
                        || batchHandler.currentBatch().lifeTime() <= batchHandler.config().batchSavePeriod())
                        || batchesCount.get() >= batchHandler.config().workers()))
                {
                    try {
                        logger.info(saverName + " BatchSaver: batch is empty!");
                        batchHandler.wait(batchHandler.config().savingOperationTimeout());
                    } catch (InterruptedException e) {
                        break;
                    }
                }
                if (!mustBeClosed && ((batchHandler.currentBatch().lifeTime() > batchHandler.config().batchSavePeriod()
                        && batchHandler.currentBatch().count() > 0) || batchHandler.currentBatch().isReady())
                        && batchesCount.get() < batchHandler.config().workers())
                {
                    try {
                        logger.info(saverName + " BatchSaver: ready=" + batchHandler.currentBatch().isReady()
                            + ", lifeTime=" + batchHandler.currentBatch().lifeTime() + ", dataWaitTimeout="
                            + batchHandler.config().batchSavePeriod() + ", count = "
                            + batchHandler.currentBatch().count());
                        if (batchHandler.saveBatch(batchHandler.resetBatch(), logger)) {
                            long cnt = batchesCount.incrementAndGet();
                            logger.info(saverName + " BatchSaver: spawned new thread, total = " + cnt);
                            batchHandler.notifyAll();
                        }
                    } catch (Exception e) {
                        errorHandler.accept(e);
                    }
                /*} else {
                    if (mustBeClosed) {
                        logger.info(saverName + " BatchSaver starts to shut down");
                    } else if (batchesCount.get() >= batchHandler.config().workers()) {
                        logger.info(saverName + " BatchSaver: batches count = " + batchesCount.get()
                            + ", but pool size = " + batchHandler.config().workers());
                    } else if (!batchHandler.currentBatch().isReady()) {
                        logger.info(saverName + " BatchSaver: batch is not ready");
                    } else {
                        logger.info(saverName + " BatchSaver: saving batch not starts due to some other reason");
                    }*/
                }
                //batchHandler.notifyAll();
            }
        }
        logger.info(saverName + " BatchSaver: start to be closed, batches count in processing = " + batchesCount.get());
        synchronized (this) {
            while (batchesCount.get() > 0) {
                try {
                    logger.info(saverName + " BatchSaver: there are " + batchesCount.get() + " batches in processing");
                    wait(batchHandler.config().batchSavePeriod());
                } catch (InterruptedException e) {
                    break;
                }
            }
            actuallyClosed = true;
            logger.info(saverName + " BatchSaver: CLOSED");
            notifyAll();
        }
        thread.interrupt();
    }

    public synchronized boolean started() {
        return actuallyClosed != null;
    }

    public synchronized boolean closed() {
        return actuallyClosed;
    }

    public void decrementBatchesCount() {
        synchronized (batchHandler) {
            batchesCount.decrementAndGet();
            batchHandler.notifyAll();
        }
    }

    public long batchesCount() {
        return batchesCount.get();
    }
}
