package ru.yandex.mail.so.logger;

import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.logging.Logger;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.parser.JsonException;
import ru.yandex.parser.uri.CgiParams;
import ru.yandex.util.timesource.TimeSource;

public class DeliveryLogRecordContext extends AbstractDeliveryLogRecordContext {
    public static final Pattern RE_MSG_DATE =
        Pattern.compile("(?i)[a-z]+ \\d\\d \\d\\d:\\d\\d:\\d\\d(?:[.0-9]+)? - (\\d+):");
    public static final Pattern RE_SOURCE_IP = Pattern.compile("source ip = (\\S+)");

    private String id;
    private long uid;
    private Long messageDate;
    private String fromAddress;
    private List<Long> recipientUids;
    private String sourceIp;    // may be InetAddress, but ...
    private String queueId;
    private String messageId;
    private SoResolution soResolution;
    private String mailBackend;
    private String localHost;
    private String stid;
    private int offset;
    private int logSize;
    private int byteOffset;
    private int logBytesSize;
    private final Logger logger;

    public DeliveryLogRecordContext(
        final SpLogger spLogger,
        final ProxySession session,
        final LogStorageType storageType,
        final String logRecord,
        final Route route)
    {
        super(spLogger, session.params(), storageType, logRecord, route);
        logger = session.logger();
        offset = byteOffset = 0;
        logBytesSize = logRecord.getBytes(StandardCharsets.UTF_8).length;
        logSize = logRecord.length();
    }

    public DeliveryLogRecordContext(final LogContext logContext, final String logRecord) {
        super(logContext);
        this.logRecord = logRecord;
        offset = byteOffset = 0;
        logBytesSize = logRecord.getBytes(StandardCharsets.UTF_8).length;
        logSize = logRecord.length();
        recipientUids = new ArrayList<>();
        logger = logContext.spLogger().logger();
    }

    public DeliveryLogRecordContext(final LogContext logContext, final JsonMap record) throws JsonException {
        super(logContext);
        recipientUids = new ArrayList<>();
        logger = logContext.spLogger().logger();
        init(record);
    }

    @Override
    public LogRecordsHandlerType handlerType() {
        return LogRecordsHandlerType.SP_DAEMON;
    }

    @Override
    public Logger logger() {
        return logger;
    }

    @Override
    @SuppressWarnings("StringSplitter")
    public void init(final CgiParams params) throws BadRequestException {
        long ts = TimeSource.INSTANCE.currentTimeMillis() / MILLIS;
        recipientUids = new ArrayList<>();
        String uidStr = params.getOrNull(SearchParam.UID.paramName());
        if (uidStr == null || uidStr.isEmpty()) {
            uid = 0L;
        } else {
            uid = Long.parseLong(uidStr);
        }
        messageDate = params.getLong(SearchParam.TS.paramName(), ts);
        fromAddress = params.getString(SearchParam.FROMADDR.paramName(), "");
        for (String rcptUid : params.getString(SearchParam.RCPT_UIDS.paramName(), "").split(",")) {
            if (rcptUid != null && !rcptUid.isEmpty()) {
                recipientUids.add(Long.parseLong(rcptUid));
            }
        }
        sourceIp = params.getString(SearchParam.SOURCE_IP.paramName(), "");
        queueId = params.getString(SearchParam.QUEUEID.paramName(), "");
        messageId = params.getString(SearchParam.MSGID.paramName(), "");
        soResolution = SoResolution.fromCode(params.getInt(SearchParam.CODE.paramName(), 0));
        mailBackend = params.getString(SearchParam.MX.paramName(), "");
        localHost = params.getString(SearchParam.LOCL.paramName(), "");
        ttl = params.getLong(SearchParam.EXPIRE_TIMESTAMP.paramName(), params.getLong("ttl", DEFAULT_TTL));
        String routeStr = params.getOrNull(SearchParam.ROUTE.paramName());
        if (routeStr != null) {
            try {
                route = Route.valueOf(routeStr.toUpperCase(Locale.ROOT));
            } catch (IllegalArgumentException e) {
                logger.warning("DeliveryLogRecordContext.init: invalid route's value: '" + routeStr + "'");
            }
        }
        stid = params.getString(SearchParam.STID.paramName(), null);
        offset = params.getInt("offset", 0);
        logSize = params.getInt("size", 0);
        byteOffset = params.getInt("bytes_offset", 0);
        logBytesSize = params.getInt("bytes_size", 0);
    }

    @Override
    public Map<String, List<String>> init(final JsonMap record) throws JsonException {
        JsonMap json =
            record.containsKey(LogContext.HITS_ARRAY) ? record.getList(LogContext.HITS_ARRAY).get(0).asMap() : record;
        Map<String, List<String>> cgiParams = new HashMap<>();
        id = json.getOrNull(IndexField.ID.fieldName());
        queueId = json.getOrNull(IndexField.QUEUEID.fieldName());
        if (queueId != null) {
            cgiParams.computeIfAbsent(SearchParam.QUEUEID.paramName(), x -> new ArrayList<>()).add(queueId);
        }
        messageId = json.getOrNull(IndexField.MSGID.fieldName());
        if (messageId != null) {
            cgiParams.computeIfAbsent(SearchParam.MSGID.paramName(), x -> new ArrayList<>()).add(messageId);
        }
        uid = json.getLong(IndexField.UID.fieldName(), 0L);
        if (uid > 0) {
            cgiParams.computeIfAbsent(SearchParam.UID.paramName(), x -> new ArrayList<>()).add(Long.toString(uid));
        }
        Long rcptUid = json.getLong(IndexField.RCPT_UID.fieldName(), 0L);
        if (rcptUid > 0) {
            recipientUids.add(rcptUid);
            cgiParams.computeIfAbsent(SearchParam.RCPT_UID.paramName(), x -> new ArrayList<>())
                .add(Long.toString(rcptUid));
        }
        messageDate = json.getLong(IndexField.TS.fieldName(), null);
        if (messageDate != null) {
            cgiParams.computeIfAbsent(SearchParam.TS.paramName(), x -> new ArrayList<>())
                .add(Long.toString(messageDate));
        }
        fromAddress = json.getOrNull(IndexField.FROMADDR.fieldName());
        if (fromAddress != null) {
            cgiParams.computeIfAbsent(SearchParam.FROMADDR.paramName(), x -> new ArrayList<>()).add(fromAddress);
        }
        sourceIp = json.getOrNull(IndexField.SOURCE_IP.fieldName());
        if (sourceIp != null) {
            cgiParams.computeIfAbsent(SearchParam.SOURCE_IP.paramName(), x -> new ArrayList<>()).add(sourceIp);
        }
        Long resolution = json.getLong(IndexField.CODE.fieldName(), null);
        if (resolution != null) {
            if (resolution == 1) {
                soResolution = SoResolution.HAM;
            } else if (resolution == 2) {
                soResolution = SoResolution.SPAM;
            }
            cgiParams.computeIfAbsent(SearchParam.CODE.paramName(), x -> new ArrayList<>())
                .add(Long.toString(resolution));
        }
        localHost = json.getOrNull(IndexField.LOCL.fieldName());
        if (localHost != null) {
            cgiParams.computeIfAbsent(SearchParam.LOCL.paramName(), x -> new ArrayList<>()).add(localHost);
        }
        mailBackend = json.getOrNull(IndexField.MX.fieldName());
        if (mailBackend != null) {
            cgiParams.computeIfAbsent(SearchParam.MX.paramName(), x -> new ArrayList<>()).add(mailBackend);
        }
        String lRoute = json.getOrNull(IndexField.ROUTE.fieldName());
        if (lRoute != null) {
            try {
                route = Route.valueOf(lRoute.toUpperCase(Locale.ROOT));
                cgiParams.computeIfAbsent(SearchParam.ROUTE.paramName(), x -> new ArrayList<>()).add(lRoute);
            } catch (IllegalArgumentException e) {
                logger.warning("DeliveryLogRecordContext.init: invalid route's value: '" + lRoute + "'");
            }
        }
        ttl = json.getLong(IndexField.EXPIRE_TIMESTAMP.fieldName(), ttl);
        if (ttl > 0) {
            cgiParams.computeIfAbsent(SearchParam.EXPIRE_TIMESTAMP.paramName(), x -> new ArrayList<>())
                .add(Long.toString(ttl));
        }
        stid = json.getOrNull(IndexField.STID.fieldName());
        if (stid != null) {
            cgiParams.computeIfAbsent(SearchParam.STID.paramName(), x -> new ArrayList<>()).add(stid);
        }
        offset = json.getInt(IndexField.OFFSET.fieldName());
        logSize = json.getInt(IndexField.SIZE.fieldName());
        byteOffset = json.getInt(IndexField.BYTES_OFFSET.fieldName(), 0);
        logBytesSize = json.getInt(IndexField.BYTES_SIZE.fieldName(), 0);
        return cgiParams;
    }

    @Override
    public String storageKey() {
        return stid;
    }

    @Override
    public int offset() {
        return offset;
    }

    @Override
    public int logSize() {
        return logSize;
    }

    @Override
    public int byteOffset() {
        return byteOffset;
    }

    @Override
    public int logBytesSize() {
        return logBytesSize;
    }

    @Override
    public void setLogRecord(final String logRecord) {
        this.logRecord = logRecord;
        this.logBytesSize = logRecord.getBytes(StandardCharsets.UTF_8).length;
    }

    @Override
    public void setStorageKey(final String stid) {
        this.stid = stid;
    }

    @Override
    public void setOffset(final int offset) {
        this.offset = offset;
    }

    @Override
    public void setByteOffset(final int byteOffset) {
        this.byteOffset = byteOffset;
    }

    public void init(final JsonList jsonList) throws JsonException {
        for (final JsonObject item : jsonList) {
            if (item.type() == JsonObject.Type.LIST) {
                JsonList list = item.asList();
                String key = list.get(0).asString();
                String value = list.get(1).asString();
                if ("from:addr".equals(key)) {
                    fromAddress = value;
                } else if ("x-yandex-queueid".equals(key)) {
                    queueId = value;
                } else if ("x-yandex-so-front".equals(key)) {
                    mailBackend = value;
                } else if ("locl".equals(key)) {
                    localHost = value;
                } else if ("mess".equals(key)) {
                    Matcher m = RE_MSG_DATE.matcher(value);
                    if (m.find()) {
                        messageDate = Long.parseLong(m.group(1));
                    }
                } else if ("messageid".equals(key)) {
                    messageId = value;
                } else if ("rcvd".equals(key) && value != null && value.startsWith("source ip = ")) {
                    Matcher m = RE_SOURCE_IP.matcher(value);
                    if (m.find()) {
                        sourceIp = m.group(1);
                    }
                } else if ("spam".equals(key)) {
                    if ("no".equals(value)) {
                        soResolution = SoResolution.HAM;
                    } else if ("yes".equals(value)) {
                        soResolution = SoResolution.SPAM;
                    } else {
                        soResolution = SoResolution.SKIP;
                    }
                }
            }
        }
    }

    @Override
    public long prefix() {
        return prefix(queueId);
    }

    public String id() {
        return id;
    }

    public long uid() {
        return uid;
    }

    public long messageDate() {
        return messageDate;
    }

    public String fromAddress() {
        return fromAddress;
    }

    public List<Long> recipientUids() {
        return recipientUids;
    }

    public String sourceIp() {
        return sourceIp;
    }

    public String queueId() {
        return queueId;
    }

    public String messageId() {
        return messageId;
    }

    public SoResolution soResolution() {
        return soResolution;
    }

    public String mailBackend() {
        return mailBackend;
    }

    public String localHost() {
        return localHost;
    }
}
