package ru.yandex.mail.so.logger;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.logging.Level;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

import org.apache.http.concurrent.FutureCallback;

import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.ErrorSuppressingFutureCallback;
import ru.yandex.http.util.MultiFutureCallback;
import ru.yandex.http.util.RequestErrorType;
import ru.yandex.json.dom.JsonNull;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.mail.so.logger.config.ImmutableLogRecordsHandlerConfig;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.string.BooleanParser;
import ru.yandex.parser.string.EnumParser;
import ru.yandex.parser.uri.CgiParams;
import ru.yandex.parser.uri.QueryConstructor;

public class DeliveryLogRecordsHandler extends AbstractLogRecordsHandler {
    public static final String SEARCH_URI = SpLogger.SEARCH + "?";
    public static final String SEARCH_GETBYID_URI = SpLogger.SEARCH + "-getbyid?";

    private static final Pattern SEARCH_EXPR =
        Pattern.compile("\\b(log_(?:source_ip|stid):)([0-9a-zA-Z.:]+|\\([^()]+?\\))");
    private static final String PREFIX = "prefix";
    private static final String GET = "get";
    private static final String TEXT = "text";
    private static final String POSTFILTER = "postfilter";
    private static final String OFFSET = "offset";
    private static final String LENGTH = "length";
    private static final String SORT = "sort";

    private Route route;

    @SuppressWarnings("StringSplitter")
    public DeliveryLogRecordsHandler(
        final ImmutableLogRecordsHandlerConfig config,
        final SpLogger spLogger,
        final String path)
        throws ConfigException
    {
        super(config, spLogger, path);
        EnumParser<Route> parser = new EnumParser<>(Route.class);
        route = config.route();
        if (route == null) {
            List<String> parts = Arrays.asList(path.split("/"));
            route = parts.size() > 1 ? parser.apply(parts.get(2)) : Route.IN;
        }
    }

    @Override
    @SuppressWarnings("StringSplitter")
    public Map<String, Long> getSearchIndexRequest(final ProxySession session) throws BadRequestException {
        CgiParams params = session.params();
        Map<String, Long> queries = new HashMap<>();
        Map<SearchParam, List<Object>> fields = new HashMap<>();
        List<SearchParam> logParams = new ArrayList<>();
        String rawParam;
        if (route == null) {
            route = Route.valueOf(params.getOrNull(SearchParam.ROUTE.paramName()).toUpperCase(Locale.ROOT));
        }
        for (final SearchParam param : SearchParam.params()) {
            rawParam = params.getOrNull(param.paramName());
            session.logger().info("getSearchIndexRequest: param=" + param.paramName() + ", value=" + rawParam);
            if (rawParam == null || rawParam.isEmpty() || param == SearchParam.ROUTE) {
                continue;
            }
            fields.put(param, new ArrayList<>());
            if (param.indexField() == null) {
                if ("Long".equals(param.itemType())) {
                    fields.get(param).add(Long.parseLong(rawParam));
                } else if ("Integer".equals(param.itemType())) {
                    fields.get(param).add(Integer.parseInt(rawParam));
                } else if ("Boolean".equals(param.itemType())) {
                    fields.get(param).add(BooleanParser.INSTANCE.apply(rawParam));
                } else {
                    fields.get(param).add(rawParam);
                }
            } else {
                logParams.add(param);
                if ("Long".equals(param.itemType())) {
                    for (String paramItem : params.getAll(param.paramName())) {
                        for (String item : paramItem.split(",")) {
                            fields.get(param).add(Long.parseLong(item));
                        }
                    }
                } else if ("Integer".equals(param.itemType())) {
                    for (String paramItem : params.getAll(param.paramName())) {
                        for (String item : paramItem.split(",")) {
                            fields.get(param).add(Integer.parseInt(item));
                        }
                    }
                } else {
                    for (String paramItem : params.getAll(param.paramName())) {
                        fields.get(param).addAll(Arrays.asList(paramItem.split(",")));
                    }
                }
            }
        }
        String service = fields.containsKey(SearchParam.SERVICE)
            ? (String) fields.get(SearchParam.SERVICE).get(0) : spLogger.config().indexingQueueName();
        Integer skip = fields.containsKey(SearchParam.SKIP)
            ? (Integer) fields.get(SearchParam.SKIP).get(0) : 0;
        int limit = fields.containsKey(SearchParam.LIMIT)
            ? (Integer) fields.get(SearchParam.LIMIT).get(0) : LogRecordsContext.DEFAULT_SEARCH_DOCS_LIMIT;

        if (fields.containsKey(SearchParam.QUEUEID)) {
            long prefix;
            Map<Long, List<Object>> prefixQueueIds = new HashMap<>();
            for (Object queueid : fields.get(SearchParam.QUEUEID)) {
                prefix = LogRecordContext.prefix((String) queueid, spLogger);
                prefixQueueIds.computeIfAbsent(prefix, x -> new ArrayList<>()).add(queueid);
            }
            fields.remove(SearchParam.QUEUEID);
            Map<SearchParam, List<Object>> fields2;
            for (Map.Entry<Long, List<Object>> entry : prefixQueueIds.entrySet()) {
                fields2 = new HashMap<>(fields);
                fields2.put(SearchParam.QUEUEID, entry.getValue());
                if (limit < entry.getValue().size()) {
                    limit = entry.getValue().size();
                }
                queries.put(
                    createSearchQuery(service, entry.getKey(), logParams, fields2, skip, limit),
                    entry.getKey());
            }
        } else {
            queries.put(createSearchQuery(service, null, logParams, fields, null, limit), null);
        }
        if (logParams.size() < 1) {
            throw new BadRequestException(
                "specify at least one of: uid, rcpt_uid, queueid, msgid, from_addr, source_ip, code, mx, "
                + "ts, expire_timestamp");
        }
        return queries;
    }

    @Override
    public void searchData(
        final LogRecordsContext<?> context,
        final Map<String, Long> queries,    // Map: query -> prefix (may be null)
        final FutureCallback<JsonObject> callback)
    {
        DeliveryLogRecordsContext logRecordsContext = (DeliveryLogRecordsContext) context;
        CgiParams params = logRecordsContext.session().params();
        int length = LogRecordsContext.DEFAULT_SEARCH_DOCS_LIMIT;
        if (!params.containsKey(SearchParam.QUEUEID.paramName())) {
            try {
                length = params.getInt(SearchParam.LIMIT.paramName(), LogRecordsContext.DEFAULT_SEARCH_DOCS_LIMIT);
            } catch (BadRequestException e) {
                context.session().logger().log(Level.SEVERE, "searchData failed: " + e, e);
                callback.completed(null);
            }
        }
        try {
            MultiFutureCallback<JsonObject> multiCallback =
                new MultiFutureCallback<>(
                    new JsonListFilterFutureCallback(context.session().logger(), 0, length, callback));
            context.session().logger().info("searchData: search queries count=" + queries.size());
            for (final Map.Entry<String, Long> query : queries.entrySet()) {
                if (query.getValue() == null) {
                    context.session().logger().info("searchData: unprefixed search for query=" + query.getKey());
                    unprefixedSearchData(
                        logRecordsContext,
                        query.getKey(),
                        new ErrorSuppressingFutureCallback<>(
                            multiCallback.newCallback(),
                            x -> RequestErrorType.HTTP,
                            JsonNull.INSTANCE));
                } else {
                    context.session().logger().info("searchData: prefixed search query=" + query.getKey()
                        + ", prefix=" + query.getValue());
                    prefixedSearchData(
                        logRecordsContext,
                        query.getKey(),
                        query.getValue(),
                        new ErrorSuppressingFutureCallback<>(
                            multiCallback.newCallback(),
                            x -> RequestErrorType.HTTP,
                            JsonNull.INSTANCE));
                }
            }
            multiCallback.done();
        } catch (Exception e) {
            context.session().logger().log(Level.SEVERE, "searchData failed: " + e, e);
            callback.completed(null);
        }
    }

    private String createSearchQuery(
        final String service,
        final Long prefix,
        final List<SearchParam> logParams,
        final Map<SearchParam, List<Object>> fields,
        final Integer skip,
        final Integer limit)
        throws BadRequestException
    {
        Set<IndexField> indexFields = new HashSet<>(
            Set.of(
                IndexField.ID,
                IndexField.QUEUEID,
                IndexField.STID,
                IndexField.TS,
                IndexField.OFFSET,
                IndexField.SIZE,
                IndexField.BYTES_OFFSET,
                IndexField.BYTES_SIZE));
        if (!fields.containsKey(SearchParam.UNIQ_MSG) || !(Boolean) fields.get(SearchParam.UNIQ_MSG).get(0)) {
            indexFields.add(IndexField.RCPT_UID);
        }
        indexFields.addAll(
            logParams.stream().map(SearchParam::indexField).filter(Objects::nonNull).collect(Collectors.toList()));
        String searchFieldsList = indexFields.stream().sorted(Comparator.comparingInt(IndexField::order))
            .map(IndexField::fieldName).collect(Collectors.joining(","));
        boolean getById = fields.containsKey(SearchParam.GETBYID) && (Boolean) fields.get(SearchParam.GETBYID).get(0);
        QueryConstructor query = new QueryConstructor(getById ? SEARCH_GETBYID_URI : SEARCH_URI, false);
        query.append(SearchParam.SERVICE.paramName(), service);
        if (prefix != null) {
            query.append(PREFIX, prefix);
        }
        query.append(GET, searchFieldsList);
        StringBuilder filterExpr = new StringBuilder();
        for (SearchParam param : logParams) {
            filterExpr.append(filterExpr.length() < 1 ? "" : " AND ").append(param.indexField().fieldName())
                .append(':');
            if (fields.get(param).size() > 1) {
                filterExpr.append('(');
            }
            filterExpr.append(fields.get(param).get(0));
            for (int i = 1; i < fields.get(param).size(); i++) {
                filterExpr.append(" OR ").append(fields.get(param).get(i));
            }
            if (fields.get(param).size() > 1) {
                filterExpr.append(')');
            }
        }
        Long minTs = fields.containsKey(SearchParam.MINTIME) ? (Long) fields.get(SearchParam.MINTIME).get(0) : null;
        Long maxTs = fields.containsKey(SearchParam.MAXTIME) ? (Long) fields.get(SearchParam.MAXTIME).get(0) : null;
        query.append(TEXT, filterExpr.toString());
        query.append(POSTFILTER, IndexField.ROUTE.fieldName() + " == " + route.lowerName());
        if (minTs != null && minTs > 0L) {
            query.append(POSTFILTER, IndexField.TS.fieldName() + " >= " + minTs);
        }
        if (maxTs != null && maxTs > 0L) {
            query.append(POSTFILTER, IndexField.TS.fieldName() + " <= " + maxTs);
        }
        if (skip != null) {
            query.append(OFFSET, skip);
        }
        if (limit != null) {
            query.append(LENGTH, limit);
        }
        query.append(SORT, IndexField.TS.fieldName());
        return quoteSearchQuery(query.toString());
    }

    private static String quoteSearchQuery(final String query) {
        Matcher matcher = SEARCH_EXPR.matcher(query);
        StringBuilder resultQuery = new StringBuilder();
        int start = 0;
        while (matcher.find()) {
            resultQuery.append(query, start, matcher.start());
            resultQuery.append(matcher.group(1));
            resultQuery.append(matcher.group(2).replaceAll(":", "\\\\:"));
            start = matcher.end();
        }
        resultQuery.append(query.substring(start));
        return resultQuery.toString();
    }
}
