package ru.yandex.mail.so.logger;

import java.util.Comparator;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.SortedSet;
import java.util.TreeSet;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.stream.Collectors;

import org.apache.http.concurrent.FutureCallback;

import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.json.dom.IdentityContainerFactory;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonLong;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.JsonType;

public class JsonListFilterFutureCallback extends AbstractFilterFutureCallback<List<JsonObject>, JsonList> {
    private final Logger logger;
    private final int offset;
    private final int length;
    private int curLength;

    @SuppressWarnings("unused")
    public JsonListFilterFutureCallback(final Logger logger, final FutureCallback<JsonObject> callback) {
        this(logger, 0, LogRecordsContext.DEFAULT_SEARCH_DOCS_LIMIT, callback);
    }

    public JsonListFilterFutureCallback(
        final Logger logger,
        final int offset,
        final int length,
        final FutureCallback<JsonObject> callback)
    {
        super(callback);
        this.logger = logger;
        this.offset = offset;
        this.length = length;
        curLength = 0;
    }

    @Override
    public void completed(final List<JsonObject> jsonObjectsList) {
        if (jsonObjectsList == null) {
            logger.info("JsonListFilterFutureCallback: obtained result = null");
            callback.completed(null);
            return;
        }
        logger.info("JsonListFilterFutureCallback: obtained " + jsonObjectsList.size() + " docs = "
            + jsonObjectsList.stream().map(JsonType.NORMAL::toString).collect(Collectors.joining(",")));
        JsonList jsonList = new JsonList(IdentityContainerFactory.INSTANCE);
        Comparator<JsonMap> comparator = new JsonDocsComparator();
        SortedSet<JsonMap> sortedSet = new TreeSet<>(comparator);
        if (jsonObjectsList.size() > 0) {
            JsonObject.Type itemType = jsonObjectsList.get(0).type();   // we assume that all items have the same type
            try {
                if (itemType == JsonObject.Type.LIST) {
                    for (JsonObject jsonObject : jsonObjectsList) {
                        addMapsList(jsonObject.asList(), sortedSet);
                    }
                } else if (itemType == JsonObject.Type.MAP) {
                    addMapsList(jsonObjectsList, sortedSet);
                }
            } catch (JsonException e) {
                logger.log(Level.SEVERE, "JsonListFilterFutureCallback failed to parse result JSON", e);
            }
            JsonMap jsonMap = new JsonMap(IdentityContainerFactory.INSTANCE);
            jsonMap.put(LogContext.HITS_COUNT, new JsonLong(sortedSet.size()));
            JsonList jsonInnerList = new JsonList(IdentityContainerFactory.INSTANCE);
            Iterator<JsonMap> iterator = sortedSet.iterator();
            curLength = 0;
            while (iterator.hasNext()) {
                ++curLength;
                if (curLength < offset) {
                    continue;
                } if (curLength <= length) {
                    jsonInnerList.add(iterator.next());
                } else {
                    break;
                }
            }
            jsonMap.put(LogContext.HITS_ARRAY, jsonInnerList);
            jsonList.add(jsonMap);
        }
        callback.completed(jsonList);
    }

    @Override
    public void failed(final Exception e) {
        logger.log(Level.SEVERE, "JsonListFilterFutureCallback failed", e);
        callback.failed(e);
    }

    private void addMapsList(final List<JsonObject> jsonListIn, final SortedSet<JsonMap> sortedSet)
        throws JsonException
    {
        Map<String, JsonMap> uniqueHits = new HashMap<>();
        for (JsonObject jsonObject : jsonListIn) {
            if (jsonObject.type() == JsonObject.Type.MAP) {
                JsonMap jsonMap = jsonObject.asMap();
                if(jsonMap.containsKey(LogContext.HITS_COUNT) && jsonMap.getLong(LogContext.HITS_COUNT) > 0) {
                    for (JsonObject hit : jsonObject.asMap().get(LogContext.HITS_ARRAY).asList()) {
                        addMapsListHelper(hit, uniqueHits);
                    }
                }
            } else if (jsonObject.type() == JsonObject.Type.LIST) {
                for (JsonObject object : jsonObject.asList()) {
                    for (JsonObject hit : object.asMap().get(LogContext.HITS_ARRAY).asList()) {
                        addMapsListHelper(hit, uniqueHits);
                    }
                }
            }
        }
        sortedSet.addAll(uniqueHits.values());
    }

    private void addMapsListHelper(final JsonObject hit, final Map<String, JsonMap> uniqueHits) throws JsonException {
        StringBuilder sb = new StringBuilder();
        if (hit.asMap().containsKey("id")) {
            sb.append(hit.asMap().get("id"));
        } else {
            for (Map.Entry<String, JsonObject> entry : hit.asMap().entrySet()) {
                sb.append(entry.getValue().asStringOrNull());
            }
        }
        String value = sb.toString();
        if (!uniqueHits.containsKey(value)) {   // doc must be unique in one hitsArray and between hitArrays
            uniqueHits.put(value, hit.asMap());
        }
    }

    private static class JsonDocsComparator implements Comparator<JsonMap>
    {
        @Override
        public int compare(final JsonMap doc1, final JsonMap doc2)
            throws ClassCastException
        {
            try {
                // all elements are "unique", the order of elements with the same timestamp is ignored
                return Long.compare(
                    doc2.asMap().getLong(IndexField.TS.fieldName(), 0L) * LogRecordContext.MILLIS + 2L,
                    doc1.asMap().getLong(IndexField.TS.fieldName(), 0L) * LogRecordContext.MILLIS + 1L);
            } catch (JsonException e) {
                throw new ClassCastException(e.toString());
            }
        }

        @Override
        public boolean equals(Object obj) {
            return obj instanceof JsonMap;
        }

        @Override
        public int hashCode() {
            return super.hashCode();
        }
    }
}
