package ru.yandex.mail.so.logger;

import java.util.concurrent.atomic.AtomicLong;
import java.util.logging.Logger;

import ru.yandex.search.prefix.StringPrefix;
import ru.yandex.util.timesource.TimeSource;

public interface LogRecordContext extends LogContext, RoutedLogRecordProducer, LogRecordStorageInfo
{
    long MILLIS = 1000L;
    /**
     * Atomic long value that used in generating of log record's ID
     */
    AtomicLong recordId = new AtomicLong(TimeSource.INSTANCE.currentTimeMillis() / MILLIS);

    /**
     * Get logger for this context
     *
     * @return logger for this context
     */
    Logger logger();

    /**
     * Symbolic name of the log record's context
     *
     * @return name of the log record's context
     */
    String name();

    /**
     * Get prefix (shard number) of the log record within this context
     *
     * @return prefix (shard number) of the log record
     */
    long prefix();

    /**
     * Calc prefix (shard number) for given key string
     *
     * @param s key string
     * @return prefix (shard number) for given key string
     */
    default long prefix(final String s) {
        return Math.abs(new StringPrefix(s).hash() % spLogger().config().searchBackendShardsNumber());
    }

    /**
     * Static function that calculates prefix (shard number) for given key string and instance of SpLogger
     *
     * @param s key string
     * @param spLogger instance of SpLogger
     * @return prefix (shard number) for given key string and instance of SpLogger
     */
    static long prefix(final String s, final SpLogger spLogger) {
        return Math.abs(new StringPrefix(s).hash() % spLogger.config().searchBackendShardsNumber());
    }

    /**
     * Static function that calculates unique log record ID from given parameters.
     * Then this log record ID is used for indexing it in search backend.
     *
     * @param logType log record's type: for example, delivery log
     * @param route route associated to the log record
     * @param id some quasi-unique identifier associated to the log record
     * @param uid user ID associated to the log record
     * @return unique log record ID from given parameters
     */
    static String logId(final String logType, final String route, final String id, final long uid) {
        return "log_" + logType + '_' + route + '_' + id.replace('-', '_')
            + '_' + uid + '_' + Long.toString(recordId.getAndIncrement(), 16)
            + '_' + Long.toString((long) (Math.random() * MILLIS), 16);
    }

    /**
     * Sets the log record for this context
     *
     * @param logRecord log record for this context
     */
    void setLogRecord(final String logRecord);

    /**
     * Sets the storage key that log record receives after saving into storage
     *
     * @param storageKey storage key that log record receives after saving into storage
     */
    void setStorageKey(final String storageKey);

    /**
     * Sets the offset that log record receives after adding to the corresponding batch
     *
     * @param offset offset that log record receives after adding to the corresponding batch
     */
    void setOffset(final int offset);

    /**
     * Sets the byte offset that log record receives after adding to the corresponding batch
     *
     * @param byteOffset byte offset that log record receives after adding to the corresponding batch
     */
    void setByteOffset(final int byteOffset);

    /**
     * Get TTl of the log records that will be saving in the storage
     *
     * @return TTl of the log records that will be saving in the storage
     */
    long ttl();

    /**
     * Sets TTl of the log records that will be saving in the storage
     *
     * @param ttl TTl of the log records that will be saving in the storage
     */
    void setTTL(final long ttl);
}
