package ru.yandex.mail.so.logger;

import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.List;

import ru.yandex.mail.so.logger.config.BatchSaverConfig;
import ru.yandex.mail.so.logger.config.ConfigFactory;

public interface LogRecordsBatch<P extends RoutedLogRecordProducer, C extends BatchSaverConfig> extends Batch<P> {
    /**
     * Get logs basic storage's type: MDS, NULL, etc.
     *
     * @return logs basic storage's type
     */
    ConfigFactory<C> type();

    /**
     * Get log records with related info (i.e. its "contexts") for this batch
     *
     * @return log records with related info (i.e. its "contexts") for this batch
     */
    List<P> records();

    /**
     * Default implementation of function for receiving of all log records from this batch
     *
     * @return all log records within this batch
     */
    default List<String> logRecords() {
        List<String> logRecords = new ArrayList<>(records().size());
        for (P record : records()) {
            logRecords.add(record.logRecord());
        }
        return logRecords;
    }

    /**
     * Get size (in chars) of all log records from this batch
     *
     * @return size (in chars) of all log records from this batch
     */
    int contentSize();

    /**
     * Get size (in bytes) of all log records from this batch + separators between them
     *
     * @return size (in bytes) of all log records from this batch
     */
    int contentBytesSize();

    /**
     * Get string separator between log records in the batch during saving
     *
     * @return string separator between log records in the batch during saving
     */
    String separator();

    /**
     * Default implementation of function to receive content of all log records from this batch.
     * This content may be formed in any arbitrary way according to design of developer.
     *
     * @return content of all log records from this batch
     */
    default byte[] content() {
        int offset = 0;
        byte[] batchContent = new byte[contentBytesSize()];
        for (int i = 0; i < count(); i++) {
            byte[] record = (records().get(i).logRecord() + separator()).getBytes(StandardCharsets.UTF_8);
            System.arraycopy(
                record,
                0,
                batchContent,
                offset,
                record.length);
            offset += record.length;
        }
        return batchContent;
    }

    @Override
    default long count() {
        return records().size();
    }

    @Override
    default boolean isEmpty() {
        return count() < 1;
    }
}
