package ru.yandex.mail.so.logger;

import java.nio.charset.StandardCharsets;
import java.util.Arrays;
import java.util.Locale;
import java.util.TimerTask;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.stream.Collectors;

import org.apache.http.concurrent.FutureCallback;
import org.apache.http.entity.ContentType;
import org.apache.http.message.BasicHttpRequest;

import ru.yandex.base64.Base64;
import ru.yandex.concurrent.TimeFrameQueue;
import ru.yandex.data.compressor.CompressorException;
import ru.yandex.data.compressor.DataCompressor;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.EmptyAsyncConsumerFactory;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.http.util.request.RequestHandlerMapper;
import ru.yandex.http.util.request.RequestInfo;
import ru.yandex.json.parser.JsonException;
import ru.yandex.mail.so.logger.config.AuxiliaryStorageConfig;
import ru.yandex.mail.so.logger.config.ImmutableAuxiliaryStorageConfig;
import ru.yandex.stater.ImmutableStatersConfig;

public class LogsConsumerAuxiliaryStorage implements AuxiliaryStorage {
    final SpLogger spLogger;
    final ImmutableAuxiliaryStorageConfig config;
    private final String storageName;
    private final AsyncClient logsConsumerClient;
    protected final DataCompressor compressor;
    private volatile LogsConsumerRecordsBatch currentBatch;
    private final BatchSaver<BasicRoutedLogRecordProducer, AuxiliaryStorageConfig> batchSaver;
    private final TimeFrameQueue<Long> batchSize;
    private final TimeFrameQueue<Long> batchCapacity;

    public LogsConsumerAuxiliaryStorage(
        final SpLogger spLogger,
        final ImmutableAuxiliaryStorageConfig config,
        final String storageName)
        throws LogStorageException
    {
        this.spLogger = spLogger;
        this.config = config;
        this.storageName = storageName;
        currentBatch = new LogsConsumerRecordsBatch(DEFAULT_BATCH_SIZE);
        logsConsumerClient = spLogger.client(name() + "Client", config);
        batchSaver = createBatchSaver(spLogger);
        try {
            this.compressor = DataCompressor.valueOf(config.compression().toUpperCase(Locale.ROOT));
        } catch (IllegalArgumentException e) {
            throw new LogStorageException(
                "Compression method <" + config.compression() + "> for path <" + config.path() + "> is unknown. "
                    + "Available methods: "
                    + Arrays.stream(DataCompressor.values()).map(x -> x.name().toLowerCase(Locale.ROOT))
                        .collect(Collectors.joining(", ")) + '.',
                e);
        }
        String prefix = storageName + '-';
        batchSize = new TimeFrameQueue<>(spLogger.config().metricsTimeFrame());
        batchCapacity = new TimeFrameQueue<>(spLogger.config().metricsTimeFrame());
        registerBatchHandlerStaters(spLogger, prefix);
    }

    @Override
    public AuxiliaryStorageType type() {
        return AuxiliaryStorageType.LOGS_CONSUMER;
    }

    @Override
    public BatchSaver<BasicRoutedLogRecordProducer, AuxiliaryStorageConfig> batchSaver() {
        return batchSaver;
    }

    @Override
    public String name() {
        return "LogsConsumer";
    }

    public String storageName() {
        return storageName;
    }

    @Override
    public Logger logger() {
        return spLogger.logger();
    }

    @Override
    public AuxiliaryStorageConfig config() {
        return config;
    }

    public AsyncClient logsConsumerClient() {
        return logsConsumerClient;
    }

    @Override
    public TimeFrameQueue<Long> batchSize() {
        return batchSize;
    }

    @Override
    public TimeFrameQueue<Long> batchCapacity() {
        return batchCapacity;
    }

    @Override
    public void save(
        final BasicRoutedLogRecordProducer logRecordProducer,
        final ProxySession session,
        final FutureCallback<Void> callback)
    {
        synchronized (LogsConsumerAuxiliaryStorage.this) {
            long newContentLength = (long) currentBatch.contentBytesSize()
                + logRecordProducer.logRecord().getBytes(StandardCharsets.UTF_8).length;
            try {
                currentBatch.add(logRecordProducer);
                if (newContentLength > config.batchMinSize()) {
                    currentBatch.ready();
                    logger().info("LogsConsumerAuxiliaryStorage.save: batch content size = "
                        + currentBatch.contentSize() + ", bytesSize = " + currentBatch.contentBytesSize());
                }
            } catch (JsonException e) {
                logger().log(Level.SEVERE, "LogsConsumerAuxiliaryStorage.save failed to add log into batch", e);
            }
            notifyAll();
        }
        callback.completed(null);
    }

    @SuppressWarnings("FutureReturnValueIgnored")
    @Override
    public boolean saveBatch(Batch<BasicRoutedLogRecordProducer> batch, Logger logger) {
        if (!(batch instanceof LogsConsumerRecordsBatch)) {
            logger.info("LogsConsumerAuxiliaryStorage.saveBatch: batch is null or unknown type");
            return false;
        }
        if (batch.isEmpty()) {
            logger.info("LogsConsumerAuxiliaryStorage.saveBatch: batch is empty");
            batch.notReady();
            return false;
        }
        if (!batch.isReady() && batch.lifeTime() <= config.batchSavePeriod()) {
            logger.info("LogsConsumerAuxiliaryStorage.saveBatch: batch is not ready and batch lifetime less then batch"
                + " save timeout");
            return false;
        }
        LogsConsumerRecordsBatch logsConsumerBatch = (LogsConsumerRecordsBatch) batch;
        String logsContent;
        byte[] content = logsConsumerBatch.content();
        if (config.compression().equals("raw")) {
            logsContent = new String(content, StandardCharsets.UTF_8);
        } else {
            try {
                logsContent = compressor.compressAndBase64(content, Base64.URL);
            } catch (CompressorException e) {
                logger.log(Level.SEVERE, "LogsConsumerAuxiliaryStorage.saveBatch failed to compress data", e);
                return false;
            }
        }
        AsyncClient client = logsConsumerClient;
        ImmutableStatersConfig statersConfig = config.statersConfig();
        if (statersConfig != null) {
            client = logsConsumerClient.adjustStater(
                statersConfig,
                new RequestInfo(new BasicHttpRequest(RequestHandlerMapper.POST, config.host().toURI())));
        }
        String uri = config.path() + '/' + logsConsumerBatch.route().lowerName();
        BasicAsyncRequestProducerGenerator producerGenerator =
            new BasicAsyncRequestProducerGenerator(uri, logsContent, ContentType.APPLICATION_OCTET_STREAM);
        client.execute(
            config.host(),
            producerGenerator,
            EmptyAsyncConsumerFactory.ANY_GOOD,
            new AuxiliaryStorageSaveResultCallback(this, logsConsumerBatch));
        batchSize(logsConsumerBatch.count());
        batchCapacity(logsConsumerBatch.contentBytesSize());
        logger.info("LogsConsumerAuxiliaryStorage.saveBatch: URL=" + config.host() + uri + ", contentSize="
            + logsConsumerBatch.contentSize() + ", bytesSize = " + logsConsumerBatch.contentBytesSize()
            + ", recordsCnt=" + logsConsumerBatch.count() + ", separatorLength = "
            + logsConsumerBatch.separator().length() + ", separator = '" + logsConsumerBatch.separator() + "'");
        return true;
    }

    @Override
    public LogsConsumerRecordsBatch currentBatch() {
        return currentBatch;
    }

    @Override
    public boolean batchIsEmpty() {
        return currentBatch.isEmpty();
    }

    @Override
    public synchronized LogsConsumerRecordsBatch resetBatch() {
        LogsConsumerRecordsBatch oldBatch = currentBatch;
        currentBatch = new LogsConsumerRecordsBatch(DEFAULT_BATCH_SIZE);
        return oldBatch;
    }

    private static class AuxiliaryStorageSaveResultCallback implements FutureCallback<Void>
    {
        private final LogsConsumerAuxiliaryStorage storage;
        private final LogsConsumerRecordsBatch batch;

        public AuxiliaryStorageSaveResultCallback(
            final LogsConsumerAuxiliaryStorage storage,
            final LogsConsumerRecordsBatch batch)
        {
            this.storage = storage;
            this.batch = batch;
        }

        @Override
        public void completed(final Void ignored) {
            storage.logger().info("AuxiliaryStorageSaveResultCallback: batch saved successfully");
            storage.decrementBatchesCount();
        }

        @Override
        public void failed(final Exception e) {
            storage.logger().log(Level.SEVERE, "AuxiliaryStorageSaveResultCallback failed to save batch!", e);
            if (BatchLogSaver.retriableFailure(e)) {
                batch.retriesInc();
                if (batch.retriesCount() <= storage.config().batchSaveRetries()) {
                    storage.logsConsumerClient().scheduleRetry(new TimerTask() {
                        @Override
                        public void run() {
                            storage.saveBatch(batch, storage.logger());
                        }
                    }, storage.config().batchSavePeriod());
                }
            }
            storage.decrementBatchesCount();
        }

        @Override
        public void cancelled() {
            storage.logger().warning("AuxiliaryStorageSaveResultCallback: further batch's processing cancelled!");
            storage.decrementBatchesCount();
        }
    }
}
