package ru.yandex.mail.so.logger;

import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.List;

import ru.yandex.json.parser.JsonException;
import ru.yandex.mail.so.logger.config.AuxiliaryStorageConfig;
import ru.yandex.util.timesource.TimeSource;

public class LogsConsumerRecordsBatch implements LogRecordsBatch<BasicRoutedLogRecordProducer, AuxiliaryStorageConfig>
{
    private static final String SEP = "";
    private static final int SEP_LENGTH = SEP.getBytes(StandardCharsets.UTF_8).length;
    private volatile List<BasicRoutedLogRecordProducer> records;
    private volatile BatchState state = BatchState.INITED;
    private volatile int retriesCount;
    private final int defaultBatchSize;
    private int contentSize;
    private int contentBytesSize;
    private long startTime;

    public LogsConsumerRecordsBatch(final int defaultBatchSize) {
        this.defaultBatchSize = defaultBatchSize;
        init();
    }

    @Override
    public AuxiliaryStorageType type() {
        return AuxiliaryStorageType.LOGS_CONSUMER;
    }

    @Override
    public List<BasicRoutedLogRecordProducer> records() {
        return records;
    }

    @Override
    public String separator() {
        return SEP;
    }

    public synchronized void init() {
        state = BatchState.INITED;
        startTime = TimeSource.INSTANCE.currentTimeMillis();
        records = new ArrayList<>(defaultBatchSize);
        contentSize = 0;
        contentBytesSize = 0;
    }

    @Override
    public int contentSize() {
        return contentSize;
    }

    @Override
    public int contentBytesSize() {
        return contentBytesSize;
    }

    @Override
    public long lifeTime() {
        return TimeSource.INSTANCE.currentTimeMillis() - startTime;
    }

    @Override
    public synchronized void reset() {
        init();
        notifyAll();
    }

    public int retriesCount() {
        return retriesCount;
    }

    public synchronized void retriesInc() {
        retriesCount++;
    }

    @Override
    public synchronized void add(BasicRoutedLogRecordProducer logRecordProducer) throws JsonException {
        if (logRecordProducer == null || logRecordProducer.logRecord() == null) {
            return;
        }
        int logByteSize = logRecordProducer.logRecord().getBytes(StandardCharsets.UTF_8).length + SEP_LENGTH;
        int logSize = logRecordProducer.logRecord().length() + SEP.length();
        records.add(logRecordProducer);
        contentSize += logSize;
        contentBytesSize += logByteSize;
    }

    @Override
    public synchronized BatchState state() {
        return state;
    }

    @Override
    public synchronized void setState(BatchState state) {
        this.state = state;
    }

    public Route route() {
        if (records.size() > 0) {
            return records.get(0).route();
        } else {
            return null;
        }
    }
}
