package ru.yandex.mail.so.logger;

import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.Iterator;
import java.util.List;
import java.util.SortedMap;
import java.util.SortedSet;
import java.util.TreeMap;
import java.util.TreeSet;

import ru.yandex.mail.so.logger.config.LogStorageConfig;
import ru.yandex.util.timesource.TimeSource;

public class MdsLogRecordsBatch implements LogRecordsBatch<LogRecordContext, LogStorageConfig> {
    private String stid;
    private volatile List<LogRecordContext> records;
    private volatile SortedMap<Integer, List<LogRecordContext>> byteOffsets;
    private volatile LogRecordContext tailRecord;
    private volatile int currentOffset = 0;
    private volatile int currentByteOffset = 0;
    private volatile BatchState state = BatchState.INITED;
    private volatile int mdsRetriesCount;
    private volatile int luceneRetriesCount;
    private final int batchSize;
    private final String mdsNamespace;
    private long startTime;

    public MdsLogRecordsBatch(final String mdsNamespace) {
        this.batchSize = MdsLogStorage.DEFAULT_BATCH_SIZE;
        this.mdsNamespace = mdsNamespace;
        startTime = TimeSource.INSTANCE.currentTimeMillis();
        mdsRetriesCount = 0;
        luceneRetriesCount = 0;
        init();
    }

    public MdsLogRecordsBatch(final int batchSize, final String mdsNamespace) {
        this.batchSize = batchSize;
        this.mdsNamespace = mdsNamespace;
        startTime = TimeSource.INSTANCE.currentTimeMillis();
        mdsRetriesCount = 0;
        luceneRetriesCount = 0;
        init();
    }

    public MdsLogRecordsBatch(final LogRecordContext tailRecord, final String mdsNamespace) {
        this.batchSize = MdsLogStorage.DEFAULT_BATCH_SIZE;
        this.mdsNamespace = mdsNamespace;
        if (tailRecord == null) {
            this.tailRecord = null;
        } else {
            this.tailRecord = tailRecord;
            this.tailRecord.setOffset(0);
            this.tailRecord.setByteOffset(0);
        }
        startTime = TimeSource.INSTANCE.currentTimeMillis();
        mdsRetriesCount = 0;
        luceneRetriesCount = 0;
        init();
    }

    @SuppressWarnings("unused")
    public MdsLogRecordsBatch(final MdsLogRecordsBatch other, final String mdsNamespace) {
        this.mdsNamespace = mdsNamespace;
        startTime = TimeSource.INSTANCE.currentTimeMillis();
        mdsRetriesCount = 0;
        luceneRetriesCount = 0;
        if (other == null) {
            this.batchSize = MdsLogStorage.DEFAULT_BATCH_SIZE;
            init();
        } else {
            this.batchSize = other.batchSize;
            synchronized (this) {
                records = new ArrayList<>(other.records);
                byteOffsets = new TreeMap<>(other.byteOffsets);
            }
        }
    }

    @Override
    public LogStorageType type() {
        return LogStorageType.MDS;
    }

    @Override
    public List<LogRecordContext> records() {
        return records;
    }

    public SortedMap<Integer, List<LogRecordContext>> byteOffsets() {
        return byteOffsets;
    }

    @Override
    public String separator() {
        return "";
    }

    @Override
    public synchronized void reset() {
        state = BatchState.INITED;
        startTime = TimeSource.INSTANCE.currentTimeMillis();
        mdsRetriesCount = 0;
        luceneRetriesCount = 0;
        init();
        if (tailRecord != null) {
            this.tailRecord.setOffset(0);
            this.tailRecord.setByteOffset(0);
        }
        notifyAll();
    }

    public String namespace() {
        return mdsNamespace;
    }

    @SuppressWarnings("unused")
    public long startTime() {
        return startTime;
    }

    @Override
    public long lifeTime() {
        return TimeSource.INSTANCE.currentTimeMillis() - startTime;
    }

    public String stid() {
        return stid;
    }

    public void setStid(final String stid) {
        this.stid = stid;
        for (LogRecordContext record : records) {
            record.setStorageKey(stid);
        }
    }

    @Override
    public int contentSize() {
        return currentOffset;
    }

    @Override
    public int contentBytesSize() {
        return currentByteOffset;
    }

    @Override
    public synchronized BatchState state() {
        return state;
    }

    @Override
    public synchronized void setState(final BatchState state) {
        this.state = state;
    }

    public int retriesCount() {
        return mdsRetriesCount;
    }

    public synchronized void mdsRetriesInc() {
        mdsRetriesCount++;
    }

    public int luceneRetriesCount() {
        return luceneRetriesCount;
    }

    public synchronized void luceneRetriesInc() {
        luceneRetriesCount++;
    }

    @Override
    public synchronized void add(final LogRecordContext record) {
        if (record == null || record.logRecord() == null) {
            return;
        }
        int logByteSize = record.logRecord().getBytes(StandardCharsets.UTF_8).length;
        int logSize = record.logRecord().length();
        record.setOffset(currentOffset);
        record.setByteOffset(currentByteOffset);
        records.add(record);
        byteOffsets.computeIfAbsent(currentByteOffset, x -> new ArrayList<>()).add(record);
        currentOffset += logSize;
        currentByteOffset += logByteSize;
    }

    public void put(final LogRecordContext record) {
        if (record == null) {
            return;
        }
        records.add(record);
        byteOffsets.computeIfAbsent(record.byteOffset(), x -> new ArrayList<>()).add(record);
        if (currentOffset < record.offset() + record.logSize()) {
            currentOffset = record.offset() + record.logSize();
        }
        if (currentByteOffset < record.byteOffset() + record.logBytesSize()) {
            currentOffset = record.byteOffset() + record.logBytesSize();
        }
    }

    @SuppressWarnings("unused")
    public void setLogByIndex(final int i, final byte[] logRecord) {
        if (i < 0 || i >= records.size()) {
            return;
        }
        String sLogRecord = new String(logRecord, StandardCharsets.UTF_8).trim();
        if (!sLogRecord.isEmpty()) {
            for (LogRecordContext recordContext : byteOffsets.get(i)) {
                recordContext.setLogRecord(sLogRecord);
            }
        }
    }

    public void setLogByOffset(final int byteOffset, final byte[] logRecord) {
        if (!byteOffsets.containsKey(byteOffset)) {
            return;
        }
        String sLogRecord = new String(logRecord, StandardCharsets.UTF_8).trim();
        if (!sLogRecord.isEmpty()) {
            for (LogRecordContext recordContext : byteOffsets.get(byteOffset)) {
                recordContext.setLogRecord(sLogRecord);
            }
        }
    }

    @SuppressWarnings("unused")
    public LogRecordContext tailRecord() {
        return tailRecord;
    }

    public void addTail(final LogRecordContext logRecord) {
        if (logRecord == null) {
            return;
        }
        tailRecord = logRecord;
    }

    @Override
    public byte[] content() {
        byte[] batchContent = new byte[contentBytesSize()];
        for (LogRecordContext record : records) {
            System.arraycopy(
                record.logRecord().getBytes(StandardCharsets.UTF_8),
                0,
                batchContent,
                record.byteOffset(),
                record.logBytesSize());
        }
        return batchContent;
    }

    public String getRangeHeaderValue() {
        StringBuilder sb = new StringBuilder();
        Comparator<LogRecordContext> comparator = new LogRecordContextComparator();
        SortedSet<LogRecordContext> sorted = new TreeSet<>(comparator);
        sorted.addAll(records);
        Iterator<LogRecordContext> iterator = sorted.iterator();
        while (iterator.hasNext()) {
            LogRecordContext record = iterator.next();
            if (record.logBytesSize() > 0) {
                sb.append(record.byteOffset()).append('-').append(record.byteOffset() + record.logBytesSize() - 1);
                if (iterator.hasNext()) {
                    sb.append(',');
                }
            }
        }
        return sb.toString();
    }

    public synchronized void init() {
        records = new ArrayList<>(batchSize);
        byteOffsets = new TreeMap<>(new IntegerComparator());
        if (tailRecord != null && tailRecord.logRecord() != null) {
            records.add(tailRecord);
            currentOffset += tailRecord.logRecord().length();
            currentByteOffset += tailRecord.logRecord().getBytes(StandardCharsets.UTF_8).length;
            tailRecord = null;
        }
    }

    private static class LogRecordContextComparator implements Comparator<LogRecordContext>
    {
        @Override
        public int compare(LogRecordContext record1, LogRecordContext record2)
            throws NullPointerException, ClassCastException
        {
            return Integer.compare(record1.byteOffset(), record2.byteOffset());
        }

        @Override
        public boolean equals(Object obj) {
            return obj instanceof LogRecordContextComparator;
        }

        @Override
        public int hashCode() {
            return super.hashCode();
        }
    }

    private static class IntegerComparator implements Comparator<Integer>
    {
        @Override
        public int compare(Integer o1, Integer o2) throws NullPointerException, ClassCastException
        {
            return o1.compareTo(o2);
        }

        @Override
        public boolean equals(Object obj) {
            return obj instanceof IntegerComparator;
        }

        @Override
        public int hashCode() {
            return super.hashCode();
        }
    }
}
