package ru.yandex.mail.so.logger;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.net.URISyntaxException;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.TimerTask;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.Consumer;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;

import org.apache.http.Header;
import org.apache.http.HttpEntity;
import org.apache.http.HttpException;
import org.apache.http.HttpHost;
import org.apache.http.HttpResponse;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.mime.FormBodyPartBuilder;
import org.apache.http.entity.mime.MultipartEntityBuilder;
import org.apache.http.entity.mime.content.StringBody;
import org.apache.http.message.BasicHttpRequest;
import org.apache.james.mime4j.MimeException;
import org.apache.james.mime4j.stream.EntityState;
import org.apache.james.mime4j.stream.Field;
import org.apache.james.mime4j.stream.RecursionMode;
import org.w3c.dom.Document;
import org.w3c.dom.NodeList;
import org.xml.sax.SAXException;

import ru.yandex.client.tvm2.Tvm2ServiceContextRenewalTask;
import ru.yandex.client.tvm2.Tvm2TicketRenewalTask;
import ru.yandex.concurrent.SingleNamedThreadFactory;
import ru.yandex.concurrent.TimeFrameQueue;
import ru.yandex.function.GenericAutoCloseable;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.HeaderUtils;
import ru.yandex.http.util.MultiFutureCallback;
import ru.yandex.http.util.YandexHeaders;
import ru.yandex.http.util.nio.AsyncStringConsumerFactory;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.BasicAsyncResponseConsumerFactory;
import ru.yandex.http.util.nio.EmptyAsyncConsumerFactory;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.http.util.request.RequestHandlerMapper;
import ru.yandex.http.util.request.RequestInfo;
import ru.yandex.io.IOStreamUtils;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.Utf8JsonWriter;
import ru.yandex.mail.so.logger.config.LogStorageConfig;
import ru.yandex.mail.so.logger.config.MdsLogStorageConfig;
import ru.yandex.parser.searchmap.SearchMap;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.search.document.mail.SafeMimeTokenStream;
import ru.yandex.stater.CountAggregatorFactory;
import ru.yandex.stater.DuplexStaterFactory;
import ru.yandex.stater.ImmutableStatersConfig;
import ru.yandex.stater.IntegralSumAggregatorFactory;
import ru.yandex.stater.NamedStatsAggregatorFactory;
import ru.yandex.stater.PassiveStaterAdapter;
import ru.yandex.util.timesource.TimeSource;

public class MdsLogStorage extends AbstractLogStorage<LogRecordContext> {
    public static final String RANGE = "Range";
    public static final String CONTENT_RANGE = "Content-Range";
    public static final Pattern CONTENT_RANGE_RE = Pattern.compile("(\\d+)-(\\d+)");
    public static final AtomicLong batchesSize = new AtomicLong(0L);

    private static final String UPDATE_URI = "/add?";

    private final SpLogger spLogger;
    private final ThreadLocal<ByteArrayOutputStream> baosTls = ThreadLocal.withInitial(ByteArrayOutputStream::new);
    private final AsyncClient mdsWriterClient;
    private final AsyncClient mdsReaderClient;
    private final Tvm2ServiceContextRenewalTask serviceContextRenewalTask;
    private final Tvm2TicketRenewalTask tvm2RenewalTask;
    private long mdsTTL;
    private volatile MdsLogRecordsBatch currentBatch;
    private volatile String currentMdsWriteHost;
    private final Logger deliveryLogger;
    private final BatchSaver<LogRecordContext, LogStorageConfig> batchSaver;
    private final MdsWriteHostRenewer mdsWriteHostRenewer;
    private final TimeFrameQueue<Long> batchSize;
    private final TimeFrameQueue<Long> batchCapacity;
    private final TimeFrameQueue<Long> logsSavedToDisk;
    private final TimeFrameQueue<Long> mdsRetriableFailures;
    private final TimeFrameQueue<Long> mdsNonRetriableFailures;
    private final TimeFrameQueue<Long> mdsRetriesExceededFailures;
    private final TimeFrameQueue<Long> luceneRetriableFailures;
    private final TimeFrameQueue<Long> luceneNonRetriableFailures;
    private final TimeFrameQueue<Long> luceneRetriesExceededFailures;

    public MdsLogStorage(final String storageName, final SpLogger spLogger, final MdsLogStorageConfig config)
        throws LogStorageException
    {
        super(storageName, config);
        this.spLogger = spLogger;
        deliveryLogger = spLogger.spareDeliveryLogger();
        currentBatch = new MdsLogRecordsBatch(DEFAULT_BATCH_SIZE, storageName);
        mdsWriterClient = spLogger.client("MDS-to-Write-" + storageName, config.backendWriterConfig());
        mdsReaderClient = spLogger.client("MDS-to-Read-" + storageName, config.backendReaderConfig());
        ThreadGroup threadGroup = new ThreadGroup(spLogger.getThreadGroup(), storageName + "-AsyncWorker");
        batchSaver = createBatchSaver(threadGroup, storageName);
        if (batchSaver == null) {
            spLogger.logger().log(Level.SEVERE, storageName + "-BatchSaver failed to be initialized!");
        }
        mdsWriteHostRenewer =
            new MdsWriteHostRenewer(
                this,
                spLogger.logger(),
                config.mdsHostnameRequestPeriod(),
                new SingleNamedThreadFactory(threadGroup, storageName + "-MdsWriteHostRenewer", true));
        mdsTTL = config.storeTTL();
        synchronized (mdsWriterClient) {
            currentMdsWriteHost = config.backendWriterConfig().host().getHostName();
        }
        try {
            serviceContextRenewalTask = new Tvm2ServiceContextRenewalTask(
                spLogger.logger().addPrefix(MdsLogStorageConfig.TVM2),
                config.tvm2ServiceConfig(),
                config.dnsConfig());
            tvm2RenewalTask = new Tvm2TicketRenewalTask(
                spLogger.logger().addPrefix(MdsLogStorageConfig.TVM2),
                serviceContextRenewalTask,
                config.tvm2ClientConfig());
            spLogger.logger().log(Level.SEVERE, "TVM2 ticket: " + mdsTvm2Ticket());
        } catch (HttpException | IOException | JsonException | URISyntaxException e) {
            spLogger.logger().log(
                Level.SEVERE,
                "Error occurred while creating TVM2 renewal task for MDS log storage plugin",
                e);
            throw new LogStorageException(
                config.type(),
                storageName,
                "Error occurred while creating of MDS log storage plugin: " + e,
                e);
        }
        String prefix = storageName + '-';
        batchSize = new TimeFrameQueue<>(spLogger.config().metricsTimeFrame());
        batchCapacity = new TimeFrameQueue<>(spLogger.config().metricsTimeFrame());
        registerBatchHandlerStaters(spLogger, prefix);
        logsSavedToDisk =  new TimeFrameQueue<>(spLogger.config().metricsTimeFrame());
        spLogger.registerStater(
            new PassiveStaterAdapter<>(
                logsSavedToDisk,
                new DuplexStaterFactory<>(
                    new NamedStatsAggregatorFactory<>(
                        prefix + "logs-saved-to-disk_ammm",
                        IntegralSumAggregatorFactory.INSTANCE),
                    new NamedStatsAggregatorFactory<>(
                        prefix + "batches-saved-to-disk_ammm",
                        CountAggregatorFactory.INSTANCE))));
        mdsRetriableFailures = new TimeFrameQueue<>(spLogger.config().metricsTimeFrame());
        spLogger.registerStater(
            new PassiveStaterAdapter<>(
                mdsRetriableFailures,
                new NamedStatsAggregatorFactory<>(
                    prefix + "storage-retriable-errors_ammm",
                    CountAggregatorFactory.INSTANCE)));
        mdsNonRetriableFailures = new TimeFrameQueue<>(spLogger.config().metricsTimeFrame());
        spLogger.registerStater(
            new PassiveStaterAdapter<>(
                mdsNonRetriableFailures,
                new NamedStatsAggregatorFactory<>(
                    prefix + "storage-non-retriable-errors_ammm",
                    CountAggregatorFactory.INSTANCE)));
        mdsRetriesExceededFailures = new TimeFrameQueue<>(spLogger.config().metricsTimeFrame());
        spLogger.registerStater(
            new PassiveStaterAdapter<>(
                mdsRetriesExceededFailures,
                new NamedStatsAggregatorFactory<>(
                    prefix + "storage-retries-exceeded-errors_ammm",
                    CountAggregatorFactory.INSTANCE)));
        luceneRetriableFailures = new TimeFrameQueue<>(spLogger.config().metricsTimeFrame());
        spLogger.registerStater(
            new PassiveStaterAdapter<>(
                luceneRetriableFailures,
                new NamedStatsAggregatorFactory<>(
                    prefix + "save-index-retriable-errors_ammm",
                    CountAggregatorFactory.INSTANCE)));
        luceneNonRetriableFailures = new TimeFrameQueue<>(spLogger.config().metricsTimeFrame());
        spLogger.registerStater(
            new PassiveStaterAdapter<>(
                luceneNonRetriableFailures,
                new NamedStatsAggregatorFactory<>(
                    prefix + "save-index-non-retriable-errors_ammm",
                    CountAggregatorFactory.INSTANCE)));
        luceneRetriesExceededFailures = new TimeFrameQueue<>(spLogger.config().metricsTimeFrame());
        spLogger.registerStater(
            new PassiveStaterAdapter<>(
                luceneRetriesExceededFailures,
                new NamedStatsAggregatorFactory<>(
                    prefix + "save-index-retries-exceeded-errors_ammm",
                    CountAggregatorFactory.INSTANCE)));
    }

    @Override
    public LogStorageType type() {
        return LogStorageType.MDS;
    }

    @Override
    public String name() {
        return "MDS";
    }

    @Override
    public Logger logger() {
        return spLogger.logger();
    }

    @Override
    public MdsLogStorageConfig config() {
        return (MdsLogStorageConfig) config;
    }

    @Override
    public TimeFrameQueue<Long> batchSize() {
        return batchSize;
    }

    @Override
    public TimeFrameQueue<Long> batchCapacity() {
        return batchCapacity;
    }

    @Override
    public BatchSaver<LogRecordContext, LogStorageConfig> batchSaver() {
        return batchSaver;
    }

    @Override
    public MdsLogRecordsBatch currentBatch() {
        return currentBatch;
    }

    @Override
    public synchronized MdsLogRecordsBatch resetBatch() {
        MdsLogRecordsBatch oldBatch = currentBatch;
        currentBatch = new MdsLogRecordsBatch(currentBatch.tailRecord(), storageName);
        return oldBatch;
    }

    @Override
    public boolean batchIsEmpty() {
        return currentBatch.isEmpty();
    }

    @Override
    public long ttl() {
        return mdsTTL;
    }

    @Override
    public void setTTL(final long ttl) {
        mdsTTL = ttl;
    }

    public void logsSavedToDisk(final long size) {
        logsSavedToDisk.accept(size);
    }

    public void mdsRetriableFailures() {
        mdsRetriableFailures.accept(1L);
    }

    public void mdsNonRetriableFailures() {
        mdsNonRetriableFailures.accept(1L);
    }

    public void mdsRetriesExceededFailures() {
        mdsRetriesExceededFailures.accept(1L);
    }

    public void luceneRetriableFailures() {
        luceneRetriableFailures.accept(1L);
    }

    public void luceneNonRetriableFailures() {
        luceneNonRetriableFailures.accept(1L);
    }

    public void luceneRetriesExceededFailures() {
        luceneRetriesExceededFailures.accept(1L);
    }

    @Override
    public void get(
        final LogRecordsContext<LogRecordContext> context,
        final FutureCallback<List<LogRecordsBatch<LogRecordContext, LogStorageConfig>>> callback)
    {
        Map<String, MdsLogRecordsBatch> logRecordsBatches = new ConcurrentHashMap<>();
        int limit = context.limit();
        for (LogRecordContext record : context.records()) {
            if (limit-- < 1) {
                break;
            }
            context.session().logger().info("MdsLogStorage.get: stid=" + record.storageKey() + ", offset="
                + record.offset() + ", logSize=" + record.logSize() + ", byteOffset=" + record.byteOffset()
                + ", bytesSize=" + record.logBytesSize());
            MdsLogRecordsBatch batch;
            if (logRecordsBatches.containsKey(record.storageKey())) {
                batch = logRecordsBatches.get(record.storageKey());
            } else {
                batch = new MdsLogRecordsBatch(config().mdsNamespace());
            }
            batch.put(record);
            batch.setStid(record.storageKey());
            logRecordsBatches.put(record.storageKey(), batch);
        }
        MultiFutureCallback<LogRecordsBatch<LogRecordContext, LogStorageConfig>> multiCallback =
            new MultiFutureCallback<>(new AbstractFilterFutureCallback<>(callback) {
                @Override
                public void completed(final List<LogRecordsBatch<LogRecordContext, LogStorageConfig>> batches) {
                    context.session().logger().info("MdsLogStorage.get: result batches count=" + batches.size());
                    callback.completed(batches);
                }

                @Override
                public void cancelled() {
                    context.session().logger().warning(
                        "MdsLogStorage GET request cancelled: " + context.session().listener().details());
                    callback.cancelled();
                }

                @Override
                public void failed(final Exception e) {
                    context.session().logger().log(Level.WARNING, "MdsLogStorage GET request failed: "
                        + context.session().listener().details() + " because of exception: " + e.toString(), e);
                    //callback.failed(e);
                    callback.completed(null);
                }
            });
        for (Map.Entry<String, MdsLogRecordsBatch> entry : logRecordsBatches.entrySet()) {
            get(context, entry.getValue(), multiCallback.newCallback());
        }
        multiCallback.done();
    }

    @SuppressWarnings("FutureReturnValueIgnored")
    public void get(
        final LogRecordsContext<LogRecordContext> context,
        final MdsLogRecordsBatch recordBatch,
        final FutureCallback<LogRecordsBatch<LogRecordContext, LogStorageConfig>> callback)
    {
        final StringBuilder request = new StringBuilder();
        request.append("/get-");
        request.append(config().mdsNamespace());
        request.append("/");
        request.append(recordBatch.stid());

        final AsyncClient client = mdsReaderClient.adjust(context.session().context());
        BasicAsyncRequestProducerGenerator producerGenerator =
            new BasicAsyncRequestProducerGenerator(new String(request));
        producerGenerator.addHeader(HeaderUtils.createHeader(YandexHeaders.X_YA_SERVICE_TICKET, mdsTvm2Ticket()));
        String bytesRanges = recordBatch.getRangeHeaderValue();
        if (!bytesRanges.isEmpty()) {
            producerGenerator.addHeader(HeaderUtils.createHeader(RANGE, "bytes=" + bytesRanges));
        }
        context.session().logger().info("MDS request: " + request + ", Range: " + bytesRanges);
        client.execute(
            config().backendReaderConfig().host(),
            producerGenerator,
            BasicAsyncResponseConsumerFactory.ANY_GOOD,
            context.session().listener().createContextGeneratorFor(client),
            new MdsGetResultCallback(context, recordBatch, callback));
    }

    @Override
    public void save(
        final LogRecordContext context,
        final ProxySession session,
        final FutureCallback<Void> callback)
    {
        long newContentLength;
        synchronized (MdsLogStorage.this) {
            newContentLength = (long) currentBatch.contentBytesSize() + context.logBytesSize();
            if (newContentLength > config.batchMinSize()
                    && ((config().batchSaveMaxRps() > 0d
                        && 1000d / Math.max(currentBatch.lifeTime(), 1) <= config().batchSaveMaxRps())
                        || config().batchSaveMaxRps() <= 0d))
            {
                if (newContentLength > config.batchMaxSize()) {
                    context.logger().info("MdsLogStorage.put: adding of payload exceeded of max batch size - "
                        + "added to tail");
                    currentBatch.addTail(context);
                } else {
                    currentBatch.add(context);
                }
                currentBatch.ready();
                context.logger().info("MdsLogStorage.put: batch content size = " + currentBatch.contentSize()
                    + ", bytesSize = " + currentBatch.contentBytesSize());
            } else {
                currentBatch.add(context);
            }
            notifyAll();
        }
        callback.completed(null);
    }

    @Override
    @SuppressWarnings("FutureReturnValueIgnored")
    public boolean saveBatch(final Batch<LogRecordContext> batch, final Logger logger) {
        if (!(batch instanceof MdsLogRecordsBatch)) {
            logger.info("MdsLogStorage.saveBatch: batch is null or unknown type");
            return false;
        }
        if (batch.isEmpty()) {
            logger.info("MdsLogStorage.saveBatch: batch is empty");
            batch.notReady();
            return false;
        }
        if (!batch.isReady() && batch.lifeTime() <= config.batchSavePeriod()) {
            logger.info("MdsLogStorage.saveBatch: batch is not ready and batch lifetime less then batch save "
                + "timeout");
            return false;
        }
        MdsLogRecordsBatch mdsBatch = (MdsLogRecordsBatch) batch;
        if (batchesSize.get() + mdsBatch.contentBytesSize() > config.batchesMemoryLimit()) {
            writeBatchOnDisk(mdsBatch);
            //batchesSize.updateAndGet(x -> x - mdsBatch.contentBytesSize());
            return false;
        }
        batchesSize.addAndGet(mdsBatch.contentBytesSize());
        LogRecordContext recordContext = mdsBatch.records().get(0);
        String uri = "/upload-" + config().mdsNamespace();
        if (recordContext.ttl() > LogRecordContext.MILLIS) {
            uri += "/?expire=" + (recordContext.ttl() / LogRecordContext.MILLIS) + "s";
        }
        AsyncClient client = mdsWriterClient;
        ImmutableStatersConfig statersConfig = config().backendWriterConfig().statersConfig();
        if (statersConfig != null) {
            client = mdsWriterClient.adjustStater(
                statersConfig,
                new RequestInfo(
                    new BasicHttpRequest(RequestHandlerMapper.POST, config().backendWriterConfig().host().toURI())));
        }
        byte[] content = mdsBatch.content();
        BasicAsyncRequestProducerGenerator producerGenerator =
            new BasicAsyncRequestProducerGenerator(uri, content, ContentType.APPLICATION_OCTET_STREAM);
        producerGenerator.addHeader(HeaderUtils.createHeader(YandexHeaders.X_YA_SERVICE_TICKET, mdsTvm2Ticket()));
        HttpHost host;
        synchronized (mdsWriterClient) {
            host = new HttpHost(currentMdsWriteHost, config().backendWriterConfig().host().getPort());
        }
        client.execute(
            host,
            producerGenerator,
            BasicAsyncResponseConsumerFactory.ANY_GOOD,
            new MdsPutResultCallback(this, logger, mdsBatch));
        batchSize(mdsBatch.count());
        batchCapacity(mdsBatch.contentBytesSize());
        logger.info("MdsLogStorage.saveBatch: URL=" + host + uri + ", contentSize=" + mdsBatch.contentSize()
            + ", bytesSize = " + mdsBatch.contentBytesSize() + ", recordsCnt=" + mdsBatch.count());
        return true;
    }

    @Override
    public void remove(final ProxySession session, final FutureCallback<Void> callback) {
    }

    @Override
    public void start() throws IOException {
        mdsWriteHostRenewer.start();
        tvm2RenewalTask.start();
        super.start();
    }

    @Override
    public void close() throws IOException {
        mdsWriteHostRenewer.close();
        tvm2RenewalTask.cancel();
        super.close();
    }

    public String mdsTvm2Ticket() {
        return tvm2RenewalTask.ticket(config().mdsTvmClientId());
    }

    @SuppressWarnings("unused")
    public final Tvm2ServiceContextRenewalTask serviceContextRenewalTask() {
        return serviceContextRenewalTask;
    }

    public void writeBatchOnDisk(final MdsLogRecordsBatch batch) {
        logsSavedToDisk(batch.count());
        for (final String logRecord : batch.logRecords()) {
            deliveryLogger.fine(logRecord);
        }
    }

    @SuppressWarnings("FutureReturnValueIgnored")
    public void saveIndexData(
        final Logger logger,
        final MdsLogRecordsBatch batch,
        final FutureCallback<Void> callback)
        throws BadRequestException
    {
        logger.info("MdsLogStorage: saving index data");
        if (batch == null || batch.records().size() < 1) {
            logger.info("MdsLogStorage.saveIndexData: empty batch will not be indexed!");
            callback.completed(null);
            return;
        }
        long prefix;
        final ByteArrayOutputStream baos = baosTls.get();
        final ContentType producerContentType =
            ContentType.APPLICATION_JSON.withCharset(spLogger.producerAsyncClient().requestCharset());
        LogRecordContext recordContext = batch.records().get(0);
        BasicAsyncRequestProducerGenerator producerGenerator;
        long curTime = TimeSource.INSTANCE.currentTimeMillis();
        long ttl = (curTime + recordContext.ttl()) / LogRecordContext.MILLIS;
        Map<SearchParam, Object> params = new HashMap<>();
        params.put(SearchParam.SERVICE, spLogger.config().indexingQueueName());
        params.put(SearchParam.ROUTE, recordContext.route().lowerName());
        //logger.info("MdsLogStorage.saveIndexData: TTL=" + ttl + ", curTime=" + curTime + ", durationTTL="
        //    + recordContext.ttl());
        if (batch.count() == 1 && (recordContext.handlerType() != LogRecordsHandlerType.SP_DAEMON
                || (recordContext.handlerType() == LogRecordsHandlerType.SP_DAEMON
                && ((DeliveryLogRecordContext) recordContext).recipientUids().size() == 1)))
        {
            prefix = recordContext.prefix();
            if (recordContext.handlerType() == LogRecordsHandlerType.SP_DAEMON) {
                params.put(SearchParam.QUEUEID, ((DeliveryLogRecordContext) recordContext).queueId());
            }
            String uri = prepareUri(prefix, params);
            byte[] body = prepareRequestBody(logger, baos, List.of(recordContext), prefix, ttl);
            producerGenerator = new BasicAsyncRequestProducerGenerator(uri, body, producerContentType);
            producerGenerator.addHeader(YandexHeaders.ZOO_SHARD_ID, Long.toString(prefix % SearchMap.SHARDS_COUNT));
        } else {
            String uri = prepareUri(null, Map.of(
                SearchParam.SERVICE, spLogger.config().indexingQueueName(),
                SearchParam.STID, batch.stid(),
                SearchParam.ROUTE, recordContext.route().lowerName()));
            MultipartEntityBuilder builder = MultipartEntityBuilder.create();
            builder.setMimeSubtype("mixed");
            String body;
            Map<Long, List<LogRecordContext>> batches = new HashMap<>();
            for (LogRecordContext logRecordContext : batch.records()) {
                batches.computeIfAbsent(logRecordContext.prefix(), x -> new ArrayList<>()).add(logRecordContext);
            }
            for (Map.Entry<Long, List<LogRecordContext>> entry : batches.entrySet()) {
                prefix = entry.getKey();
                body =
                    new String(
                        prepareRequestBody(logger, baos, entry.getValue(), prefix, ttl),
                        StandardCharsets.UTF_8);
                if (recordContext.handlerType() == LogRecordsHandlerType.SP_DAEMON) {
                    StringBuilder queueIds = new StringBuilder();
                    queueIds.append(((DeliveryLogRecordContext) entry.getValue().get(0)).queueId());
                    for (int i = 1; i < entry.getValue().size(); i++) {
                        queueIds.append(',').append(((DeliveryLogRecordContext) entry.getValue().get(i)).queueId());
                    }
                    params.put(SearchParam.QUEUEID, queueIds.toString());
                }
                //logger.info("saveIndexData: prefix=" + prefix + ", body=" + body);
                builder.addPart(
                    FormBodyPartBuilder
                        .create()
                        .addField(YandexHeaders.ZOO_SHARD_ID, Long.toString(prefix % SearchMap.SHARDS_COUNT))
                        .addField(YandexHeaders.URI, prepareUri(prefix, params))
                        .setBody(new StringBody(body, producerContentType))
                        .setName("envelope.json")
                        .build());
            }
            try {
                producerGenerator = new BasicAsyncRequestProducerGenerator(uri, builder.build());
            } catch (IOException e) {
                logger.log(Level.SEVERE, "saveIndexData failed to create BasicAsyncRequestProducerGenerator", e);
                callback.failed(e);
                return;
            }
        }
        producerGenerator.addHeader(YandexHeaders.SERVICE, spLogger.config().indexingQueueName());
        AsyncClient client = spLogger.producerAsyncClient();
        HttpHost producerHost = spLogger.config().producerClientConfig().host();
        ImmutableStatersConfig statersConfig = spLogger.producerAsyncClient().statersConfig();
        if (statersConfig != null) {
            client = client.adjustStater(
                statersConfig,
                new RequestInfo(new BasicHttpRequest(RequestHandlerMapper.POST, producerHost.toURI())));
        }
        client.execute(producerHost, producerGenerator, EmptyAsyncConsumerFactory.OK, callback);
        //logger.info("MdsLogStorage: request for save index data has sent");
    }

    public static String prepareUri(final Long prefix, final Map<SearchParam, Object> params) throws BadRequestException
    {
        QueryConstructor query = new QueryConstructor(UPDATE_URI);
        if (prefix != null) {
            query.append(LogContext.PREFIX, prefix);
        }
        for (Map.Entry<SearchParam, Object> entry : params.entrySet()) {
            if ("Long".equals(entry.getKey().itemType())) {
                query.append(entry.getKey().paramName(), (Long) entry.getValue());
            } else if ("Integer".equals(entry.getKey().itemType())) {
                query.append(entry.getKey().paramName(), (Integer) entry.getValue());
            } else {
                query.append(entry.getKey().paramName(), (String) entry.getValue());
            }
        }
        return query.toString();
    }

    public static byte[] prepareRequestBody(
        final Logger logger,
        final ByteArrayOutputStream baos,
        final List<? extends LogRecordContext> recordContexts,
        final long prefix,
        final long ttl)
        throws BadRequestException
    {
        String route;
        List<Long> rcptUids = new ArrayList<>();
        baos.reset();
        try (Utf8JsonWriter writer = JsonType.NORMAL.create(baos)) {
            writer.startObject();
            writer.key(LogContext.PREFIX);
            writer.value(prefix);
            writer.key("docs");
            writer.startArray();
            for (LogRecordContext logRecordContext : recordContexts) {
                rcptUids.clear();
                DeliveryLogRecordContext recordContext = (DeliveryLogRecordContext) logRecordContext;
                route = recordContext.route() == null ? null : recordContext.route().lowerName();
                if (recordContext.recipientUids().size() > 0) {
                    rcptUids.addAll(recordContext.recipientUids());
                } else {
                    rcptUids.add(0L);
                }
                for (long rcptUid : rcptUids) {
                    writer.startObject();
                    writer.key(IndexField.ID.fieldName());
                    writer.value(LogRecordContext.logId(
                        logRecordContext.name(),
                        route,
                        recordContext.queueId(),
                        rcptUid));
                    writer.key(IndexField.QUEUEID.fieldName());
                    writer.value(recordContext.queueId());
                    writer.key(IndexField.TYPE.fieldName());
                    writer.value(logRecordContext.name());
                    writer.key(IndexField.ROUTE.fieldName());
                    writer.value(route);
                    writer.key(IndexField.TS.fieldName());
                    writer.value(recordContext.messageDate());
                    writer.key(IndexField.EXPIRE_TIMESTAMP.fieldName());
                    writer.value(ttl);
                    if (recordContext.soResolution() != null) {
                        writer.key(IndexField.CODE.fieldName());
                        writer.value(recordContext.soResolution().code());
                    }
                    writer.key(IndexField.UID.fieldName());
                    writer.value(recordContext.uid());
                    writer.key(IndexField.RCPT_UID.fieldName());
                    writer.value(rcptUid);
                    writer.key(IndexField.MSGID.fieldName());
                    writer.value(recordContext.messageId());
                    writer.key(IndexField.FROMADDR.fieldName());
                    writer.value(recordContext.fromAddress());
                    writer.key(IndexField.SOURCE_IP.fieldName());
                    writer.value(recordContext.sourceIp());
                    writer.key(IndexField.MX.fieldName());
                    writer.value(recordContext.mailBackend());
                    writer.key(IndexField.LOCL.fieldName());
                    writer.value(recordContext.localHost());
                    writer.key(IndexField.STID.fieldName());
                    writer.value(recordContext.storageKey());
                    if (recordContext.storageType() == LogStorageType.MDS) {
                        writer.key(IndexField.OFFSET.fieldName());
                        writer.value(recordContext.offset());
                        writer.key(IndexField.BYTES_OFFSET.fieldName());
                        writer.value(recordContext.byteOffset());
                    }
                    writer.key(IndexField.SIZE.fieldName());
                    writer.value(recordContext.logSize());
                    writer.key(IndexField.BYTES_SIZE.fieldName());
                    writer.value(recordContext.logBytesSize());
                    writer.endObject();
                }
            }
            writer.endArray();
            writer.endObject();
            writer.flush();
        } catch (IOException e) {
            throw new BadRequestException(e);
        }
        return baos.toByteArray();
    }

    private static class MdsGetResultCallback
        extends AbstractFilterFutureCallback<HttpResponse, LogRecordsBatch<LogRecordContext, LogStorageConfig>>
    {
        private static final String MDS_GET_ERROR = "MdsGetResultCallback: MDS get failed!";

        private final LogRecordsContext<LogRecordContext> context;
        private final MdsLogRecordsBatch batch;

        public MdsGetResultCallback(
            final LogRecordsContext<LogRecordContext> context,
            final MdsLogRecordsBatch batch,
            final FutureCallback<LogRecordsBatch<LogRecordContext, LogStorageConfig>> callback)
        {
            super(callback);
            this.context = context;
            this.batch = batch;
        }

        @Override
        public void completed(final HttpResponse response) {
            if (response == null) {
                callback.completed(null);
                return;
            }
            try {
                HttpEntity entity = response.getEntity();
                if (entity == null || entity.getContentLength() < 10) { // content empty or invalid
                    callback.completed(null);
                    return;
                }
                if (batch.byteOffsets().size() > 1) {
                    handleMultipartPayload(entity);
                } else if (batch.byteOffsets().size() == 1) {
                    //session.logger().info("MdsGetResultCallback obtained simple answer")
                    Header[] contentRangeHeaders = response.getHeaders(CONTENT_RANGE);
                    handleSimplePayload(entity, contentRangeHeaders.length > 0);
                } else {
                    context.session().logger().info("MdsGetResultCallback: there are no record offsets!");
                }
                callback.completed(batch);
            } catch (IOException e) {
                context.session().logger().log(Level.SEVERE, "MdsGetResultCallback failed to parse MDS answer", e);
                callback.failed(new LogStorageException(batch.type(), batch.namespace(), MDS_GET_ERROR, e));
            }
        }

        @Override
        public void failed(final Exception e) {
            context.session().logger().log(Level.WARNING, "MdsGetResultCallback failed: "
                + context.session().listener().details() + " because of exception: " + e.toString(), e);
            callback.completed(null);
        }

        @Override
        public void cancelled() {
            context.session().logger().warning("MdsGetResultCallback request cancelled: "
                + context.session().listener().details());
            callback.cancelled();
        }

        private void handleSimplePayload(final HttpEntity entity, final boolean contentRange) throws IOException {
            ByteArrayOutputStream out = new ByteArrayOutputStream();
            entity.writeTo(out);
            int offset = 0;
            int size;
            byte[] body = out.toByteArray();
            String sBody = new String(body, StandardCharsets.UTF_8);
            byte[] log;
            for (Map.Entry<Integer, List<LogRecordContext>> entry : batch.byteOffsets().entrySet()) {
                LogRecordContext record = entry.getValue().get(0);
                if (record.logBytesSize() > 0) {
                    if (!contentRange) {
                        offset = record.byteOffset();
                    }
                    size = record.logBytesSize();
                    //session.logger().info("MdsGetResultCallback: obtained bytes offset=" + offset + ", size=" + size
                    //    + ", charsCnt=" + sBody.length() + ", bytesCnt=" + sBody.getBytes(StandardCharsets.UTF_8).length
                    //    + " log=" + sBody);
                    batch.setLogByOffset(entry.getKey(), Arrays.copyOfRange(body, offset, offset + size));
                    if (contentRange) {
                        offset += size;
                    }
                } else if (record.logSize() > 0) {
                    if (!contentRange) {
                        offset = record.offset();
                    }
                    size = record.logSize();
                    log = sBody.substring(offset, offset + size).getBytes(StandardCharsets.UTF_8);
                    batch.setLogByOffset(entry.getKey(), log);
                    //session.logger().info("MdsGetResultCallback: obtained chars offset=" + offset + ", size=" + size
                    //    + " log=" + new String(log, StandardCharsets.UTF_8));
                    if (contentRange) {
                        offset += size;
                    }
                }
            }
            //session.logger().info("MdsGetResultCallback: obtained log="
            //    + new String(batch.content(), StandardCharsets.UTF_8));
        }

        private void handleMultipartPayload(final HttpEntity entity) throws IOException {
            SafeMimeTokenStream tokenStream = new SafeMimeTokenStream();
            tokenStream.setRecursionMode(RecursionMode.M_NO_RECURSE);
            //session.logger().info("MdsGetResultCallback.handleMultipartPayload: contentType="
            //    + entity.getContentType());
            tokenStream.parseHeadless(entity.getContent(), entity.getContentType().getValue());
            try {
                Integer byteOffset = null;
                EntityState state = tokenStream.getState();
                while (state != EntityState.T_END_OF_STREAM) {
                    //session.logger().info("MdsGetResultCallback.handleMultipartPayload: state=" + state
                    //    + ", byteOffset=" + byteOffset);
                    if (state == EntityState.T_START_BODYPART) {
                        byteOffset = null;
                        while (state != EntityState.T_END_BODYPART) {
                            //session.logger().info("MdsGetResultCallback.handleMultipartPayload: bodypart state="
                            //    + state + ", byteOffset=" + byteOffset);
                            if (byteOffset == null) {
                                byteOffset = handleMultipartPayloadHelper(tokenStream, state, null);
                            } else {
                                handleMultipartPayloadHelper(tokenStream, state, byteOffset);
                            }
                            state = tokenStream.next();
                        }
                    } else {
                        if (byteOffset == null) {
                            byteOffset = handleMultipartPayloadHelper(tokenStream, state, null);
                        } else {
                            handleMultipartPayloadHelper(tokenStream, state, byteOffset);
                        }
                    }
                    state = tokenStream.next();
                }
                //session.logger().info("MdsGetResultCallback.handleMultipartPayload successfully finished");
            } catch (MimeException | BadRequestException e) {
                context.session().logger().info("MdsGetResultCallback.handleMultipartPayload failed: " + e);
                failed(e);
            }
        }

        private Integer handleMultipartPayloadHelper(
            final SafeMimeTokenStream tokenStream,
            final EntityState state,
            final Integer byteOffset)
            throws BadRequestException, IOException
        {
            if (state == EntityState.T_FIELD) {
                Field field = tokenStream.getField();
                if (CONTENT_RANGE.equals(field.getName())) {
                    Matcher m = CONTENT_RANGE_RE.matcher(field.getBody());
                    if (m.find()) {
                        return Integer.parseInt(m.group(1));
                    }
                }
            } else if (state == EntityState.T_BODY) {
                byte[] entityData = IOStreamUtils.consume(tokenStream.getInputStream()).toByteArray();
                if (byteOffset == null) {
                    throw new BadRequestException("No valid info found about content bytes range for part: "
                        + new String(entityData, StandardCharsets.UTF_8));
                }
                batch.setLogByOffset(byteOffset, entityData);
            }
            return null;
        }
    }

    private static class MdsPutResultCallback implements FutureCallback<HttpResponse>
    {
        private final MdsLogStorage logStorage;
        private final Logger logger;
        private final MdsLogRecordsBatch batch;

        MdsPutResultCallback(final MdsLogStorage logStorage, final Logger logger, final MdsLogRecordsBatch batch) {
            this.logStorage = logStorage;
            this.logger = logger;
            this.batch = batch;
        }

        @Override
        public void completed(final HttpResponse response) {
            HttpEntity entity = response.getEntity();
            try {
                DocumentBuilder documentBuilder = DocumentBuilderFactory.newInstance().newDocumentBuilder();
                Document document = documentBuilder.parse(entity.getContent());
                NodeList nodes = document.getElementsByTagName("post");
                String stid = nodes.item(0).getAttributes().getNamedItem("key").getNodeValue();
                logger.severe("MDS put: obtained STID = '" + stid + "', offset=" + batch.records().get(0).offset()
                    + ", logSize=" + batch.records().get(0).logSize());
                batch.setStid(stid);
                try {
                    logStorage.saveIndexData(logger, batch, new SaveIndexCallback(logStorage, logger, batch));
                } catch (BadRequestException r) {
                    logger.log(
                        Level.SEVERE,
                        "MdsPutResultCallback: failed to save index data for stid=" + batch.stid(),
                        r);
                    logStorage.decrementBatchesCount();
                }
                batchesSize.updateAndGet(x -> x - batch.contentBytesSize());
            } catch (ParserConfigurationException | SAXException | IOException | RuntimeException e) {
                logger.log(
                    Level.SEVERE,
                    "MdsPutResultCallback: error occurred while parsing of received MDS answer's XML for stid="
                        + batch.stid(),
                    e);
                failed(e);
            }
        }

        @Override
        public void failed(final Exception e) {
            logger.log(Level.SEVERE, "MdsPutResultCallback: MDS put failed!", e);
            batchesSize.updateAndGet(x -> x - batch.contentBytesSize());
            if (BatchLogSaver.retriableFailure(e)) {
                logStorage.mdsRetriableFailures();
                batch.mdsRetriesInc();
                if (batch.retriesCount() <= logStorage.config().batchSaveRetries()) {
                    logStorage.mdsWriterClient.scheduleRetry(new TimerTask() {
                        @Override
                        public void run() {
                            logStorage.saveBatch(batch, logger);
                        }
                    }, logStorage.config().batchSaveRetryTimeout());
                } else {
                    logStorage.mdsRetriesExceededFailures();
                }
            } else {
                logStorage.mdsNonRetriableFailures();
            }
            logStorage.decrementBatchesCount();
        }

        @Override
        public void cancelled() {
            logger.warning("MdsPutResultCallback: further batch's processing cancelled!");
            batchesSize.updateAndGet(x -> x - batch.contentBytesSize());
            logStorage.decrementBatchesCount();
        }
    }

    private static class SaveIndexCallback implements FutureCallback<Void> {
        private final MdsLogStorage logStorage;
        private final Logger logger;
        private final MdsLogRecordsBatch batch;

        public SaveIndexCallback(final MdsLogStorage logStorage, final Logger logger, final MdsLogRecordsBatch batch) {
            this.logStorage = logStorage;
            this.logger = logger;
            this.batch = batch;
        }

        @Override
        public void completed(final Void unused) {
            logger.info("SaveIndexCallback successfully done for stid=" + batch.stid());
            logStorage.decrementBatchesCount();
        }

        @Override
        public void failed(final Exception e) {
            logger.log(Level.SEVERE, "SaveIndexCallback failed for stid=" + batch.stid(), e);
            if (BatchLogSaver.retriableFailure(e)) {
                logStorage.luceneRetriableFailures();
                batch.luceneRetriesInc();
                if (batch.luceneRetriesCount() <= logStorage.config().indexRetriesCount()) {
                    logStorage.mdsWriterClient.scheduleRetry(new TimerTask() {
                        @Override
                        public void run() {
                            try {
                                logStorage.saveIndexData(
                                    logger,
                                    batch,
                                    new SaveIndexCallback(logStorage, logger, batch));
                            } catch (BadRequestException e) {
                                logger.log(
                                    Level.SEVERE,
                                    "SaveIndexCallback repeated request failed for stid=" + batch.stid(),
                                    e);
                                logStorage.luceneNonRetriableFailures();
                            }
                        }
                    }, logStorage.config().batchSaveRetryTimeout());
                } else {
                    logStorage.luceneRetriesExceededFailures();
                }
            } else {
                logStorage.luceneNonRetriableFailures();
            }
            logStorage.decrementBatchesCount();
        }

        @Override
        public void cancelled() {
            logger.warning("SaveIndexCallback: further batch's processing cancelled for stid=" + batch.stid());
            logStorage.decrementBatchesCount();
        }
    }

    private static class MdsWriteHostRenewer implements GenericAutoCloseable<RuntimeException>, Runnable
    {
        private final MdsLogStorage logStorage;
        private final Logger logger;
        private final long mdsHostnameRequestPeriod;
        private final Thread thread;
        private final Consumer<? super Exception> errorHandler;
        private volatile boolean closed = false;

        public MdsWriteHostRenewer(
            final MdsLogStorage logStorage,
            final Logger logger,
            final long mdsHostnameRequestPeriod,
            final ThreadFactory threadFactory)
        {
            this.logStorage = logStorage;
            this.logger = logger;
            this.mdsHostnameRequestPeriod = mdsHostnameRequestPeriod;
            this.errorHandler = e -> this.logger.log(Level.WARNING, "MDS-write-hostname requester failed: " + e, e);
            thread = threadFactory.newThread(this);
        }

        public void start() {
            thread.start();
        }

        @Override
        public void close() {
            closed = true;
            thread.interrupt();
        }

        @Override
        public void run() {
            while (!closed) {
                synchronized (logStorage.mdsWriterClient) {
                    try {
                        logStorage.mdsWriterClient.wait(mdsHostnameRequestPeriod);
                    } catch (InterruptedException e) {
                        break;
                    }
                    try {
                        logStorage.requestMdsWriteHostname(logger);
                    } catch (Exception e) {
                        errorHandler.accept(e);
                    }
                }
            }
        }
    }

    @SuppressWarnings("FutureReturnValueIgnored")
    private void requestMdsWriteHostname(final Logger logger) {
        AsyncClient client = mdsWriterClient;
        ImmutableStatersConfig statersConfig = config().backendWriterConfig().statersConfig();
        if (statersConfig != null) {
            client = mdsWriterClient.adjustStater(
                statersConfig,
                new RequestInfo(
                    new BasicHttpRequest(RequestHandlerMapper.GET, config().backendWriterConfig().host().toURI())));
        }
        BasicAsyncRequestProducerGenerator producerGenerator = new BasicAsyncRequestProducerGenerator("/hostname");
        //producerGenerator.addHeader(HeaderUtils.createHeader(YandexHeaders.X_YA_SERVICE_TICKET, mdsTvm2Ticket()));
        client.execute(
            config().backendWriterConfig().host(),
            producerGenerator,
            AsyncStringConsumerFactory.ANY_GOOD,
            new MdsWriteHostnameCallback(this, logger));
    }

    private static class MdsWriteHostnameCallback implements FutureCallback<String> {
        private final MdsLogStorage logStorage;
        private final Logger logger;

        MdsWriteHostnameCallback(final MdsLogStorage logStorage, final Logger logger) {
            this.logStorage = logStorage;
            this.logger = logger;
        }

        @Override
        public void completed(final String response) {
            if (response == null || response.trim().isEmpty()) {
                logger.log(Level.SEVERE, "MdsWriteHostnameCallback: empty response!");
                return;
            }
            synchronized (logStorage.mdsWriterClient) {
                logStorage.currentMdsWriteHost = response.trim();
            }
            logger.info("MdsWriteHostnameCallback: new host for MDS writing - '" + response + "'");
        }

        @Override
        public void failed(Exception e) {
            logger.log(Level.SEVERE, "MdsWriteHostnameCallback failed", e);
        }

        @Override
        public void cancelled() {
            logger.log(Level.SEVERE, "MdsWriteHostnameCallback cancelled");
        }
    }
}
