package ru.yandex.mail.so.logger;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Timer;
import java.util.TimerTask;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.logging.Level;
import java.util.logging.Logger;

import com.mongodb.Function;
import com.mongodb.MongoClientException;
import com.mongodb.MongoException;
import com.mongodb.ReadConcern;
import com.mongodb.ReadPreference;
import com.mongodb.TransactionOptions;
import com.mongodb.WriteConcern;
import com.mongodb.bulk.BulkWriteResult;
import com.mongodb.client.model.UpdateOneModel;
import com.mongodb.client.model.UpdateOptions;
import com.mongodb.client.model.Updates;
import com.mongodb.client.result.UpdateResult;
import com.mongodb.lang.NonNull;
import com.mongodb.lang.Nullable;
import com.mongodb.reactivestreams.client.ClientSession;
import com.mongodb.reactivestreams.client.MongoClient;
import com.mongodb.reactivestreams.client.MongoCollection;
import com.mongodb.reactivestreams.client.MongoDatabase;
import org.apache.http.concurrent.FutureCallback;
import org.bson.BsonDocument;
import org.bson.BsonDocumentWriter;
import org.bson.Document;
import org.bson.codecs.Encoder;
import org.bson.codecs.EncoderContext;
import org.bson.codecs.configuration.CodecRegistry;
import org.bson.conversions.Bson;
import org.joda.time.DateTime;
import org.joda.time.DateTimeZone;
import org.joda.time.format.DateTimeFormat;
import org.joda.time.format.DateTimeFormatter;
import reactor.core.publisher.Mono;

import ru.yandex.concurrent.TimeFrameQueue;
import ru.yandex.function.GenericAutoCloseable;
import ru.yandex.http.proxy.HttpProxy;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.DoubleFutureCallback;
import ru.yandex.http.util.MultiFutureCallback;
import ru.yandex.mail.so.logger.config.BatchSaverConfig;
import ru.yandex.mail.so.logger.config.EnvironmentType;
import ru.yandex.mail.so.logger.config.MongoRulesStatDatabaseConfig;
import ru.yandex.mail.so.logger.config.RulesStatDatabaseConfig;
import ru.yandex.mail.so.logger.config.RulesStatDatabasesOperatorConfig;
import ru.yandex.util.timesource.TimeSource;

import static com.mongodb.assertions.Assertions.notNull;

public class MongoRulesStatDatabase extends AbstractRulesStatDatabase<BasicRoutedLogRecordProducer> {
    //public static final AtomicLong batchesSize = new AtomicLong(0L);

    private static final DateTimeFormatter formatter = DateTimeFormat.forPattern("yyyy-MM-dd");
    private static final DateTimeZone TIMEZONE = DateTimeZone.forID("Europe/Moscow");
    private final HttpProxy<?> httpProxy;
    private final RulesStatDatabasesOperatorConfig<BasicRoutedLogRecordProducer> rulesStatDatabasesOperatorConfig;
    private volatile SpDaemonRulesStatBatch currentBatch;
    private final BatchSaver<BasicRoutedLogRecordProducer, RulesStatDatabaseConfig<BasicRoutedLogRecordProducer>>
        batchSaver;
    private final ThreadPoolExecutor executor;
    private final MongoClient mongoClient;
    private final Timer timer = new Timer(name() + "-RetryTimer", true);
    private final TimeFrameQueue<Long> batchSize;
    private final TimeFrameQueue<Long> batchCapacity;

    public MongoRulesStatDatabase(
        final HttpProxy<?> httpProxy,
        final RulesStatDatabasesOperatorConfig<BasicRoutedLogRecordProducer> rulesStatDatabasesOperatorConfig,
        final String dbConfigName)
    {
        super(
            dbConfigName,
            rulesStatDatabasesOperatorConfig.rulesStatDatabasesConfig().rulesStatDatabases().get(dbConfigName));
        this.httpProxy = httpProxy;
        this.rulesStatDatabasesOperatorConfig = rulesStatDatabasesOperatorConfig;
        String prefix = dbConfigName + '-';
        batchSize = new TimeFrameQueue<>(httpProxy.config().metricsTimeFrame());
        batchCapacity = new TimeFrameQueue<>(httpProxy.config().metricsTimeFrame());
        registerBatchHandlerStaters(httpProxy, prefix);
        currentBatch = new SpDaemonRulesStatBatch();
        batchSaver = createBatchSaver(httpProxy);
        executor = createThreadPoolExecutor(
            new ThreadGroup(httpProxy.getThreadGroup(), name() + "-AsyncWorker"),
            config.workers());
        httpProxy.logger().info("MongoRulesStatDatabase (" + rulesStatDatabasesOperatorConfig.envType()
            + "): MongoDB connection string = " + config().connectionString());
        mongoClient = config().createMongoClient();
    }

    @Override
    public RulesStatDatabaseType type() {
        return RulesStatDatabaseType.MONGODB;
    }

    @Override
    public BatchSaver<BasicRoutedLogRecordProducer, RulesStatDatabaseConfig<BasicRoutedLogRecordProducer>> batchSaver()
    {
        return batchSaver;
    }

    @Override
    public String name() {
        return "MongoRulesStat";
    }

    @Override
    public Logger logger() {
        return httpProxy.logger();
    }

    @Override
    public MongoRulesStatDatabaseConfig config() {
        return (MongoRulesStatDatabaseConfig) config;
    }

    @Override
    public TimeFrameQueue<Long> batchSize() {
        return batchSize;
    }

    @Override
    public TimeFrameQueue<Long> batchCapacity() {
        return batchCapacity;
    }

    public MongoClient mongoClient() {
        return mongoClient;
    }

    @Override
    public SpDaemonRulesStatBatch currentBatch() {
        return currentBatch;
    }

    @Override
    public synchronized SpDaemonRulesStatBatch resetBatch() {
        SpDaemonRulesStatBatch oldBatch = currentBatch;
        currentBatch = new SpDaemonRulesStatBatch();
        return oldBatch;
    }

    @Override
    public boolean batchIsEmpty() {
        return currentBatch.isEmpty();
    }

    public void stat(final MongoResultType errorType, final long duration) {
        rulesStatDatabasesOperatorConfig.rulesStatDatabasesConfig().stat(configName, errorType, duration);
    }

    public void stat(final Exception e, final long duration) {
        MongoResultType errorType;
        if (e instanceof MongoException) {
            MongoException me = (MongoException) e;
            errorType = MongoResultType.fromCode(me.getCode());
        } else {
            errorType = MongoResultType.OTHER;
        }
        stat(errorType, duration);
    }

    @Override
    public void save(
        final BasicRoutedLogRecordProducer logRecordProducer,
        final ProxySession session,
        final FutureCallback<Void> callback)
    {
        //session.logger().info("MongoRulesStatDatabase.saveStat: start adding data for route=" + route);
        if (logRecordProducer.route() == null) {
            session.logger().warning("MongoRulesStatDatabase.saveStat skipped because of route=null and therefore "
                + "I don't know in which collection this log record should be saved.");
        } else {
            synchronized (MongoRulesStatDatabase.this) {
                try {
                    currentBatch.add(logRecordProducer);
                    //session.logger().info("MongoRulesStatDatabase.saveStat: record added to batch");
                } catch (Exception e) {
                    session.logger().log(Level.SEVERE, "MongoRulesStatDatabase.saveStat failed", e);
                }
                if (currentBatch.lifeTime() > config.batchSavePeriod()) {
                    currentBatch.ready();
                }
                notifyAll();
            }
        }
        callback.completed(null);
    }

    @Override
    public boolean saveBatch(final Batch<BasicRoutedLogRecordProducer> batch, final Logger logger) {
        if (!(batch instanceof SpDaemonRulesStatBatch)) {
            logger.info("MongoRulesStatDatabase.saveBatch: batch is null or unknown type");
            return false;
        }
        if (batch.isEmpty()) {
            logger.info("MongoRulesStatDatabase.saveBatch: batch is empty");
            batch.notReady();
            return false;
        }
        if (!batch.isReady() && batch.lifeTime() <= config.batchSavePeriod()) {
            logger.info("MongoRulesStatDatabase.saveBatch: batch is not ready");
            return false;
        }
        SpDaemonRulesStatBatch rulesStatBatch = (SpDaemonRulesStatBatch) batch;
        for (Route route : rulesStatBatch.routes()) {
            Map<Long, Map<String, Map<String, Long>>> data = rulesStatBatch.routeData(route);
            if (data.size() < 1) {
                continue;
            }
            executor.execute(new SavingStatistics(
                this,
                data,
                route,
                tableName(route, rulesStatDatabasesOperatorConfig.envType()),
                config.batchSaveRetries()));
            batchSize(rulesStatBatch.count());
            batchCapacity(data.entrySet().iterator().next().getValue().size());
        }
        logger.info("MongoRulesStatDatabase.saveBatch: successfully start to save rules statistics");
        return true;
    }

    public void scheduleRetry(final SavingStatistics task, final long delay) {
        if (delay == 0L) {
            executor.execute(task);
        } else {
            try {
                timer.schedule(new TimerTask() {
                    @Override
                    public void run() {
                        executor.execute(task);
                    }
                }, delay);
            } catch (RuntimeException e) {
                httpProxy.logger().log(
                    Level.SEVERE,
                    "Failed to schedule SavingStatistics task for route = " + task.route(),
                    e);
                executor.execute(task);
            }
        }
    }

    public static class SavingStatistics extends TimerTask implements GenericAutoCloseable<RuntimeException>
    {
        private final MongoRulesStatDatabase rulesStatDatabase;
        private final MongoClient mongoClient;
        private final MongoRulesStatDatabaseConfig config;
        private final Map<Long, Map<String, Map<String, Long>>> data;
        private final Route route;
        private final String tableName;
        private final int retriesLeft;
        private final Logger logger;
        private long startTime;
        private final CountDownLatch latch;

        public SavingStatistics(
            final MongoRulesStatDatabase rulesStatDatabase,
            final Map<Long, Map<String, Map<String, Long>>> data,
            final Route route,
            final String tableName,
            final int retriesLeft)
        {
            this.rulesStatDatabase = rulesStatDatabase;
            this.mongoClient = rulesStatDatabase.mongoClient();
            this.config = rulesStatDatabase.config();
            this.data = data;
            this.route = route;
            this.tableName = tableName;
            this.retriesLeft = retriesLeft - 1;
            this.logger = rulesStatDatabase.logger();
            latch = new CountDownLatch(1);
            startTime = 0L;
        }

        public Route route() {
            return route;
        }

        @Override
        public void run() {
            startTime = TimeSource.INSTANCE.currentTimeMillis();
            try (this) {
                saveStatistics(retriesLeft);
            } catch (Exception e) {
                logger.log(Level.SEVERE, "SavingStatistics failed (latch=" + latch.getCount() + ")", e);
                if (latch.getCount() > 0) {
                    latch.countDown();
                }
                rulesStatDatabase.decrementBatchesCount();
                rulesStatDatabase.stat(e, TimeSource.INSTANCE.currentTimeMillis() - startTime);
            }
        }

        @Override
        public void close() throws RuntimeException {
            try {
                latch.await();
            } catch (InterruptedException e) {
                throw new RuntimeException("SavingStatistics finishing is interrupted", e);
            }
        }

        public void saveStatistics(@NonNull final int retriesLeft) {
            logger.info("saveStatistics started: attempt #" + (config.batchSaveRetries() - retriesLeft));
            Map<String, Map<String, Map<String, Long>>> dailyInfo = new HashMap<>();
            DoubleFutureCallback<Void, Void> doubleCallback =
                new DoubleFutureCallback<>(new SavingStatisticsCallback<>(route, retriesLeft));
            ClientSession sessionDetailed = startSession(mongoClient, logger);
            if (sessionDetailed != null
                    || config.savingMethod() == MongoRulesStatDatabaseConfig.SavingMethod.SYNC_REQUESTS)
            {
                try {
                    if (config.savingMethod() == MongoRulesStatDatabaseConfig.SavingMethod.TRANSACTIONS) {
                        dailyInfo = runTransactionWithRetry(
                            sessionDetailed,
                            config,
                            data,
                            logger,
                            statData -> saveDetailedStatistics(
                                sessionDetailed,
                                mongoClient,
                                config,
                                "detailed_" + tableName,
                                statData,
                                route,
                                retriesLeft,
                                logger,
                                doubleCallback.first()),
                            retriesLeft);
                    } else {
                        dailyInfo = saveDetailedStatistics(
                            sessionDetailed,
                            mongoClient,
                            config,
                            "detailed_" + tableName,
                            data,
                            route,
                            retriesLeft,
                            logger,
                            doubleCallback.first());
                    }
                    logger.info("saveStatistics successfully sent detailed rules statistics for route = " + route);
                } catch (Exception e) {
                    logger.info("saveStatistics failed to send detailed rules statistics for route = " + route);
                    doubleCallback.first().failed(e);
                }
            } else {
                doubleCallback.first().failed(new MongoClientException("Unable to open client session"));
            }
            ClientSession sessionDaily = startSession(mongoClient, logger);
            if ((sessionDaily != null
                    || config.savingMethod() == MongoRulesStatDatabaseConfig.SavingMethod.SYNC_REQUESTS)
                    && dailyInfo != null && dailyInfo.size() > 0)
            {
                try {
                    if (config.savingMethod() == MongoRulesStatDatabaseConfig.SavingMethod.TRANSACTIONS) {
                        runTransactionWithRetry(
                            sessionDaily,
                            config,
                            dailyInfo,
                            logger,
                            statData -> saveDailyStatistics(
                                sessionDaily,
                                mongoClient,
                                config,
                                tableName,
                                statData,
                                route,
                                retriesLeft,
                                logger,
                                doubleCallback.second()),
                            retriesLeft);
                    } else {
                        saveDailyStatistics(
                            sessionDaily,
                            mongoClient,
                            config,
                            tableName,
                            dailyInfo,
                            route,
                            retriesLeft,
                            logger,
                            doubleCallback.second());
                    }
                    logger.info("MongoRulesStatDatabase.saveStatistics successfully sent daily rules statistics for "
                        + "route = " + route);
                } catch (Exception e) {
                    logger.info("MongoRulesStatDatabase.saveStatistics failed to send daily rules statistics for "
                        + "route = " + route);
                    doubleCallback.second().failed(e);
                }
            } else {
                doubleCallback.second().failed(
                    new MongoClientException("Unable to open client session or unable to obtain daily statistics"));
            }
            logger.info("MongoRulesStatDatabase.saveStatistics finished");
        }

        private class SavingStatisticsCallback<T> implements FutureCallback<T>
        {
            private final Route route;
            private final int retriesLeft;

            public SavingStatisticsCallback(final Route route, final int retriesLeft) {
                this.route = route;
                this.retriesLeft = retriesLeft;
            }

            @Override
            public void completed(final T ignored) {
                logger.info("SavingStatisticsCallback: rules statistics saved for route = " + route);
                rulesStatDatabase.decrementBatchesCount();
                rulesStatDatabase.stat(MongoResultType.SUCCESS, TimeSource.INSTANCE.currentTimeMillis() - startTime);
                latch.countDown();
            }

            @Override
            public void failed(Exception e) {
                logger.log(Level.SEVERE, "SavingStatisticsCallback failed (" + retriesLeft + " attempts left)", e);
                if (e instanceof MongoException
                        && config.savingMethod() == MongoRulesStatDatabaseConfig.SavingMethod.TRANSACTIONS)
                {
                    MongoException me = (MongoException) e;
                    if (me.hasErrorLabel(MongoException.TRANSIENT_TRANSACTION_ERROR_LABEL) && retriesLeft != 0) {
                        logger.warning("SavingStatisticsCallback failed: there are " + retriesLeft
                            + " attempts left");
                        rulesStatDatabase.scheduleRetry(
                            new SavingStatistics(rulesStatDatabase, data, route, tableName, retriesLeft),
                            config.batchSaveRetryTimeout()
                        );
                        return;
                    }
                } else {
                    logger.log(Level.SEVERE, "SavingStatisticsCallback failed", e);
                }
                if (retriesLeft == 0) {
                    rulesStatDatabase.decrementBatchesCount();
                    rulesStatDatabase.stat(e, TimeSource.INSTANCE.currentTimeMillis() - startTime);
                }
                latch.countDown();
            }

            @Override
            public void cancelled() {
                if (config.savingMethod() == MongoRulesStatDatabaseConfig.SavingMethod.TRANSACTIONS) {
                    logger.warning("SavingStatisticsCallback cancelled on attempt #"
                        + (config.batchSaveRetries() - retriesLeft) + ')');
                } else {
                    logger.warning("SavingStatisticsCallback cancelled");
                }
                if (retriesLeft == 0) {
                    rulesStatDatabase.decrementBatchesCount();
                    rulesStatDatabase.stat(MongoResultType.OTHER, TimeSource.INSTANCE.currentTimeMillis() - startTime);
                }
                latch.countDown();
            }
        }
    }

    private static Map<String, Map<String, Map<String, Long>>> saveDetailedStatistics(
        final ClientSession session,
        final MongoClient mongoClient,
        final MongoRulesStatDatabaseConfig config,
        final String tableName,
        final Map<Long, Map<String, Map<String, Long>>> data,
        final Route route,
        final int retries,
        final Logger logger,
        final FutureCallback<Void> callback)
    {
        long counter;
        MongoDatabase database = mongoClient.getDatabase(config.dbName());
        Map<String, Map<String, Map<String, Long>>> dailyInfo = new HashMap<>();
        MultiFutureCallback<Void> detailedMultiCallback =
            new MultiFutureCallback<>(
                new MongoBatchOperationCallback(
                    session,
                    config,
                    "detailed statistics of " + route + " rules",
                    retries,
                    logger,
                    callback));
        MongoCollection<Document> collection = database.getCollection(tableName);
        UpdateOptions updateOps = new UpdateOptions().upsert(true);
        List<UpdateOneModel<Document>> bulkUpdates = new ArrayList<>();
        for (Map.Entry<Long, Map<String, Map<String, Long>>> entry : data.entrySet()) {
            String date = isoDate(entry.getKey());
            dailyInfo.computeIfAbsent(date, x -> new HashMap<>());
            for (Map.Entry<String, Map<String, Long>> ruleStat : entry.getValue().entrySet()) {
                dailyInfo.get(date).computeIfAbsent(ruleStat.getKey(), x -> new HashMap<>());
                try {
                    List<Bson> ruleStatInfo = new ArrayList<>();
                    for (Map.Entry<String, Long> info : ruleStat.getValue().entrySet()) {
                        ruleStatInfo.add(Updates.inc(info.getKey(), info.getValue()));
                        if (dailyInfo.get(date).get(ruleStat.getKey()).containsKey(info.getKey())) {
                            counter = dailyInfo.get(date).get(ruleStat.getKey()).get(info.getKey());
                            dailyInfo.get(date).get(ruleStat.getKey()).put(info.getKey(), counter + info.getValue());
                        } else {
                            dailyInfo.get(date).get(ruleStat.getKey()).put(info.getKey(), info.getValue());
                        }
                    }
                    logger.info("MongoRulesStatDatabase.saveBatch: route=" + route + ", ts=" + entry.getKey());
                    CompositeEncodingFilter filter =
                        new CompositeEncodingFilter("rule", ruleStat.getKey())
                            .add("time", (int) (long) entry.getKey());
                    if (config.savingMethod() == MongoRulesStatDatabaseConfig.SavingMethod.BULK_REQUESTS) {
                        bulkUpdates.add(new UpdateOneModel<>(filter, Updates.combine(ruleStatInfo), updateOps));
                    } else {
                        //OperationSubscriberWithCallback<UpdateResult> rulesStatSubscriber =
                        singleUpdate(
                            collection,
                            session,
                            ruleStatInfo,
                            filter,
                            updateOps,
                            "Saving of detailed statistics: rule=" + ruleStat.getKey() + ", time=" + entry.getKey(),
                            config,
                            logger,
                            detailedMultiCallback.newCallback());
                    }
                } catch (Exception e) {
                    logger.warning("MongoRulesStatDatabase.saveBatch failed to save detailed statistics: " + e);
                }
            }
        }
        if (config.savingMethod() == MongoRulesStatDatabaseConfig.SavingMethod.BULK_REQUESTS && bulkUpdates.size() > 0)
        {
            BulkWriteSubscriber bulkWriteSubscriber = new BulkWriteSubscriber(
                detailedMultiCallback.newCallback(),
                "Bulk saving of detailed statistics",
                logger);
            ((Mono<BulkWriteResult>) collection.bulkWrite(bulkUpdates)).subscribe(bulkWriteSubscriber);
            logger.warning("MongoRulesStatDatabase.saveDetailedStatistics: save requests count = "
                + bulkUpdates.size());
        }
        detailedMultiCallback.done();
        return dailyInfo;
    }

    private static Void saveDailyStatistics(
        final ClientSession session,
        final MongoClient mongoClient,
        final MongoRulesStatDatabaseConfig config,
        final String tableName,
        final Map<String, Map<String, Map<String, Long>>> data,
        final Route route,
        final int retries,
        final Logger logger,
        final FutureCallback<Void> callback)
    {
        MongoDatabase database = mongoClient.getDatabase(config.dbName());
        MultiFutureCallback<Void> dailyMultiCallback =
            new MultiFutureCallback<>(
                new MongoBatchOperationCallback(
                    session,
                    config,
                    "daily statistics of " + route + " rules",
                    retries,
                    logger,
                    callback));
        MongoCollection<Document> collection = database.getCollection(tableName);
        UpdateOptions updateOps = new UpdateOptions().upsert(true);
        List<UpdateOneModel<Document>> bulkUpdates = new ArrayList<>();
        for (Map.Entry<String, Map<String, Map<String, Long>>> dayStat : data.entrySet()) {
            for (Map.Entry<String, Map<String, Long>> ruleStat : dayStat.getValue().entrySet()) {
                try {
                    List<Bson> ruleStatInfo = new ArrayList<>();
                    for (Map.Entry<String, Long> info : ruleStat.getValue().entrySet()) {
                        ruleStatInfo.add(Updates.inc(info.getKey(), info.getValue()));
                    }
                    logger.info(
                       "MongoRulesStatDatabase.saveBatch: route=" + route + ", date=" + dayStat.getKey());
                    CompositeEncodingFilter filter =
                        new CompositeEncodingFilter("rule", ruleStat.getKey()).add("date", dayStat.getKey());
                    if (config.savingMethod() == MongoRulesStatDatabaseConfig.SavingMethod.BULK_REQUESTS) {
                        bulkUpdates.add(new UpdateOneModel<>(filter, Updates.combine(ruleStatInfo), updateOps));
                    } else {
                        singleUpdate(
                            collection,
                            session,
                            ruleStatInfo,
                            filter,
                            updateOps,
                            "Saving of daily statistics: rule=" + ruleStat.getKey() + ", date=" + dayStat.getKey(),
                            config,
                            logger,
                            dailyMultiCallback.newCallback());
                    }
                } catch (Exception e) {
                    logger.warning("MongoRulesStatDatabase.saveBatch failed to save day statistics: " + e);
                }
            }
        }
        if (config.savingMethod() == MongoRulesStatDatabaseConfig.SavingMethod.BULK_REQUESTS && bulkUpdates.size() > 0)
        {
            BulkWriteSubscriber bulkWriteSubscriber =
                new BulkWriteSubscriber(dailyMultiCallback.newCallback(), "Bulk saving of daily statistics", logger);
            ((Mono<BulkWriteResult>) collection.bulkWrite(bulkUpdates)).subscribe(bulkWriteSubscriber);
            logger.warning("MongoRulesStatDatabase.saveDailyStatistics: save requests count = " + bulkUpdates.size());
        }
        dailyMultiCallback.done();
        return null;
    }

    private static OperationSubscriberWithCallback<UpdateResult> singleUpdate(
        final MongoCollection<Document> collection,
        final ClientSession session,
        final List<Bson> ruleStatInfo,
        final Bson filter,
        final UpdateOptions updateOps,
        final String updateOpName,
        final MongoRulesStatDatabaseConfig config,
        final Logger logger,
        final FutureCallback<Void> callback)
    {
        final long startTime = TimeSource.INSTANCE.currentTimeMillis();
        OperationSubscriberWithCallback<UpdateResult> rulesStatSubscriber =
            new OperationSubscriberWithCallback<>(callback, updateOpName, logger);
        try {
            if (config.savingMethod() == MongoRulesStatDatabaseConfig.SavingMethod.TRANSACTIONS) {
                ((Mono<UpdateResult>) collection.updateOne(session, filter, Updates.combine(ruleStatInfo), updateOps))
                    .subscribe(rulesStatSubscriber);
            } else {
                Mono<UpdateResult> updateResult =
                    ((Mono<UpdateResult>) collection.updateOne(filter, Updates.combine(ruleStatInfo), updateOps));
                updateResult.subscribe(rulesStatSubscriber);
                if (config.savingMethod() == MongoRulesStatDatabaseConfig.SavingMethod.SYNC_REQUESTS) {
                    //logger.info("MongoRulesStatDatabase.singleUpdate: waiting for update result ('" + updateOpName
                    //    + "') " + batchSaveTimeout + " milliseconds");
                    rulesStatSubscriber.await(config.savingOperationTimeout(), TimeUnit.MILLISECONDS);
                    //UpdateResult result = updateResult.block(Duration.ofMillis(batchSaveTimeout));
                    logger.info("MongoRulesStatDatabase.singleUpdate: update's result obtained for update result "
                        + "('" + updateOpName + "', duration: " + (TimeSource.INSTANCE.currentTimeMillis()
                        - startTime) + "): " + rulesStatSubscriber.result());
                }
            }
        } catch (Exception e) {
            logger.log(
                Level.SEVERE,
                "MongoRulesStatDatabase.singleUpdate failed for '" + updateOpName + "', " + "duration: "
                    + (TimeSource.INSTANCE.currentTimeMillis() - startTime),
                e);
            if (rulesStatSubscriber.isDisposed()) {
                rulesStatSubscriber.hookOnError(e);
            } else {
                rulesStatSubscriber.onError(e);
            }
        }
        return rulesStatSubscriber;
    }

    /**
     * Runs given function over given data inside separate MongoDB's transaction.
     * Uses retries as workaround due to MongoDB issue: https://jira.mongodb.org/browse/SERVER-36428.
     * The idea was borrowed from this article:
     * https://www.mongodb.com/community/forums/t/given-transaction-number-does-not-match-any-in-progress-transactions/703/3.
     *
     * @param session MongoDB's client session
     * @param config base config for this batch operation
     * @param data input data which to be saved into DB
     * @param logger logger for logging of diagnostic messages
     * @param execFunc the function to be executed upon data mentioned above and which saves it
     * @param retries number of attempts to execute the transaction
     * @param <D> type data collection object
     * @param <R> type return value for this function
     * @return calculated related data
     * @throws MongoException exception during execution of the transaction
     */
    public static <D, R> R runTransactionWithRetry(
        final ClientSession session,
        final BatchSaverConfig config,
        final D data,
        final Logger logger,
        final Function<D, R> execFunc,
        final int retries)
        throws MongoException
    {
        TransactionOptions trOps = TransactionOptions.merge(
            TransactionOptions.builder()
                .readPreference(ReadPreference.primary())
                .readConcern(ReadConcern.DEFAULT)
                .writeConcern(WriteConcern.ACKNOWLEDGED)
                .maxCommitTime(config.savingOperationTimeout(), TimeUnit.MILLISECONDS)
                .build(),
            session.getOptions().getDefaultTransactionOptions());
        session.startTransaction(trOps);
        try {
            return execFunc.apply(data);
        } catch (MongoException me) {
            if (me.hasErrorLabel(MongoException.TRANSIENT_TRANSACTION_ERROR_LABEL) && retries != 0) {
                logger.warning("runTransactionWithRetry: there are " + (retries - 1) + " attempts left");
                try {
                    Thread.sleep(config.batchSaveRetryTimeout());
                } catch (InterruptedException ie) {
                    abortTansaction(session, config.savingOperationTimeout(), logger, null);
                    throw me;
                }
                return runTransactionWithRetry(session, config, data, logger, execFunc, retries - 1);
            } else {
                abortTansaction(session, config.savingOperationTimeout(), logger, null);
                throw me;
            }
        }
    }

    public static ClientSession startSession(final MongoClient mongoClient, final Logger logger) {
        OperationSubscriber<ClientSession> getSessionOp = new OperationSubscriber<>("MongoDB startSession");
        mongoClient.startSession().subscribe(getSessionOp);
        ClientSession session = null;
        try {
            session = getSessionOp.get().get(0);
        } catch (Exception e) {
            logger.log(Level.SEVERE, "MongoRulesStatDatabase.saveBatch: failed to open session", e);
        }
        return session;
    }

    public static void commitTransaction(
        final ClientSession session,
        final long timeout,
        final int retries,
        final Logger logger,
        final String message)
        throws MongoException
    {
        OperationSubscriber<Void> opSubscriber =
            new OperationSubscriber<>("MongoRulesStatDatabase commitTransaction");
        String msg = message == null ? "" : (" for " + message);
        try {
            session.commitTransaction().subscribe(opSubscriber);
            opSubscriber.await(timeout, TimeUnit.MILLISECONDS);
        } catch (MongoException e) {
            if (e.hasErrorLabel(MongoException.TRANSIENT_TRANSACTION_ERROR_LABEL) && retries != 0) {
                logger.warning("commitTransaction: there are " + (retries - 1) + " attempts left" + msg);
                commitTransaction(session, timeout, retries - 1, logger, message);
            } else {
                abortTansaction(session, timeout, logger, message);
                throw e;
            }
        }
    }

    public static void abortTansaction(
        final ClientSession session,
        final long timeout,
        final Logger logger,
        final String message)
    {
        OperationSubscriber<Void> opSubscriber =
            new OperationSubscriber<>("MongoRulesStatDatabase abortTansaction");
        String msg = message == null ? "" : (" for " + message);
        try (session) {
            session.abortTransaction().subscribe(opSubscriber);
            opSubscriber.await(timeout, TimeUnit.MILLISECONDS);
        } catch (MongoException e) {
            logger.log(
                Level.SEVERE,
                "MongoRulesStatDatabase: aborting of mongo transaction" + msg + " failed due to error",
                e);
        }
    }

    public static String isoDate(final Long unixtime) {
        return (unixtime == null || unixtime < 1 ? ""
            : formatter.print(new DateTime(unixtime * LogRecordContext.MILLIS, TIMEZONE)));
    }

    public static String tableName(final Route route, final EnvironmentType envType) {
        String strRoute = route.lowerName();
        String tableName = "Rules_";
        if (envType == EnvironmentType.PRODUCTION) {
            tableName += strRoute.substring(0, 1).toUpperCase(Locale.ROOT) + strRoute.substring(1);
        } else {
            tableName += strRoute + "_tmp";
        }
        return tableName;
    }

    private static class MongoBatchOperationCallback implements FutureCallback<List<Void>> {
        private final ClientSession session;
        private final MongoRulesStatDatabaseConfig config;
        private final String message;
        private final Logger logger;
        private final FutureCallback<Void> callback;
        private final int retries;

        MongoBatchOperationCallback(
            final ClientSession session,
            final MongoRulesStatDatabaseConfig config,
            final String message,
            final int retries,
            final Logger logger,
            final FutureCallback<Void> callback)
        {
            this.session = session;
            this.config = config;
            this.message = message;
            this.retries = retries;
            this.logger = logger;
            this.callback = callback;
        }

        @Override
        public void cancelled() {
            if (config.savingMethod() == MongoRulesStatDatabaseConfig.SavingMethod.TRANSACTIONS) {
                logger.warning("MongoRulesStatDatabase.MongoBatchOperationCallback: Mongo operation for " + message
                    + " cancelled (" + (retries - 1) + " attempts left)");
                abortTansaction(session, config.savingOperationTimeout(), logger, message);
            } else {
                logger.warning("MongoRulesStatDatabase.MongoBatchOperationCallback: Mongo operation for " + message
                    + " cancelled");
            }
            callback.cancelled();
        }

        @Override
        public void completed(final List<Void> ignored) {
            logger.info("MongoRulesStatDatabase.MongoBatchOperationCallback: Mongo operation for " + message
                + " completed successfully");
            try {
                if (config.savingMethod() == MongoRulesStatDatabaseConfig.SavingMethod.TRANSACTIONS) {
                    commitTransaction(session, config.savingOperationTimeout(), retries, logger, message);
                }
                session.close();
                callback.completed(null);
            } catch (MongoException e) {
                logger.log(Level.SEVERE, "MongoRulesStatDatabase.MongoBatchOperationCallback: committing of mongo "
                    + "transaction for " + message + " failed due to error", e);
                if (session.hasActiveTransaction()) {
                    session.close();
                }
                callback.failed(e);
            }
        }

        @Override
        public void failed(final Exception e) {
            if (config.savingMethod() == MongoRulesStatDatabaseConfig.SavingMethod.TRANSACTIONS) {
                logger.log(
                    Level.SEVERE,
                    "MongoRulesStatDatabase.MongoBatchOperationCallback: Mongo operation for " + message
                        + " failed (" + (retries - 1) + " attempts left)",
                    e);
                abortTansaction(session, config.savingOperationTimeout(), logger, message);
            } else {
                logger.log(
                    Level.SEVERE,
                    "MongoRulesStatDatabase.MongoBatchOperationCallback: Mongo operation for " + message + " failed",
                    e);
            }
            callback.failed(e);
        }
    }

    @SuppressWarnings("unused")
    private static class MongoAllOperationsCallback implements FutureCallback<List<Void>> {
        private final MongoRulesStatDatabase rulesStatDatabase;
        private final SpDaemonRulesStatBatch rulesStatBatch;
        private final long startTime;
        private final Logger logger;

        MongoAllOperationsCallback(
            final MongoRulesStatDatabase rulesStatDatabase,
            final SpDaemonRulesStatBatch rulesStatBatch,
            final long startTime,
            final Logger logger)
        {
            this.rulesStatDatabase = rulesStatDatabase;
            this.rulesStatBatch = rulesStatBatch;
            this.startTime = startTime;
            this.logger = logger;
        }

        @Override
        public void cancelled() {
            logger.warning("MongoRulesStatDatabase.MongoAllOperationsCallback: operations cancelled");
            rulesStatDatabase.stat(MongoResultType.OTHER, TimeSource.INSTANCE.currentTimeMillis() - startTime);
            rulesStatDatabase.decrementBatchesCount();
            rulesStatBatch.setState(BatchState.FAILED_TO_SAVE);
        }

        @Override
        public void completed(final List<Void> ignored) {
            logger.info("MongoRulesStatDatabase.MongoAllOperationsCallback: operations completed successfully");
            rulesStatDatabase.stat(MongoResultType.SUCCESS, TimeSource.INSTANCE.currentTimeMillis() - startTime);
            rulesStatDatabase.decrementBatchesCount();
            rulesStatBatch.setState(BatchState.SAVED);
        }

        @Override
        public void failed(final Exception e) {
            logger.log(Level.SEVERE, "MongoRulesStatDatabase.MongoAllOperationsCallback: operations failed", e);
            rulesStatDatabase.stat(e, TimeSource.INSTANCE.currentTimeMillis() - startTime);
            rulesStatDatabase.decrementBatchesCount();
            rulesStatBatch.setState(BatchState.FAILED_TO_SAVE);
        }
    }

    private static class CompositeEncodingFilter implements Bson {
        private final Map<String, Object> fields;

        public CompositeEncodingFilter() {
            fields = new HashMap<>();
        }

        public CompositeEncodingFilter(final String fieldName, @Nullable final Object value) {
            this();
            add(fieldName, value);
        }

        public CompositeEncodingFilter add(final String fieldName, @Nullable final Object value) {
            fields.put(notNull("fieldName", fieldName), value);
            return this;
        }

        public int size() {
            return fields.size();
        }

        public boolean contains(final String fieldName) {
            return fields.containsKey(fieldName);
        }

        public Object get(final String fieldName) {
            return fields.get(fieldName);
        }

        @Override
        public <TDocument> BsonDocument toBsonDocument(
            final Class<TDocument> documentClass,
            final CodecRegistry codecRegistry)
        {
            BsonDocumentWriter writer = new BsonDocumentWriter(new BsonDocument());

            writer.writeStartDocument();
            for (Map.Entry<String, Object> entry : fields.entrySet()) {
                writer.writeName(entry.getKey());
                encodeValue(writer, entry.getValue(), codecRegistry);
            }
            writer.writeEndDocument();

            return writer.getDocument();
        }

        @Override
        @SuppressWarnings("EqualsGetClass")
        public boolean equals(final Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }
            CompositeEncodingFilter that = (CompositeEncodingFilter) o;
            if (size() != that.size()) {
                return false;
            }
            for (Map.Entry<String, Object> entry : fields.entrySet()) {
                if (!that.contains(entry.getKey()) || (entry.getValue() == null && that.get(entry.getKey()) == null)
                        || (entry.getValue() != null && !entry.getValue().equals(that.get(entry.getKey())))) {
                    return false;
                }
            }
            return true;
        }

        @Override
        public int hashCode() {
            int result = 0;
            for (Map.Entry<String, Object> entry : fields.entrySet()) {
                result += 31 * entry.getKey().hashCode() + (entry.getValue() != null ? entry.getValue().hashCode() : 0);
            }
            return result;
        }

        @Override
        public String toString() {
            StringBuilder sb = new StringBuilder("Filter{");
            for (Map.Entry<String, Object> entry : fields.entrySet()) {
                sb.append('\'').append(entry.getKey()).append("':'").append(entry.getValue()).append('\'');
            }
            sb.append('}');
            return sb.toString();
        }
    }

    @SuppressWarnings("unchecked")
    private static <TItem> void encodeValue(
        final BsonDocumentWriter writer,
        final TItem value,
        final CodecRegistry codecRegistry)
    {
        if (value == null) {
            writer.writeNull();
        } else if (value instanceof Bson) {
            codecRegistry.get(BsonDocument.class).encode(
                writer,
                ((Bson) value).toBsonDocument(BsonDocument.class, codecRegistry),
                EncoderContext.builder().build());
        } else {
            ((Encoder<TItem>) codecRegistry.get(value.getClass()))
                .encode(writer, value, EncoderContext.builder().build());
        }
    }
}
