package ru.yandex.mail.so.logger;

import java.util.ArrayList;
import java.util.List;
import java.util.Locale;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;

import javax.annotation.Nonnull;

import com.mongodb.MongoException;
import com.mongodb.MongoInterruptedException;
import com.mongodb.MongoTimeoutException;
import com.mongodb.lang.NonNull;
import com.mongodb.lang.Nullable;
import org.reactivestreams.Subscription;
import reactor.core.publisher.BaseSubscriber;

public class OperationSubscriber<T> extends BaseSubscriber<T> {
    protected final List<T> result;
    protected final List<RuntimeException> errors;
    protected final CountDownLatch latch;
    protected String opName;

    public OperationSubscriber() {
        this("noname");

    }

    public OperationSubscriber(final String opName) {
        this.result = new ArrayList<>();
        this.errors = new ArrayList<>();
        this.latch = new CountDownLatch(1);
        this.opName = opName;
    }

    public void setName(final String opName) {
        this.opName = opName;
    }

    @Override
    public boolean isDisposed() {
        return super.isDisposed() || latch.getCount() == 0;
    }

    @Override
    public void hookOnSubscribe(@NonNull final Subscription s) {
        super.hookOnSubscribe(s);
        //request(1);
    }

    @Override
    public void hookOnNext(@Nullable final T t) {
        result.add(t);
    }

    @Override
    public void hookOnError(@NonNull final Throwable t) {
        RuntimeException e =
            t instanceof RuntimeException ? (RuntimeException) t : new RuntimeException("Unexpected exception", t);
        errors.add(e);
        latch.countDown();
    }

    @Override
    protected void hookOnCancel() {
        latch.countDown();
    }

    @Override
    public void hookOnComplete() {
        latch.countDown();
    }

    /**
     * Get received elements
     *
     * @return the list of received elements
     */
    public List<T> result() {
        return result;
    }

    /**
     * Get error from subscription
     *
     * @return the error, which may be null
     */
    public RuntimeException error() {
        if (errors.size() > 0) {
            return errors.get(0);
        }
        return null;
    }

    /**
     * Get received elements.
     *
     * @return the list of receive elements
     */
    public List<T> get() throws MongoException {
        return await().result();
    }

    /**
     * Get received elements.
     *
     * @param timeout how long to wait
     * @param unit the time unit
     * @return the list of receive elements
     */
    public List<T> get(final long timeout, @Nonnull final TimeUnit unit) throws MongoException {
        return await(timeout, unit).result();
    }

    /**
     * Await completion or error
     *
     * @return this
     */
    public OperationSubscriber<T> await() throws MongoException {
        //request(Integer.MAX_VALUE);
        try {
            latch.await();
        } catch (InterruptedException e) {
            throw new MongoInterruptedException("Operation '" + opName + "' interrupted waiting for observation", e);
        }
        if (!errors.isEmpty()) {
            throw errors.get(0);
        }
        return this;
    }

    /**
     * Await completion or error
     *
     * @param timeout how long to wait
     * @param unit the time unit
     * @return this
     */
    public OperationSubscriber<T> await(final long timeout, @Nonnull final TimeUnit unit) throws MongoException {
        //request(Integer.MAX_VALUE);
        try {
            if (!latch.await(timeout, unit)) {
                throw new MongoTimeoutException("Operation '" + opName + "': publisher onComplete timed out (timeout = "
                    + timeout + ' ' + unit.name().toLowerCase(Locale.ROOT) + ")");
            }
        } catch (InterruptedException e) {
            throw new MongoInterruptedException("Operation '" + opName + "' interrupted waiting for observation", e);
        }
        if (!errors.isEmpty()) {
            throw errors.get(0);
        }
        return this;
    }
}
