package ru.yandex.mail.so.logger;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.dom.TypesafeValueContentHandler;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.JsonType;
import ru.yandex.util.timesource.TimeSource;

public class SpDaemonRulesStatBatch implements RulesStatBatch<BasicRoutedLogRecordProducer> {
    public static final int STEP_DETAILED = 600;

    protected static final String MESS = "mess";
    protected static final String R_SP = "r_sp";
    protected static final String R_DL = "r_dl";
    protected static final String R_NL = "r_nl";

    private static final Pattern MESS_RE =
        Pattern.compile("[A-Za-z]+\\s+\\d+\\s+\\d\\d:\\d\\d:\\d\\d(?:\\.\\d+)?\\s+-\\s+(\\d+)");
    private static final Pattern R_RE = Pattern.compile(",\\s*R(\\d+),\\s*$");
    private static final Pattern PERSONAL_CORRECT_RE = Pattern.compile("\\b(PERSONAL_CORRECT)\\b");
    private static final String ERROR_MSG = "SpDaemonRulesStatBatch: Bad format of delivery-log: ";

    /**
     * Batch object is map: route -> (ts -> (rule -> (param -> counters)))
     */
    private final Map<Route, Map<Long, Map<String, Map<String, Long>>>> batch;
    private volatile BatchState state = BatchState.INITED;
    private long batchSize;
    private long startTime;

    public SpDaemonRulesStatBatch() {
        batchSize = 0L;
        batch = new HashMap<>();
        startTime = TimeSource.INSTANCE.currentTimeMillis();
    }

    @Override
    public synchronized void reset() {
        batch.clear();
        state = BatchState.INITED;
        startTime = TimeSource.INSTANCE.currentTimeMillis();
        notifyAll();
    }

    /**
     * Parse rules statistics from the given log record
     *
     * @param logRecord given log record
     * @param outRulesStat output data
     * @throws JsonException exception, which may be thrown while parsing of the log record
     */
    @SuppressWarnings("StringSplitter")
    public static void parseRulesStatistics(
        final String logRecord,
        final Route route,
        final Map<Long, Map<String, Map<String, Long>>> outRulesStat)
        throws JsonException
    {
        JsonList jsonList = TypesafeValueContentHandler.parse(logRecord).asListOrNull();
        if (jsonList != null) {
            Matcher m;
            Map<String, String> headers = new HashMap<>();
            List<String> absentHeaders = new ArrayList<>(List.of(MESS, R_SP, R_NL));
            if (route == Route.IN || route == Route.CORP) {
                absentHeaders.add(R_DL);
            }
            for (JsonObject item : jsonList) {
                JsonList headerItem = item.asList();
                if (headerItem.size() < 2) {
                    throw new JsonException(ERROR_MSG + "wrong header=" + JsonType.NORMAL.toString(headerItem));
                }
                String key = headerItem.get(0).asString();
                if (absentHeaders.contains(key)) {
                    headers.put(key, headerItem.get(1).asString());
                    absentHeaders.remove(key);
                }
            }
            if (absentHeaders.size() > 0) {
                throw new JsonException(ERROR_MSG + "absent headers=" + absentHeaders);
            }
            m = MESS_RE.matcher(headers.get(MESS));
            long ts = (m.find() ? Long.parseUnsignedLong(m.group(1))
                    : (TimeSource.INSTANCE.currentTimeMillis() / LogRecordContext.MILLIS)) / STEP_DETAILED
                    * STEP_DETAILED;
            int rc;
            m = R_RE.matcher(headers.get(R_NL));
            if (m.find()) {
                rc = Integer.parseInt(m.group(1));
                if (PERSONAL_CORRECT_RE.matcher(headers.get(R_SP)).find()
                        || PERSONAL_CORRECT_RE.matcher(headers.get(R_NL)).find())
                {
                    if (rc == 4) {
                        rc = 127;
                    } else if (rc == 1) {
                        rc = 8;
                    }
                }
            } else {
                throw new JsonException(ERROR_MSG + "absent pseudo-rule 'R' in 'r_nl' header " + headers.get(R_NL));
            }
            String ruleR = "R" + rc;
            long counter;
            String rule;
            List<String> parsedHeaders = new ArrayList<>(List.of(R_SP, R_NL));
            if (route == Route.IN || route == Route.CORP) {
                absentHeaders.add(R_DL);
            }
            for (String specHeader : parsedHeaders) {
                for (String ruleInfo : headers.get(specHeader).split("[,;]\\s*")) {
                    if (!ruleInfo.isEmpty()) {
                        if (ruleInfo.contains(" ")) {
                            rule = ruleInfo.substring(0, ruleInfo.indexOf(" "));
                        } else {
                            rule = ruleInfo;
                        }
                        Map<String, Long> counters = outRulesStat.computeIfAbsent(ts, x -> new HashMap<>())
                            .computeIfAbsent(rule, y -> new HashMap<>());
                        if (counters.containsKey(ruleR)) {
                            counter = counters.get(ruleR);
                            counters.put(ruleR, counter + 1L);
                        } else {
                            counters.put(ruleR, 1L);
                        }
                    }
                }
            }
        }
    }

    public static void parseRulesStatistics(
        final BasicRoutedLogRecordProducer logRecordProducer,
        final Map<Long, Map<String, Map<String, Long>>> outRulesStat)
        throws JsonException
    {
        parseRulesStatistics(logRecordProducer.logRecord(), logRecordProducer.route(), outRulesStat);
    }

    @Override
    public synchronized void add(final BasicRoutedLogRecordProducer logRecordProducer) throws JsonException {
        if (logRecordProducer.logRecord() != null && !logRecordProducer.logRecord().isEmpty()
                && logRecordProducer.route() != null)
        {
            try {
                batchSize++;
                parseRulesStatistics(
                    logRecordProducer,
                    batch.computeIfAbsent(logRecordProducer.route(), x -> new HashMap<>()));
            } catch (Exception e) {
                throw new JsonException(ERROR_MSG + e);
            }
        }
    }

    public Map<Long, Map<String, Map<String, Long>>> routeData(final Route route) {
        return batch.get(route);
    }

    public Set<Route> routes() {
        return batch.keySet();
    }

    @SuppressWarnings("unused")
    public long startTime() {
        return startTime;
    }

    @Override
    public long lifeTime() {
        return TimeSource.INSTANCE.currentTimeMillis() - startTime;
    }

    @Override
    public long count() {
        return batchSize;
    }

    @Override
    public synchronized BatchState state() {
        return state;
    }

    @Override
    public synchronized void setState(final BatchState state) {
        this.state = state;
        notifyAll();
    }

    @Override
    public boolean isEmpty() {
        return batch.isEmpty();
    }

    @Override
    public LogRecordsHandlerType type() {
        return LogRecordsHandlerType.SP_DAEMON;
    }
}
