package ru.yandex.mail.so.logger;

import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.apache.http.HttpHost;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.mime.FormBodyPartBuilder;
import org.apache.http.entity.mime.MultipartEntityBuilder;
import org.apache.http.entity.mime.content.StringBody;

import ru.yandex.collection.Pattern;
import ru.yandex.function.GenericBiConsumer;
import ru.yandex.http.config.ImmutableHttpHostConfig;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.ErrorSuppressingFutureCallback;
import ru.yandex.http.util.MultiFutureCallback;
import ru.yandex.http.util.RequestErrorType;
import ru.yandex.http.util.YandexHeaders;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.EmptyAsyncConsumerFactory;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.http.util.request.RequestHandlerMapper;
import ru.yandex.http.util.request.RequestInfo;
import ru.yandex.http.util.request.RequestPatternParser;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumerFactory;
import ru.yandex.json.dom.JsonNull;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.logger.HandlersManager;
import ru.yandex.mail.so.logger.config.ImmutableLogRecordsHandlerConfig;
import ru.yandex.mail.so.logger.config.ImmutableSpLoggerConfig;
import ru.yandex.mail.so.logger.config.RulesStatDatabasesConfig;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.searchmap.SearchMap;
import ru.yandex.parser.searchmap.SearchMapHost;
import ru.yandex.parser.searchmap.SearchMapShard;
import ru.yandex.parser.uri.CgiParams;
import ru.yandex.search.proxy.universal.UniversalSearchProxy;

public class SpLogger
    extends UniversalSearchProxy<ImmutableSpLoggerConfig>
    implements RulesStatDatabasesOperator<BasicRoutedLogRecordProducer>
{
    public static final String CHECK = "/check";
    public static final String SEARCH = "/search";
    public static final String UNPREFIXED_PARALLEL = "/unprefixed-parallel";
    public static final String DELETE_FROM_MDS = "/delete-from-mds";
    public static final String INDEX_SEARCH = "/index-search";
    public static final ContentType TEXT_PLAIN = ContentType.TEXT_PLAIN.withCharset(StandardCharsets.UTF_8);

    private static final String SERVICE = "service";
    private static final StringBody EMPTY_BODY = new StringBody("", ContentType.TEXT_PLAIN);

    private final Map<String, LogStorage<LogRecordContext>> logStorages;
    private final Map<String, AuxiliaryStorage> auxiliaryStorages;
    private final Map<String, RulesStatDatabase<BasicRoutedLogRecordProducer>> rulesStatDatabases;
    private final AsyncClient producerAsyncClient;
    private final Logger spareDeliveryLogger;
    private final List<MdsLogStorage> mdsLogStorages;

    public SpLogger(final ImmutableSpLoggerConfig config)
        throws ConfigException, IOException, LogStorageException
    {
        super(config);
        System.setProperty("java.version", "16");   // stupid patch for successful running of mongodb java driver
        {
            HandlersManager handlersManager = new HandlersManager();
            spareDeliveryLogger = config.spareDeliveryLog().build(handlersManager);
            registerLoggerForLogrotate(spareDeliveryLogger);
        }
        logStorages = config.logStoragesConfig().prepareStorages(this);
        auxiliaryStorages = config.auxiliaryStoragesConfig().prepareStorages(this);
        rulesStatDatabases = prepareClients(config, this);
        config.rulesStatDatabasesConfig().registerComponents(this);
        config.logRecordsHandlersConfig().logRecordsHandlers().traverse(new LogRecordsHandlerVisitor());
        ImmutableHttpHostConfig producerAsyncClientConfig = config.producerAsyncClientConfig();
        if (producerAsyncClientConfig == null) {
            producerAsyncClient = null;
        } else {
            producerAsyncClient = client("AsyncProducer", producerAsyncClientConfig);
        }
        register(new Pattern<>(UNPREFIXED_PARALLEL + '/', true), new UnprefixedParallelRequestHandler(this));
        register(
            new Pattern<>(INDEX_SEARCH + '/', true),
            new SearchBackendProxyHandler(config.indexSearchStaterConfig(), this));
        mdsLogStorages = new ArrayList<>();
        for (LogStorage<LogRecordContext> logStorage : logStorages.values()) {
            if (logStorage.type() == LogStorageType.MDS) {
                if (mdsLogStorages.size() == 0) {
                    logger().info("Registering of handler <DELETE_FROM_MDS>: " + DELETE_FROM_MDS);
                    register(new Pattern<>(DELETE_FROM_MDS, true), new DeleteFromMdsHandler(this));
                }
                mdsLogStorages.add((MdsLogStorage) logStorage);
            }
        }
    }

    @Override
    public RulesStatDatabasesConfig<BasicRoutedLogRecordProducer> rulesStatDatabasesConfig() {
        return config.rulesStatDatabasesConfig();
    }

    @Override
    public Map<String, RulesStatDatabase<BasicRoutedLogRecordProducer>> rulesStatDatabases() {
        return rulesStatDatabases;
    }

    public Map<String, AuxiliaryStorage> auxiliaryStorages() {
        return auxiliaryStorages;
    }

    public LogStorage<LogRecordContext> logStorage(final String storageName) {
        return logStorages.get(storageName);
    }

    @SuppressWarnings("unused")
    public List<MdsLogStorage> mdsLogStorages() {
        return mdsLogStorages;
    }

    public AsyncClient producerAsyncClient() {
        return producerAsyncClient;
    }

    @Override
    public void start() throws IOException {
        for (LogStorage<LogRecordContext> logStorage : logStorages.values()) {
            logStorage.start();
            closeChain.add(logStorage);
        }
        for (AuxiliaryStorage auxiliaryStorage : auxiliaryStorages.values()) {
            auxiliaryStorage.start();
            closeChain.add(auxiliaryStorage);
        }
        for (RulesStatDatabase<BasicRoutedLogRecordProducer> rulesStatDatabase : rulesStatDatabases.values()) {
            rulesStatDatabase.start();
            closeChain.add(rulesStatDatabase);
        }
        super.start();
    }

    @Override
    public void close() throws IOException {
        super.close();
    }

    public Logger spareDeliveryLogger() {
        return spareDeliveryLogger;
    }

    @SuppressWarnings("FutureReturnValueIgnored")
    public void unprefixedParallelRequest(
        final ProxySession session,
        final String query,
        final String service,
        final FutureCallback<JsonObject> callback)
    {
        CgiParams params = session.params();
        int offset = 0;
        int length = LogRecordsContext.DEFAULT_SEARCH_DOCS_LIMIT;
        if (!session.params().containsKey(SearchParam.QUEUEID.paramName())) {
            try {
                offset = session.params().getInt(SearchParam.SKIP.paramName(), 0);
                length = params.getInt(SearchParam.LIMIT.paramName(), LogRecordsContext.DEFAULT_SEARCH_DOCS_LIMIT);
            } catch (BadRequestException e) {
                session.logger().log(Level.SEVERE, "unprefixedParallelRequest failed: " + e, e);
                callback.completed(null);
            }
        }
        MultiFutureCallback<JsonObject> multiCallback =
            new MultiFutureCallback<>(new JsonListFilterFutureCallback(session.logger(), offset, length, callback));
        final AsyncClient client = searchClient.adjust(session.context());
        for (int i = 0; i < config().searchBackendShardsNumber(); i++) {
            HashSet<HttpHost> hostsSet = new HashSet<>();
            //searchMap().searchHosts();
            SearchMapShard shard = searchMap().shardGet(i, service);
            for (SearchMapHost searchMapHost : shard) {
                if (searchMapHost.searchHost() != null) {
                    hostsSet.add(searchMapHost.searchHost());
                }
            }
            ArrayList<HttpHost> hosts = new ArrayList<>(hostsSet);
            client.execute(
                hosts,
                new BasicAsyncRequestProducerGenerator(query + "&offset=0&prefix=" + i),
                JsonAsyncTypesafeDomConsumerFactory.OK,
                session.listener().createContextGeneratorFor(client),
                new ErrorSuppressingFutureCallback<>(
                    multiCallback.newCallback(),
                    x -> RequestErrorType.HTTP,
                    JsonNull.INSTANCE));
            logger.info("unprefixedParallelRequest: " + query + "&offset=0&prefix=" + i);
        }
        multiCallback.done();
    }

    public void unprefixedParallelRequest(
        final ProxySession session,
        final String query,
        final FutureCallback<JsonObject> callback)
    {
        final String query2 =
            query.startsWith(UNPREFIXED_PARALLEL) ? query.substring(UNPREFIXED_PARALLEL.length()) : query;
        unprefixedParallelRequest(session, query2, config().indexingQueueName(), callback);
    }

    @SuppressWarnings("FutureReturnValueIgnored")
    public void deleteFromMdsRequest(final ProxySession session, final FutureCallback<Void> callback) {
        BasicAsyncRequestProducerGenerator producerGenerator;
        MdsLogStorage mdsLogStorage = mdsLogStorages.get(0);
        String mdsNamespace =
            session.params().getString("mds_namespace", mdsLogStorage.config().mdsNamespace());
        String mdsDeletesQueueName =
            session.params().getString("mds_namespace", mdsLogStorage.config().mdsDeletesQueueName());
        for (MdsLogStorage logStorage : mdsLogStorages) {
            if (!logStorage.config().mdsNamespace().equals(mdsLogStorage.config().mdsNamespace())) {
                mdsLogStorage = logStorage;
            }
        }
        int shardId;
        List<String> stids = new ArrayList<>();
        try {
            shardId = session.params().getInt("shard", 0);
            for (String stidsInfo : session.params().getAll("stids")) {
                stids.addAll(Arrays.asList(stidsInfo.split(",")));
            }
        } catch (Exception e) {
            session.logger().log(Level.SEVERE, "deleteFromMdsRequest failed to parse input CGI-parameters", e);
            callback.failed(e);
            return;
        }
        String uri = "/delete-" + mdsNamespace + '/';
        StringBuilder mainUri = new StringBuilder("/delete?");
        mainUri.append(SERVICE).append('=').append(mdsDeletesQueueName);
        if (stids.size() == 1) {
            producerGenerator = new BasicAsyncRequestProducerGenerator(uri + stids.get(0), "", ContentType.TEXT_PLAIN);
            producerGenerator.addHeader(YandexHeaders.ZOO_SHARD_ID, Long.toString(shardId % SearchMap.SHARDS_COUNT));
            //producerGenerator.addHeader(
            //    YandexHeaders.ZOO_HASH,
            //    stids.get(0).replaceAll("[/.:]", "").toLowerCase(Locale.ROOT));
            producerGenerator.addHeader(YandexHeaders.X_YA_SERVICE_TICKET, mdsLogStorage.mdsTvm2Ticket());
            producerGenerator.addHeader(YandexHeaders.CHECK_DUPLICATE, "true");
        } else {
            MultipartEntityBuilder builder = MultipartEntityBuilder.create();
            builder.setMimeSubtype("mixed");
            for (String stid : stids) {
                builder.addPart(
                    FormBodyPartBuilder
                        .create()
                        .addField(YandexHeaders.ZOO_SHARD_ID, Long.toString(shardId % SearchMap.SHARDS_COUNT))
                        //.addField(YandexHeaders.ZOO_HASH, stid.replaceAll("[/.:]", "").toLowerCase(Locale.ROOT))
                        .addField(YandexHeaders.CHECK_DUPLICATE, "true")
                        .addField(YandexHeaders.X_YA_SERVICE_TICKET, mdsLogStorage.mdsTvm2Ticket())
                        .addField(SERVICE, mdsDeletesQueueName)
                        .addField(YandexHeaders.URI, uri + stid)
                        .setBody(EMPTY_BODY)
                        .setName("empty_body")
                        .build());
            }
            try {
                producerGenerator = new BasicAsyncRequestProducerGenerator(mainUri.toString(), builder.build());
            } catch (IOException e) {
                logger.log(Level.SEVERE, "deleteFromMdsRequest failed to create BasicAsyncRequestProducerGenerator", e);
                callback.failed(e);
                return;
            }
        }
        producerGenerator.addHeader(YandexHeaders.SERVICE, mdsDeletesQueueName);
        AsyncClient client = producerAsyncClient();
        HttpHost producerHost = config().producerClientConfig().host();
        client.execute(producerHost, producerGenerator, EmptyAsyncConsumerFactory.OK, callback);
    }

    private class LogRecordsHandlerVisitor
        implements GenericBiConsumer<Pattern<RequestInfo>, ImmutableLogRecordsHandlerConfig, ConfigException>
    {
        @Override
        public void accept(final Pattern<RequestInfo> pattern, final ImmutableLogRecordsHandlerConfig config)
            throws ConfigException
        {
            if (config.type() == null) {
                return;
            }
            logger().info("Registering of log records handler <" + config.type() + ">: " + pattern);
            AbstractLogRecordsHandler handler = config.type().createHandler(config, SpLogger.this, pattern.toString());
            register(pattern, handler);
            if (config.type() != LogRecordsHandlerType.NULL) {
                registerStater(handler);
                if (config.route() != null && SpLogger.this.config.route() == config.route()) {
                    logger().info("Registering of log records handler <" + config.type() + ">: " + CHECK);
                    register(new Pattern<>(CHECK, true), handler, RequestHandlerMapper.POST);
                }
                if (config.route() != null) {
                    logger().info("Registering of log records handler <" + config.type() + ">: " + SEARCH
                        + "*{arg_" + SearchParam.ROUTE.paramName() + ":" + config.route().lowerName() + "}");
                    register(
                        RequestPatternParser.INSTANCE.apply(SEARCH + "*{arg_" + SearchParam.ROUTE.paramName() + ":"
                            + config.route().lowerName() + "}"),
                        handler,
                        RequestHandlerMapper.GET);
                } else if (config.route() != null) {
                    logger().info("Registering of log records handler <" + config.type() + ">: " + SEARCH);
                    register(new Pattern<>(SEARCH, true), handler, RequestHandlerMapper.GET);
                }
            }
        }
    }
}
