package ru.yandex.mail.so.logger.config;

import java.util.ArrayList;
import java.util.List;
import java.util.Locale;

import ru.yandex.function.StringProcessor;
import ru.yandex.mail.so.logger.LogRecordsHandlerType;
import ru.yandex.mail.so.logger.LogStorage;
import ru.yandex.mail.so.logger.Route;
import ru.yandex.parser.config.ConfigBuilder;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.config.IniConfig;
import ru.yandex.parser.string.CollectionParser;
import ru.yandex.parser.string.DurationParser;
import ru.yandex.parser.string.EnumParser;
import ru.yandex.stater.AbstractStaterConfigBuilder;
import ru.yandex.stater.StaterConfigDefaults;
import ru.yandex.util.string.StringUtils;

public abstract class AbstractLogRecordsHandlerConfigBuilder<T extends AbstractLogRecordsHandlerConfigBuilder<T>>
    extends AbstractStaterConfigBuilder<T>
    implements ConfigBuilder<T>, LogRecordsHandlerConfig
{
    private static final StringProcessor<List<String>, RuntimeException> commaSeparatedParser =
        new StringProcessor<>(new CollectionParser<>(x -> x, ArrayList::new));

    private LogRecordsHandlerType logRecordsHandlerType;
    private String logStorage;
    private long storeTTL;
    private String rulesStatDatabase;
    private String decompression;
    private Route route;
    private List<String> auxiliaryStorages;

    protected AbstractLogRecordsHandlerConfigBuilder(final LogRecordsHandlerConfig config) {
        super(config);
        logRecordsHandlerType = config.type();
        logStorage = config.logStorage();
        storeTTL = config.storeTTL();
        rulesStatDatabase = config.rulesStatDatabase();
        decompression = config.decompression();
        route = config.route();
        auxiliaryStorages = new ArrayList<>(config.auxiliaryStorages());
    }

    protected AbstractLogRecordsHandlerConfigBuilder(final IniConfig config, final LogRecordsHandlerConfig defaults)
        throws ConfigException
    {
        super(config.section("stat"), defaults == null ? StaterConfigDefaults.INSTANCE : defaults);
        logRecordsHandlerType = config.get("type", null, new EnumParser<>(LogRecordsHandlerType.class));
        if (logRecordsHandlerType != null) {
            String typePrefix = logRecordsHandlerType.name().toLowerCase(Locale.ROOT).replace('_', '-');
            if (!typePrefix.equals(prefix())) {
                prefix(StringUtils.concat(typePrefix, '-', prefix()));
            }
        }
        logStorage = config.getOrNull("storage-type");
        if (logStorage == null && (defaults == null || defaults.logStorage() == null)) {
            if (logRecordsHandlerType != null && logRecordsHandlerType == LogRecordsHandlerType.NULL) {
                logStorage = LogStorage.NULL;
            } else if (logRecordsHandlerType != null) {
                throw new ConfigException("log storage must be specified for log records handler type <"
                    + logRecordsHandlerType + ">");
            }
        }
        long defaultTTL = defaults == null ? 0L : defaults.storeTTL();
        String defaultRulesStatDatabase = defaults == null ? null : defaults.rulesStatDatabase();
        String defaultDecompression = defaults == null ? null : defaults.decompression();
        storeTTL = config.get("store-ttl", defaultTTL, DurationParser.POSITIVE_LONG);
        rulesStatDatabase = config.getString("rules-stat-db", defaultRulesStatDatabase);
        decompression = config.getString("decompression", defaultDecompression);
        route = config.get("route", defaults == null ? null : defaults.route(), new EnumParser<>(Route.class));
        String auxStorages = config.getOrNull("auxiliary-storage");
        auxiliaryStorages = auxStorages == null ? List.of() : commaSeparatedParser.process(auxStorages);
    }

    @Override
    public LogRecordsHandlerType type() {
        return logRecordsHandlerType;
    }

    public T type(final LogRecordsHandlerType logRecordsHandlerType) {
        this.logRecordsHandlerType = logRecordsHandlerType;
        return self();
    }

    public T type(final String logRecordsHandlerTypeStr) {
        EnumParser<LogRecordsHandlerType> parser = new EnumParser<>(LogRecordsHandlerType.class);
        try {
            this.logRecordsHandlerType = parser.apply(logRecordsHandlerTypeStr);
        } catch (Exception e) {
            this.logRecordsHandlerType = LogRecordsHandlerConfigDefaults.INSTANCE.type();
        }
        return self();
    }

    @Override
    public String logStorage() {
        return logStorage;
    }

    public T logStorage(final String logStorage) {
        this.logStorage = logStorage;
        return self();
    }

    @Override
    public long storeTTL() {
        return storeTTL;
    }

    public T storeTTL(final long storeTTL) {
        this.storeTTL = storeTTL;
        return self();
    }

    public T storeTTL(final String storeTtlStr) {
        try {
            this.storeTTL = DurationParser.POSITIVE_LONG.apply(storeTtlStr);
        } catch (Exception e) {
            this.storeTTL = LogRecordsHandlerConfigDefaults.INSTANCE.storeTTL();
        }
        return self();
    }

    @Override
    public String rulesStatDatabase() {
        return rulesStatDatabase;
    }

    public T rulesStatDatabase(final String rulesStatDatabase) {
        this.rulesStatDatabase = rulesStatDatabase;
        return self();
    }

    @Override
    public String decompression() {
        return decompression;
    }

    public T decompression(final String decompression) {
        this.decompression = decompression;
        return self();
    }

    @Override
    public Route route() {
        return route;
    }

    public T route(final Route route) {
        this.route = route;
        return self();
    }

    @Override
    public List<String> auxiliaryStorages() {
        return auxiliaryStorages;
    }

    public T auxiliaryStorages(final List<String> auxiliaryStorages) {
        this.auxiliaryStorages = auxiliaryStorages;
        return self();
    }
}
