package ru.yandex.mail.so.logger.config;

import java.util.ArrayList;
import java.util.List;

import ru.yandex.http.config.ClientHttpsConfig;
import ru.yandex.http.config.ClientHttpsConfigBuilder;
import ru.yandex.http.config.ClientHttpsConfigDefaults;
import ru.yandex.http.config.ImmutableClientHttpsConfig;
import ru.yandex.http.config.ImmutableHttpHostConfig;
import ru.yandex.mail.so.logger.RoutedLogRecordProducer;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.config.IniConfig;
import ru.yandex.parser.string.DurationParser;
import ru.yandex.parser.string.IntegerParser;
import ru.yandex.parser.string.NonNegativeIntegerValidator;

public abstract class AbstractRulesStatDatabaseConfig<P extends RoutedLogRecordProducer>
    implements RulesStatDatabaseConfig<P>
{
    public static final String DATABASE = "database";

    protected final String configName;
    protected final long batchSize;
    protected final long batchSavePeriod;
    protected final int batchSaveRetries;
    protected final long batchSaveRetryTimeout;
    protected final int workers;
    protected final long savingOperationTimeout;
    protected final List<ImmutableHttpHostConfig> hosts;
    protected final int port;
    protected final String dbName;
    protected final String userName;
    protected final String password;
    protected ImmutableClientHttpsConfig sslConfig;
    protected final long connectTimeout;
    protected final long socketTimeout;
    protected final int minPoolSize;
    protected final int maxPoolSize;

    protected AbstractRulesStatDatabaseConfig() {
        configName = null;
        batchSize = 0L;
        batchSavePeriod = 60000L;
        batchSaveRetries = 0;
        batchSaveRetryTimeout = 120000L;
        savingOperationTimeout = 30000L;
        workers = 1;
        hosts = new ArrayList<>();
        port = 0;
        dbName = null;
        userName = null;
        password = null;
        sslConfig = null;
        connectTimeout = 120000L;
        socketTimeout = 120000L;
        minPoolSize = 0;
        maxPoolSize = 100;
    }

    protected AbstractRulesStatDatabaseConfig(
        final IniConfig config,
        final String configName,
        final RulesStatDatabaseConfig<P> defaults)
        throws ConfigException
    {
        this.configName = configName;
        IniConfig database = null;
        if (config == null) {
            batchSize = defaults == null ? 0L : defaults.batchMinSize();
            batchSavePeriod = defaults == null ? 60000L : defaults.batchSavePeriod();
            batchSaveRetries = defaults == null ? 0 : defaults.batchSaveRetries();
            batchSaveRetryTimeout = defaults == null ? 120000L : defaults.batchSaveRetryTimeout();
            workers = defaults == null ? 1 : defaults.workers();
            savingOperationTimeout = defaults == null ? 30000L : defaults.savingOperationTimeout();
        } else {
            batchSize = config.get(
                "batch-min-size",
                defaults == null ? 0L : defaults.batchMinSize(),
                DurationParser.POSITIVE_LONG);
            batchSavePeriod = config.get(
                "batch-save-period",
                defaults == null ? 60000L : defaults.batchSavePeriod(),
                DurationParser.POSITIVE_LONG);
            batchSaveRetries = config.get(
                "batch-save-retries",
                defaults == null ? -1 : defaults.batchSaveRetries(),
                IntegerParser.INSTANCE);
            batchSaveRetryTimeout = config.get(
                "batch-save-retry-timeout",
                defaults == null ? 120000L : defaults.batchSaveRetryTimeout(),
                DurationParser.POSITIVE_LONG);
            workers = config.get(
                "workers",
                defaults == null ? 1 : defaults.workers(),
                IntegerParser.INSTANCE);
            savingOperationTimeout = config.get(
                "saving-operation-timeout",
                defaults == null ? 30000L : defaults.savingOperationTimeout(),
                DurationParser.POSITIVE_LONG);
            database = config.section(DATABASE);
        }
        hosts = new ArrayList<>();
        port = database == null ? 0 :
            database.get("port", defaults == null ? 0 : defaults.port(), NonNegativeIntegerValidator.INSTANCE);
        dbName = database == null ? null : database.getString("db", defaults == null ? null : defaults.dbName());
        userName = database == null ? null : database.getString("user", defaults == null ? null : defaults.userName());
        password = database == null ? null : database.getString("secret", defaults == null ? null : defaults.secret());
        if (database == null || database.section("ssl") == null
                || (database.section("ssl").keys().isEmpty() && database.section("ssl").sections().isEmpty()))
        {
            sslConfig = null;
        } else {
            sslConfig = new ClientHttpsConfigBuilder(
                database.section("ssl"),
                defaults == null ? ClientHttpsConfigDefaults.INSTANCE : defaults.sslConfig())
                .build();
        }
        connectTimeout =
            database == null ? 120000L : database.get("connect-timeout", 120000L, DurationParser.POSITIVE_LONG);
        socketTimeout =
            database == null ? 120000L : database.get("socket-timeout", 120000L, DurationParser.POSITIVE_LONG);
        minPoolSize = database == null ? 0 : database.get("min-pool-size", 0, IntegerParser.INSTANCE);
        maxPoolSize = database == null ? 100 : database.get("max-pool-size", 100, IntegerParser.INSTANCE);
        parseDatabaseConfig(database, defaults);
    }

    protected abstract void parseDatabaseConfig(final IniConfig config, final RulesStatDatabaseConfig<P> defaults)
        throws ConfigException;

    @Override
    public String name() {
        return configName;
    }

    @Override
    public long batchMinSize() {
        return batchSize;
    }

    @Override
    public long batchSavePeriod() {
        return batchSavePeriod;
    }

    @Override
    public int batchSaveRetries() {
        return batchSaveRetries;
    }

    @Override
    public long batchSaveRetryTimeout() {
        return batchSaveRetryTimeout;
    }

    @Override
    public int workers() {
        return workers;
    }

    @Override
    public long savingOperationTimeout() {
        return savingOperationTimeout;
    }

    @Override
    public List<ImmutableHttpHostConfig> hosts() {
        return hosts;
    }

    @Override
    public int port() {
        return port;
    }

    @Override
    public String dbName() {
        return dbName;
    }

    @Override
    public String userName() {
        return userName;
    }

    @Override
    public String secret() {
        return password;
    }

    @Override
    public ClientHttpsConfig sslConfig() {
        return sslConfig;
    }
}
