package ru.yandex.mail.so.logger.config;

import java.util.concurrent.TimeUnit;

import ru.yandex.client.tvm2.ImmutableTvm2ClientConfig;
import ru.yandex.client.tvm2.ImmutableTvm2ServiceConfig;
import ru.yandex.client.tvm2.Tvm2ClientConfig;
import ru.yandex.client.tvm2.Tvm2ClientConfigBuilder;
import ru.yandex.client.tvm2.Tvm2ServiceConfig;
import ru.yandex.client.tvm2.Tvm2ServiceConfigBuilder;
import ru.yandex.function.GenericFunction;
import ru.yandex.http.config.DnsConfigBuilder;
import ru.yandex.http.config.HttpHostConfigBuilder;
import ru.yandex.http.config.ImmutableDnsConfig;
import ru.yandex.http.config.ImmutableHttpHostConfig;
import ru.yandex.mail.so.logger.LogRecordContext;
import ru.yandex.mail.so.logger.LogStorage;
import ru.yandex.mail.so.logger.LogStorageException;
import ru.yandex.mail.so.logger.LogStorageType;
import ru.yandex.mail.so.logger.MdsLogStorage;
import ru.yandex.mail.so.logger.SpLogger;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.config.IniConfig;
import ru.yandex.parser.string.DoubleParser;
import ru.yandex.parser.string.DurationParser;
import ru.yandex.parser.string.LongMemorySizeParser;
import ru.yandex.parser.string.NonEmptyValidator;
import ru.yandex.parser.string.NonNegativeIntegerValidator;
import ru.yandex.parser.string.PositiveIntegerValidator;
import ru.yandex.stater.StaterConfig;
import ru.yandex.stater.StaterConfigBuilder;
import ru.yandex.stater.StatersConfigBuilder;

public class MdsLogStorageConfig extends NullLogStorageConfig {
    public static final MdsLogStorageConfig DEFAULTS = new MdsLogStorageConfig();
    public static final String TVM2 = "tvm2";

    private final String configName;
    private final ImmutableHttpHostConfig backendWriterConfig;
    private final ImmutableHttpHostConfig backendReaderConfig;
    private final ImmutableDnsConfig dnsConfig;
    private final long storeTTL;
    private final long batchMinSize;
    private final long batchMaxSize;
    private final long batchesMemoryLimit;
    private final String mdsNamespace;
    private final int workers;
    private final double batchSaveMaxRps;
    private final long batchSavePeriod;
    private final long batchSaveRetryTimeout;
    private final int batchSaveRetries;
    private final long savingOperationTimeout;
    private final int indexRetriesCount;
    private final long mdsHostnameRequestPeriod;
    private final String mdsDeletesQueueName;

    private ImmutableTvm2ClientConfig tvm2ClientConfig;
    private ImmutableTvm2ServiceConfig tvm2ServiceConfig;

    protected MdsLogStorageConfig() {
        super();
        configName = null;
        backendWriterConfig = null;
        backendReaderConfig = null;
        dnsConfig = null;
        tvm2ServiceConfig = null;
        tvm2ClientConfig = null;
        workers = INSTANCE.workers();
        storeTTL = TimeUnit.DAYS.toMillis(30L);
        batchMinSize = INSTANCE.batchMinSize();
        batchMaxSize = INSTANCE.batchMaxSize();
        batchesMemoryLimit = INSTANCE.batchesMemoryLimit();
        batchSaveMaxRps = INSTANCE.batchSaveMaxRps();
        batchSavePeriod = INSTANCE.batchSavePeriod();
        batchSaveRetryTimeout = INSTANCE.batchSaveRetryTimeout();
        batchSaveRetries = INSTANCE.batchSaveRetries();
        savingOperationTimeout = INSTANCE.savingOperationTimeout();
        indexRetriesCount = INSTANCE.indexRetriesCount();
        mdsHostnameRequestPeriod = 10000L;
        mdsNamespace = "sologger";
        mdsDeletesQueueName = null;
    }

    public MdsLogStorageConfig(final IniConfig config, final String name, final MdsLogStorageConfig other)
        throws ConfigException
    {
        configName = name;
        GenericFunction<StaterConfig, StaterConfigBuilder, RuntimeException> transformer = staterConfig -> {
            if (staterConfig == null) {
                return null;
            } else if (name == null || name.isEmpty()) {
                return new StaterConfigBuilder(staterConfig);
            } else {
                return new StaterConfigBuilder(staterConfig).prefix(name + '-' + staterConfig.prefix());
            }
        };
        MdsLogStorageConfig defaults = other;
        if (defaults == null) {
            defaults = DEFAULTS;
        }
        workers = config.get("workers", 1, PositiveIntegerValidator.INSTANCE);
        HttpHostConfigBuilder backendWriterConfigBuilder = new HttpHostConfigBuilder(config.section("to-write"));
        StatersConfigBuilder statersConfigBuilder = backendWriterConfigBuilder.statersConfig();
        if (statersConfigBuilder != null) {
            statersConfigBuilder.staters(statersConfigBuilder.staters().transform(transformer));
        }
        backendWriterConfig = backendWriterConfigBuilder.build();
        HttpHostConfigBuilder backendReaderConfigBuilder = new HttpHostConfigBuilder(config.section("to-read"));
        statersConfigBuilder = backendReaderConfigBuilder.statersConfig();
        if (statersConfigBuilder != null) {
            statersConfigBuilder.staters(statersConfigBuilder.staters().transform(transformer));
        }
        backendReaderConfig = backendReaderConfigBuilder.build();
        dnsConfig = new DnsConfigBuilder(config.section("dns")).build();
        tvm2ServiceConfig = new Tvm2ServiceConfigBuilder(config.section(TVM2)).build();
        tvm2ClientConfig = new Tvm2ClientConfigBuilder(config.section(TVM2)).build();
        storeTTL = config.get("store-ttl", defaults.storeTTL(), DurationParser.POSITIVE_LONG);
        batchMinSize = config.get("batch-min-size", defaults.batchMinSize(), LongMemorySizeParser.INSTANCE);
        batchMaxSize = config.get("batch-max-size", defaults.batchMaxSize(), LongMemorySizeParser.INSTANCE);
        if (batchMinSize > batchMaxSize) {
            throw new ConfigException("Batch's min size must be not less then batch's max size!");
        }
        batchesMemoryLimit =
            config.get("batches-memory-limit", defaults.batchesMemoryLimit(), LongMemorySizeParser.INSTANCE);
        batchSaveMaxRps = config.get("batch-save-max-rps", defaults.batchSaveMaxRps(), DoubleParser.INSTANCE);
        batchSavePeriod = config.get("batch-save-period", defaults.batchSavePeriod(), DurationParser.POSITIVE_LONG);
        batchSaveRetryTimeout =
            config.get("batch-save-retry-timeout", defaults.batchSaveRetryTimeout(), DurationParser.POSITIVE_LONG);
        batchSaveRetries =
            config.get("batch-save-retries", defaults.batchSaveRetries(), NonNegativeIntegerValidator.INSTANCE);
        savingOperationTimeout =
            config.get("saving-operation-timeout", defaults.savingOperationTimeout(), DurationParser.POSITIVE_LONG);
        indexRetriesCount =
            config.get(
                "index-retries-count",
                defaults.batchSaveRetries(),
                NonNegativeIntegerValidator.INSTANCE);
        mdsHostnameRequestPeriod =
            config.get(
                "mds-hostname-request-period",
                defaults.mdsHostnameRequestPeriod(),
                DurationParser.POSITIVE_LONG);
        mdsNamespace = config.get("namespace", defaults.mdsNamespace(), NonEmptyValidator.INSTANCE);
        mdsDeletesQueueName =
            config.get(
                "mds-deletes-queue-name",
                defaults.mdsDeletesQueueName(),
                NonEmptyValidator.INSTANCE);
    }

    @Override
    public String name() {
        return configName;
    }

    public ImmutableHttpHostConfig backendWriterConfig() {
        return backendWriterConfig;
    }

    public ImmutableHttpHostConfig backendReaderConfig() {
        return backendReaderConfig;
    }

    public ImmutableDnsConfig dnsConfig() {
        return dnsConfig;
    }

    public ImmutableTvm2ServiceConfig tvm2ServiceConfig() {
        return tvm2ServiceConfig;
    }

    public void tvm2ServiceConfig(final Tvm2ServiceConfig tvm2ServiceConfig) throws ConfigException
    {
        this.tvm2ServiceConfig = new ImmutableTvm2ServiceConfig(tvm2ServiceConfig);
    }

    public ImmutableTvm2ClientConfig tvm2ClientConfig() {
        return tvm2ClientConfig;
    }

    public void tvm2ClientConfig(final Tvm2ClientConfig tvm2ClientConfig) throws ConfigException
    {
        this.tvm2ClientConfig = new ImmutableTvm2ClientConfig(tvm2ClientConfig);
    }

    public String mdsTvmClientId() {
        return tvm2ClientConfig.destinationClientId();
    }

    @Override
    public long storeTTL() {
        return storeTTL;
    }

    @Override
    public long batchMinSize() {
        return batchMinSize;
    }

    @Override
    public long batchMaxSize() {
        return batchMaxSize;
    }

    @Override
    public long batchesMemoryLimit() {
        return batchesMemoryLimit;
    }

    @Override
    public double batchSaveMaxRps() {
        return batchSaveMaxRps;
    }

    @Override
    public long batchSavePeriod() {
        return batchSavePeriod;
    }

    @Override
    public long batchSaveRetryTimeout() {
        return batchSaveRetryTimeout;
    }

    @Override
    public int batchSaveRetries() {
        return batchSaveRetries;
    }

    @Override
    public int indexRetriesCount() {
        return indexRetriesCount;
    }

    public long mdsHostnameRequestPeriod() {
        return mdsHostnameRequestPeriod;
    }

    public String mdsNamespace() {
        return mdsNamespace;
    }

    public String mdsDeletesQueueName() {
        return mdsDeletesQueueName;
    }

    @Override
    public LogStorageType type() {
        return LogStorageType.MDS;
    }

    @Override
    public int workers() {
        return workers;
    }

    @Override
    public long savingOperationTimeout() {
        return savingOperationTimeout;
    }

    @Override
    public LogStorage<LogRecordContext> createStorage(final SpLogger spLogger, final String name)
        throws LogStorageException
    {
        return new MdsLogStorage(name, spLogger, this);
    }
}
