package ru.yandex.mail.so.logger.config;

import java.util.Locale;

import com.mongodb.ConnectionString;
import com.mongodb.MongoClientSettings;
import com.mongodb.reactivestreams.client.MongoClient;
import com.mongodb.reactivestreams.client.MongoClients;
import org.apache.http.HttpHost;

import ru.yandex.http.config.HttpHostConfig;
import ru.yandex.http.config.HttpHostConfigBuilder;
import ru.yandex.http.proxy.HttpProxy;
import ru.yandex.mail.so.logger.BasicRoutedLogRecordProducer;
import ru.yandex.mail.so.logger.MongoRulesStatDatabase;
import ru.yandex.mail.so.logger.RulesStatDatabase;
import ru.yandex.mail.so.logger.RulesStatDatabaseType;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.config.IniConfig;
import ru.yandex.parser.string.EnumParser;

public class MongoRulesStatDatabaseConfig extends AbstractRulesStatDatabaseConfig<BasicRoutedLogRecordProducer> {
    public static final MongoRulesStatDatabaseConfig DEFAULTS = new MongoRulesStatDatabaseConfig();

    public enum SavingMethod {
        SIMPLE_REQUESTS("simple-requests"),
        SYNC_REQUESTS("sync-requests"),
        TRANSACTIONS("transactions"),
        BULK_REQUESTS("bulk-requests");

        private final String name;

        SavingMethod(final String name) {
            this.name = name;
        }

        public String method() {
            return name;
        }
    }

    protected SavingMethod savingMethod;

    protected MongoRulesStatDatabaseConfig() {
        super();
    }

    public MongoRulesStatDatabaseConfig(
        final IniConfig config,
        final String configName,
        final MongoRulesStatDatabaseConfig other)
        throws ConfigException
    {
        super(config, configName, other);
        savingMethod = SavingMethod.TRANSACTIONS;
        if (config != null) {
            try {
                savingMethod = config.get(
                    "saving-method",
                    other == null ? savingMethod : other.savingMethod(),
                    new EnumParser<>(SavingMethod.class));
            } catch (RuntimeException e) {
                throw new ConfigException("Unable to parse saving-method", e);
            }
        }
    }

    @Override
    public RulesStatDatabaseType type() {
        return RulesStatDatabaseType.MONGODB;
    }

    public SavingMethod savingMethod() {
        return savingMethod;
    }

    @Override
    public RulesStatDatabase<BasicRoutedLogRecordProducer> createClient(
        final HttpProxy<?> httpProxy,
        final RulesStatDatabasesOperatorConfig<BasicRoutedLogRecordProducer> rulesStatDatabasesOperatorConfig,
        final String name)
    {
        return new MongoRulesStatDatabase(httpProxy, rulesStatDatabasesOperatorConfig, name);
    }

    public MongoClient createMongoClient() {
        ConnectionString connString = new ConnectionString(connectionString());
        MongoClientSettings.Builder settingsBuilder = MongoClientSettings.builder()
            .applyConnectionString(connString)
            .retryWrites(false);
        if (sslConfig != null && sslConfig.protocol() != null) {
            settingsBuilder.applyToSslSettings(builder -> {
                builder.enabled(true);
                builder.context(sslConfig.getSSLContext());
            });
        } else {
            settingsBuilder.applyToSslSettings(builder -> {
                builder.enabled(false);
                builder.invalidHostNameAllowed(true);
            });
        }
        return MongoClients.create(settingsBuilder.build());
    }

    @Override
    @SuppressWarnings("StringSplitter")
    protected void parseDatabaseConfig(
        final IniConfig config,
        final RulesStatDatabaseConfig<BasicRoutedLogRecordProducer> defaults)
        throws ConfigException
    {
        if (config != null) {
            String hostsStr = config.getOrNull("hosts");
            if (hostsStr == null) {
                hostsStr = "localhost";
            }
            for (final String host : hostsStr.split(",")) {
                hosts.add(new HttpHostConfigBuilder().host(new HttpHost(host, port)).connections(100).build());
            }
        }
    }

    public String connectionString() {
        StringBuilder sb = new StringBuilder("mongodb://");
        if (userName != null) {
            sb.append(userName);
            if (password != null) {
                sb.append(':').append(password);
            }
            sb.append('@');
        }
        boolean notFirst = false;
        for (HttpHostConfig host : hosts) {
            if (notFirst) {
                sb.append(',');
            } else {
                notFirst = true;
            }
            sb.append(host.host().toHostString());
        }
        if (dbName != null) {
            sb.append('/').append(dbName);
        }
        sb.append("?retryWrites=false&appname=SpLogger-").append(name()).append("-RulesStatSaver");
        if (sslConfig != null && sslConfig.protocol() != null) {
            sb.append('&').append(sslConfig.protocol().toLowerCase(Locale.ROOT)).append("=true");
        }
        if (connectTimeout > 0) {
            sb.append("&connectTimeoutMS=").append(connectTimeout);
        }
        if (socketTimeout > 0) {
            sb.append("&socketTimeoutMS=").append(socketTimeout);
        }
        if (minPoolSize > 0) {
            sb.append("&minPoolSize=").append(minPoolSize);
        }
        if (maxPoolSize > minPoolSize) {
            sb.append("&maxPoolSize=").append(maxPoolSize);
        }
        return sb.toString();
    }
}
