package ru.yandex.mail.so.logger;

import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.stream.Collectors;

import org.apache.http.HttpEntityEnclosingRequest;
import org.apache.http.HttpException;
import org.apache.http.HttpHeaders;
import org.apache.http.HttpRequest;
import org.apache.http.HttpStatus;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.entity.ContentType;
import org.apache.http.nio.protocol.HttpAsyncExchange;
import org.apache.http.nio.protocol.HttpAsyncRequestConsumer;
import org.apache.http.nio.protocol.HttpAsyncRequestHandler;
import org.apache.http.protocol.HttpContext;

import ru.yandex.collection.Pattern;
import ru.yandex.data.compressor.CompressorException;
import ru.yandex.data.compressor.DataCompressor;
import ru.yandex.function.GenericBiConsumer;
import ru.yandex.http.proxy.AbstractProxySessionCallback;
import ru.yandex.http.proxy.BasicProxySession;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.BadResponseException;
import ru.yandex.http.util.DoubleFutureCallback;
import ru.yandex.http.util.MultiFutureCallback;
import ru.yandex.http.util.ServerException;
import ru.yandex.http.util.YandexHttpStatus;
import ru.yandex.http.util.nio.AsyncStringConsumer;
import ru.yandex.http.util.request.RequestHandlerMapper;
import ru.yandex.http.util.request.RequestInfo;
import ru.yandex.mail.so.logger.config.ImmutableStoreLogHandlerConfig;
import ru.yandex.mail.so.logger.config.ImmutableStoreLoggerConfig;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.stater.RequestsStater;
import ru.yandex.util.timesource.TimeSource;

public class StoreLogsHandler extends RequestsStater implements HttpAsyncRequestHandler<String>
{
    private final SpLoggerConsumer spLoggerConsumer;
    private final ImmutableStoreLogHandlerConfig config;
    /**
     * Loggers related to this store handler: uri -> logger
     */
    private final Map<String, Logger> specLoggers;
    /**
     * Routes related to this store handler: uri -> route
     */
    private final Map<String, Route> routes;
    private final String path;
    private final DataCompressor decompressor;
    private final RulesStatDatabase<BasicRoutedLogRecordProducer> dbClient;

    public StoreLogsHandler(
        final ImmutableStoreLogHandlerConfig config,
        final SpLoggerConsumer spLoggerConsumer,
        final String path)
        throws ConfigException
    {
        super(config);
        this.config = config;
        this.spLoggerConsumer = spLoggerConsumer;
        this.path = path;
        specLoggers = new HashMap<>();
        config.build(spLoggerConsumer.handlersManager()).traverse(new LoggersRegistrator());
        routes = new HashMap<>();
        config.loggers().traverse(new StoreLoggerConfigVisitor());
        try {
            this.decompressor = DataCompressor.valueOf(config.decompression().toUpperCase(Locale.ROOT));
        } catch (IllegalArgumentException e) {
            throw new ConfigException("StoreLogsHandler: decompression method <" + config.decompression()
                + "> for path <" + path + "> is unknown. Available methods: "
                + Arrays.stream(DataCompressor.values()).map(x -> x.name().toLowerCase(Locale.ROOT))
                .collect(Collectors.joining(", ")) + '.');
        }
        dbClient = spLoggerConsumer.rulesStatDatabase(
            config.rulesStatDatabase() == null ? NullRulesStatDatabase.NULL : config.rulesStatDatabase());
        spLoggerConsumer.logger().info("StoreLogsHandler: dbClient=" + dbClient + ", rulesStatDatabase="
            + config.rulesStatDatabase() + ", rulesStatDatabases size="
            + spLoggerConsumer.rulesStatDatabasesConfig().rulesStatDatabases().size());
        if (dbClient == null && config.rulesStatDatabase() != null
            && spLoggerConsumer.rulesStatDatabasesConfig().rulesStatDatabases().size() > 0
            && spLoggerConsumer.rulesStatDatabasesConfig().rulesStatDatabases()
                .get(config.rulesStatDatabase()).type() != RulesStatDatabaseType.NULL)
        {
            throw new ConfigException("StoreLogsHandler: rules stat's database with name <" + config.rulesStatDatabase()
                + "> for path <" + path + "> is not configured");
        }
    }

    public ImmutableStoreLogHandlerConfig config() {
        return config;
    }

    @SuppressWarnings("unused")
    public String path() {
        return path;
    }

    public void saveLogRecord(
        final LogRecordProducer logRecordProducer,
        final String uri,
        final ProxySession session,
        final FutureCallback<Void> callback)
    {
        try {
            specLoggers.get(uri).fine(logRecordProducer.logRecord().stripTrailing());
        } catch (Exception e) {
            session.logger().log(Level.SEVERE, "StoreLogsHandler.saveLogRecord failed", e);
            callback.failed(e);
            return;
        }
        callback.completed(null);
    }

    public void saveRulesStat(
        final BasicRoutedLogRecordProducer logRecordProducer,
        final String uri,
        final ProxySession session,
        final FutureCallback<Void> callback)
    {
        MultiFutureCallback<Void> multiCallback =
            new MultiFutureCallback<>(new DbOperationsCallback(session.logger(), callback));
        for (BasicLogRecordProducer basicRoutedLogRecordProducer : logRecordProducer.split()) {
            dbClient.save(
                (BasicRoutedLogRecordProducer) basicRoutedLogRecordProducer,
                session,
                multiCallback.newCallback());
        }
        multiCallback.done();
    }

    @Override
    public HttpAsyncRequestConsumer<String> processRequest(final HttpRequest request, final HttpContext context)
        throws BadRequestException
    {
        if (RequestHandlerMapper.POST.equals(request.getRequestLine().getMethod())
                && !(request instanceof HttpEntityEnclosingRequest))
        {
            throw new BadRequestException("Payload expected");
        }
        return new AsyncStringConsumer();
    }

    @Override
    public void handle(final String payload, final HttpAsyncExchange exchange, final HttpContext context)
        throws HttpException
    {
        long startTime = TimeSource.INSTANCE.currentTimeMillis();
        ProxySession session = new BasicProxySession(spLoggerConsumer, exchange, context);
        String method = exchange.getRequest().getRequestLine().getMethod();
        String uri = exchange.getRequest().getRequestLine().getUri();
        session.logger().info("StoreLogsHandler: method=" + method + ", uri=" + uri);
        if (RequestHandlerMapper.POST.equals(method)) {
            if (payload == null || payload.isEmpty()) {
                session.logger().log(Level.SEVERE, "StoreLogsHandler: empty payload!");
            } else if (!specLoggers.containsKey(uri)) {
                session.logger().log(Level.SEVERE, "StoreLogsHandler: logger for uri " + uri + " is not configured!");
            } else {
                BasicRoutedLogRecordProducer logRecordProducer;
                try {
                    logRecordProducer = new BasicRoutedLogRecordProducer(
                        payload,
                        routes.get(uri),
                        ContentType.APPLICATION_OCTET_STREAM,
                        ContentType.parse(exchange.getRequest().getHeaders(HttpHeaders.CONTENT_TYPE)[0].getValue()),
                        decompressor);
                } catch (CompressorException e) {
                    stat(YandexHttpStatus.SC_NOT_ACCEPTABLE, startTime);
                    throw new ServerException(HttpStatus.SC_NOT_ACCEPTABLE, "Uncompressing of payload failed: " + e, e);
                }
                session.logger().info("StoreLogsHandler: method = " + method + ", uri = " + uri + ", route = "
                    + logRecordProducer.route() + ", decompression = " + decompressor + ", MongoDB client's type: "
                    + dbClient.type());
                DoubleFutureCallback<Void, Void> doubleCallback =
                    new DoubleFutureCallback<>(
                        new AbstractProxySessionCallback<>(session) {
                            @Override
                            public void completed(final Map.Entry<Void, Void> unused) {
                                stat(YandexHttpStatus.SC_OK, startTime);
                                session.logger().info("StoreLogsHandler.handle: finished");
                                session.response(HttpStatus.SC_OK);
                            }

                            @Override
                            public void cancelled() {
                                super.cancelled();
                                stat(YandexHttpStatus.SC_CLIENT_CLOSED_REQUEST, startTime);
                            }

                            @Override
                            public void failed(final Exception e) {
                                super.failed(e);
                                if (e instanceof BadResponseException) {
                                    stat(((BadResponseException) e).statusCode(), startTime);
                                } else {
                                    stat(YandexHttpStatus.SC_REMOTE_CLOSED_REQUEST, startTime);
                                }
                            }
                        });
                saveLogRecord(logRecordProducer, uri, session, doubleCallback.first());
                saveRulesStat(logRecordProducer, uri, session, doubleCallback.second());
            }
        } else {
            stat(YandexHttpStatus.SC_METHOD_NOT_ALLOWED, startTime);
            throw new ServerException(HttpStatus.SC_METHOD_NOT_ALLOWED, "Method Not Allowed");
        }
    }

    public void stat(final int status, final long startTime) {
        accept(new ru.yandex.stater.RequestInfo(
            TimeSource.INSTANCE.currentTimeMillis(),
            status,
            startTime,
            startTime,
            0L,
            0L));
    }

    private class LoggersRegistrator implements GenericBiConsumer<Pattern<RequestInfo>, Logger, ConfigException>
    {
        @Override
        public void accept(final Pattern<RequestInfo> pattern, final Logger logger) throws ConfigException
        {
            specLoggers.put(pattern.path(), logger);
            spLoggerConsumer.registerLoggerForLogrotate(logger);
        }
    }

    private class StoreLoggerConfigVisitor
        implements GenericBiConsumer<Pattern<RequestInfo>, ImmutableStoreLoggerConfig, ConfigException>
    {
        @Override
        public void accept(final Pattern<RequestInfo> pattern, final ImmutableStoreLoggerConfig config)
            throws ConfigException
        {
            routes.put(pattern.path(), config.route());
        }
    }

    private static class DbOperationsCallback extends AbstractFilterFutureCallback<List<Void>, Void> {
        private final Logger logger;

        public DbOperationsCallback(final Logger logger, final FutureCallback<Void> callback) {
            super(callback);
            this.logger = logger;
        }

        @Override
        public void completed(final List<Void> ignored) {
            logger.info("DbOperationsCallback: operations completed successfully");
            callback.completed(null);
        }

        @Override
        public void failed(final Exception e) {
            logger.log(Level.SEVERE, "DbOperationsCallback: operations failed", e);
            callback.failed(e);
        }

        @Override
        public void cancelled() {
            logger.warning("DbOperationsCallback: operations cancelled");
            callback.cancelled();
        }
    }
}
