package ru.yandex.mail.so.logger.config;

import java.util.Map;

import ru.yandex.collection.Pattern;
import ru.yandex.collection.PatternMap;
import ru.yandex.collection.PatternStringPredicate;
import ru.yandex.function.GenericFunction;
import ru.yandex.http.util.request.RequestInfo;
import ru.yandex.http.util.request.RequestPatternParser;
import ru.yandex.logger.LoggerFileConfigDefaults;
import ru.yandex.mail.so.logger.LogRecordsHandlerType;
import ru.yandex.parser.config.ConfigBuilder;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.config.IniConfig;
import ru.yandex.parser.string.EnumParser;
import ru.yandex.stater.AbstractStaterConfigBuilder;
import ru.yandex.stater.StaterConfigDefaults;

public abstract class AbstractStoreLogHandlerConfigBuilder<T extends AbstractStoreLogHandlerConfigBuilder<T>>
    extends AbstractStaterConfigBuilder<T>
    implements ConfigBuilder<T>, StoreLogHandlerConfig
{
    public static final DefaultLoggerFileConfig DEFAILT_LOG_CONFIG = new DefaultLoggerFileConfig();

    private static final GenericFunction<StoreLoggerConfig, StoreLoggerConfigBuilder, RuntimeException>
        TRANSFORMER = config -> {
            if (config == null) {
                return null;
            } else {
                return new StoreLoggerConfigBuilder(config);
            }
        };

    private String baseDir;
    private String decompression;
    private Pattern<RequestInfo> basePattern;
    private LogRecordsHandlerType logRecordsHandlerType;
    private PatternMap<RequestInfo, ? extends StoreLoggerConfig> loggers;
    private String rulesStatDatabase;

    protected AbstractStoreLogHandlerConfigBuilder(final StoreLogHandlerConfig config) {
        super(config);
        baseDir = config.baseDir();
        decompression = config.decompression();
        basePattern = config.basePattern();
        rulesStatDatabase = config.rulesStatDatabase();
        logRecordsHandlerType = config.type();
        loggers = config.loggers().transform(TRANSFORMER);
    }

    protected AbstractStoreLogHandlerConfigBuilder(
        final IniConfig config,
        final Pattern<RequestInfo> basePattern,
        final StoreLogHandlerConfig defaults)
        throws ConfigException
    {
        super(config.section("stat"), defaults == null ? StaterConfigDefaults.INSTANCE : defaults);
        baseDir = config.getString("base-dir", defaults == null ? null : defaults.baseDir());
        decompression = config.getString("decompression", defaults == null ? null : defaults.decompression());
        this.basePattern = basePattern;
        logRecordsHandlerType = config.get("type", null, new EnumParser<>(LogRecordsHandlerType.class));
        rulesStatDatabase = config.getString("rules-stat-db", defaults == null ? null : defaults.rulesStatDatabase());
        loggers =
            loadLoggers(config.section("logger"), basePattern, baseDir, defaults == null ? null : defaults.loggers());
    }

    @Override
    public String baseDir() {
        return baseDir;
    }

    public T baseDir(final String baseDir) {
        this.baseDir = baseDir;
        return self();
    }

    @Override
    public String decompression() {
        return decompression;
    }

    public T decompression(final String decompression) {
        this.decompression = decompression;
        return self();
    }

    @Override
    public Pattern<RequestInfo> basePattern() {
        return basePattern;
    }

    public T basePattern(final Pattern<RequestInfo> basePattern) {
        this.basePattern = basePattern;
        return self();
    }

    @Override
    public LogRecordsHandlerType type() {
        return logRecordsHandlerType;
    }

    public T type(final LogRecordsHandlerType logRecordsHandlerType) {
        this.logRecordsHandlerType = logRecordsHandlerType;
        return self();
    }

    public T type(final String logRecordsHandlerTypeStr) {
        EnumParser<LogRecordsHandlerType> parser = new EnumParser<>(LogRecordsHandlerType.class);
        try {
            this.logRecordsHandlerType = parser.apply(logRecordsHandlerTypeStr);
        } catch (Exception e) {
            this.logRecordsHandlerType = LogRecordsHandlerConfigDefaults.INSTANCE.type();
        }
        return self();
    }

    @Override
    public PatternMap<RequestInfo, ? extends StoreLoggerConfig> loggers() {
        return loggers;
    }

    public T loggers(final PatternMap<RequestInfo, ? extends StoreLoggerConfig> loggers) {
        this.loggers = loggers;
        return self();
    }

    @Override
    public String rulesStatDatabase() {
        return rulesStatDatabase;
    }

    public T rulesStatDatabase(final String rulesStatDatabase) {
        this.rulesStatDatabase = rulesStatDatabase;
        return self();
    }

    public static PatternMap<RequestInfo, StoreLoggerConfigBuilder> loadLoggers(
        final IniConfig section,
        final Pattern<RequestInfo> basePattern,
        final String baseDir,
        final PatternMap<RequestInfo, ? extends StoreLoggerConfig> defaults)
        throws ConfigException
    {
        PatternMap<RequestInfo, StoreLoggerConfigBuilder> loggers = new PatternMap<>();
        for (Map.Entry<String, IniConfig> entry : section.sections().entrySet()) {
            String subsection = entry.getKey();
            if (PatternStringPredicate.INSTANCE.test(subsection)) {
                Pattern<RequestInfo> pattern = RequestPatternParser.INSTANCE.apply(basePattern.path() + subsection);
                IniConfig logConfig = entry.getValue();
                String filePath = logConfig.getOrNull("file");
                if (filePath != null) {
                    filePath = (baseDir == null ? ""
                        : (baseDir.endsWith("/") && baseDir.length() > 1
                            ? baseDir.substring(0, baseDir.length() - 2) : baseDir))
                        + '/' + (filePath.startsWith("/") ? filePath.substring(1) : filePath);
                    logConfig.put("file", filePath);
                }
                loggers.put(
                    pattern,
                    new StoreLoggerConfigBuilder(logConfig, defaults.get(pattern), DEFAILT_LOG_CONFIG));
            }
        }
        return loggers;
    }

    public static class DefaultLoggerFileConfig extends LoggerFileConfigDefaults {
        @Override
        public String logFormat() {
            return "%{message}";
        }
    }
}
