package ru.yandex.mail.so.logger.config;

import java.util.Map;

import ru.yandex.collection.Pattern;
import ru.yandex.collection.PatternMap;
import ru.yandex.function.GenericFunction;
import ru.yandex.http.util.request.RequestInfo;
import ru.yandex.http.util.request.RequestPatternParser;
import ru.yandex.parser.config.ConfigBuilder;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.config.IniConfig;

public abstract class AbstractStoreLogHandlersConfigBuilder<T extends AbstractStoreLogHandlersConfigBuilder<T>>
    implements ConfigBuilder<T>, StoreLogHandlersConfig
{
    public static final String SECTION = "store-handler";

    private static final GenericFunction<StoreLogHandlerConfig, StoreLogHandlerConfigBuilder, RuntimeException>
        TRANSFORMER = config -> {
            if (config == null) {
                return null;
            } else {
                return new StoreLogHandlerConfigBuilder(config);
            }
        };

    private PatternMap<RequestInfo, StoreLogHandlerConfigBuilder> storeLogHandlers;

    protected AbstractStoreLogHandlersConfigBuilder() {
        this(StoreLogHandlersConfigDefaults.INSTANCE);
    }

    protected AbstractStoreLogHandlersConfigBuilder(final StoreLogHandlersConfig config)
    {
        storeLogHandlers = config.storeLogHandlers().transform(TRANSFORMER);
    }

    protected AbstractStoreLogHandlersConfigBuilder(final IniConfig config) throws ConfigException
    {
        this(config, StoreLogHandlersConfigDefaults.INSTANCE);
    }

    protected AbstractStoreLogHandlersConfigBuilder(final IniConfig config, final StoreLogHandlersConfig defaults)
        throws ConfigException
    {
        storeLogHandlers = loadStoreLogHandlers(config, defaults);
    }

    public static PatternMap<RequestInfo, StoreLogHandlerConfigBuilder> loadStoreLogHandlers(
        final IniConfig section,
        final StoreLogHandlersConfig defaults)
        throws ConfigException
    {
        PatternMap<RequestInfo, StoreLogHandlerConfigBuilder> logRecordsHandlers = new PatternMap<>();
        for (Map.Entry<String, IniConfig> entry : section.sections().entrySet()) {
            String subsection = entry.getKey();
            if (subsection.indexOf('/') != -1 || subsection.indexOf(':') != -1 || subsection.indexOf('*') != -1) {
                if (!subsection.endsWith("*")) {
                    subsection += '*';
                }
                Pattern<RequestInfo> pattern = RequestPatternParser.INSTANCE.apply(subsection);
                logRecordsHandlers.put(
                    pattern,
                    new StoreLogHandlerConfigBuilder(
                        entry.getValue(),
                        pattern,
                        defaults.storeLogHandlers().get(pattern)));
            }
        }
        return logRecordsHandlers;
    }

    @Override
    public PatternMap<RequestInfo, StoreLogHandlerConfigBuilder> storeLogHandlers() {
        return storeLogHandlers;
    }

    public T storeLogHandlers(final PatternMap<RequestInfo, ? extends StoreLogHandlerConfig> storeLogHandlers)
    {
        this.storeLogHandlers = storeLogHandlers.transform(TRANSFORMER);
        return self();
    }

    public T storeLogHandler(
        final Pattern<RequestInfo> pattern,
        final StoreLogHandlerConfigBuilder storeLogHandler)
    {
        storeLogHandlers.put(pattern, storeLogHandler);
        return self();
    }

    public T asterisk(final StoreLogHandlerConfigBuilder storeLogHandler) {
        storeLogHandlers.put(new Pattern<>("", true), storeLogHandler);
        return self();
    }

    public ImmutableStoreLogHandlersConfig build() throws ConfigException {
        return new ImmutableStoreLogHandlersConfig(this);
    }
}
