package ru.yandex.mail.so.logger.launcher;

import java.io.IOException;
import java.util.List;
import java.util.Map;

import org.apache.http.HttpException;
import org.apache.http.HttpRequest;
import org.apache.http.entity.ContentType;
import org.apache.http.message.BasicHttpEntityEnclosingRequest;
import org.apache.http.nio.entity.NStringEntity;
import org.apache.http.nio.protocol.HttpAsyncExchange;
import org.apache.http.nio.protocol.HttpAsyncRequestConsumer;
import org.apache.http.nio.protocol.HttpAsyncRequestHandler;
import org.apache.http.protocol.HttpContext;

import ru.yandex.http.util.ServiceUnavailableException;
import ru.yandex.http.util.YandexHeaders;
import ru.yandex.http.util.nio.FilterHttpAsyncExchange;
import ru.yandex.http.util.request.RequestHandlerMapper;
import ru.yandex.io.StringBuilderWriter;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonWriter;
import ru.yandex.mail.so.logger.IndexField;
import ru.yandex.mail.so.logger.LogRecordContext;
import ru.yandex.parser.string.DurationParser;
import ru.yandex.parser.string.NonEmptyValidator;
import ru.yandex.parser.uri.CgiParams;
import ru.yandex.parser.uri.UriParser;
import ru.yandex.util.string.StringUtils;
import ru.yandex.util.timesource.TimeSource;

public class SpLoggerIndexUpdateHandler implements HttpAsyncRequestHandler<HttpRequest>
{
    private static final String PREFIX = "prefix";
    private static final String LOG_PREFIX = "log_";

    private final String service;
    private final HttpAsyncRequestHandler<HttpRequest> next;

    public SpLoggerIndexUpdateHandler(final String service, final HttpAsyncRequestHandler<HttpRequest> next) {
        this.service = service;
        this.next = next;
    }

    @Override
    public HttpAsyncRequestConsumer<HttpRequest> processRequest(final HttpRequest request, final HttpContext context)
        throws HttpException, IOException
    {
        return next.processRequest(request, context);
    }

    @Override
    public void handle(final HttpRequest request, final HttpAsyncExchange exchange, final HttpContext context)
        throws HttpException
    {
        String uri = request.getRequestLine().getUri();
        UriParser uriParser = new UriParser(uri);
        CgiParams params = new CgiParams(uriParser.queryParser());
        StringBuilderWriter sbw = new StringBuilderWriter();
        List<String> longFields = List.of("ts", "code", "uid", "rcpt_uid");
        List<String> skipFields = List.of("prefix", "type", "route", "ttl");
        try (JsonWriter writer = JsonType.DOLLAR.create(sbw)) {
            writer.startObject();
            String logType = params.get(IndexField.TYPE.fieldName(), "delivery", NonEmptyValidator.TRIMMED);
            String route = params.get(IndexField.ROUTE.fieldName(), "in", NonEmptyValidator.TRIMMED);
            String rcptUid = params.getOrNull(IndexField.RCPT_UID.fieldName());
            long prefix = params.getLong(PREFIX, 0L);
            writer.key(PREFIX);
            writer.value(prefix);
            writer.key("docs");
            writer.startArray();
            writer.startObject();
            writer.key(IndexField.ID.fieldName());
            writer.value(
                LogRecordContext.logId(
                    logType,
                    route,
                    Long.toString(prefix),
                    rcptUid == null || rcptUid.isEmpty() ? 0 : Long.parseLong(rcptUid)));
            if (!uriParser.path().decode().startsWith("/delete")) {
                writer.key(IndexField.TYPE.fieldName());
                writer.value(logType);
                long ttl = params.get("ttl", DurationParser.POSITIVE_LONG);
                writer.key(IndexField.EXPIRE_TIMESTAMP.fieldName());
                writer.value(TimeSource.INSTANCE.currentTimeMillis() + ttl);
                for (final Map.Entry<String, List<String>> entry : params.entrySet()) {
                    if (skipFields.contains(entry.getKey()) || skipFields.contains(LOG_PREFIX + entry.getKey())) {
                        continue;
                    }
                    writer.key(StringUtils.concat(LOG_PREFIX, entry.getKey()));
                    if (longFields.contains(entry.getKey())) {
                        writer.value(Long.parseLong(entry.getValue().get(0)));
                    } else {
                        writer.value(entry.getValue().get(0));
                    }
                }
            }
            writer.endObject();
            writer.endArray();
            writer.endObject();
            BasicHttpEntityEnclosingRequest nextRequest =
                new BasicHttpEntityEnclosingRequest(RequestHandlerMapper.POST, uri);
            nextRequest.setEntity(new NStringEntity(sbw.toString(), ContentType.APPLICATION_JSON));
            nextRequest.setHeader(YandexHeaders.SERVICE, service);
            nextRequest.setHeader(YandexHeaders.ZOO_SHARD_ID, Long.toString(prefix));
            next.handle(nextRequest, new OverridenRequestExchange(exchange, nextRequest), context);
        } catch (IOException e) {
            throw new ServiceUnavailableException(e);
        }
    }

    private static class OverridenRequestExchange extends FilterHttpAsyncExchange
    {
        private final HttpRequest request;

        OverridenRequestExchange(final HttpAsyncExchange exchange, final HttpRequest request) {
            super(exchange);
            this.request = request;
        }

        @Override
        public HttpRequest getRequest() {
            return request;
        }
    }
}

