package ru.yandex.mail.so.spampkin;

import java.util.List;
import java.util.logging.Logger;

import org.apache.http.concurrent.FutureCallback;

import ru.yandex.client.so.shingler.MassShingleResult;
import ru.yandex.client.so.shingler.MassShingleStats;
import ru.yandex.client.so.shingler.MassShinglerClient;
import ru.yandex.client.so.shingler.MassShinglerResult;
import ru.yandex.client.so.shingler.ShingleException;
import ru.yandex.client.so.shingler.ShingleType;
import ru.yandex.client.so.shingler.ShinglerClientException;
import ru.yandex.client.so.shingler.Shingles;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.mail.so.api.v1.Email;
import ru.yandex.mail.so.api.v1.EmailInfo;
import ru.yandex.parser.mail.envelope.SmtpEnvelopeHolder;

public class ShinglesFastChecker implements SpampkinFastChecker {
    private static final String AFTER_SHINGLE = "After processing shingle ";
    private static final double FROM_ADDR_RATIO1 = 0.2d;
    private static final double FROM_ADDR_RATIO2 = 0.01d;
    private static final long FROM_ADDR_LIM1 = 1000L;
    private static final long FROM_ADDR_LIM2 = 200L;
    private static final long FROM_ADDR_SPAM_LIM = 65500L;
    private static final long FROM_ADDR_TO_SUID_LIM1 = 1000L;
    private static final long FROM_ADDR_TO_SUID_LIM2 = 600L;
    private static final long FROM_ADDR_TO_SUID_SPAM_LIM1 = 50000L;
    private static final long FROM_ADDR_TO_SUID_SPAM_LIM2 = 10000L;
    private static final long FROM_ADDR_TO_SUID_MALIC_LIM = 10000L;

    private final MassShinglerClient client;

    public ShinglesFastChecker(final MassShinglerClient client) {
        this.client = client;
    }

    @Override
    public void check(
        final SmtpEnvelopeHolder envelope,
        final ProxySession session,
        final FutureCallback<SoResolution> callback)
    {
        Shingles shingles = new Shingles();
        Email from = envelope.from();
        if (from != null) {
            shingles.add(
                ShingleType.FROM_ADDR_SHINGLE,
                from.getNormalizedEmail());
        }
        String ip = envelope.ipString();
        if (ip != null) {
            StringBuilder sb = new StringBuilder();
            for (EmailInfo recipient: envelope.recipients()) {
                if (recipient.hasSuid()) {
                    sb.setLength(0);
                    sb.append(ip);
                    sb.append('_');
                    sb.append(recipient.getSuid().getValue());
                    shingles.add(
                        ShingleType.FROM_ADDR_TO_SUID_SHINGLE,
                        sb.toString());
                }
            }
        }
        MassShinglerClient client = this.client.adjust(session.context());
        try {
            client.getShingles(
                    shingles,
                    session.listener().createContextGeneratorFor(client),
                    new Callback(callback, session.logger()));
        } catch (ShinglerClientException | ShingleException e) {
            /*impossible*/
        }
    }

    private static boolean checkFromAddr(final MassShingleStats stats) {
        return
            (stats.allSpam() > FROM_ADDR_LIM1
                && stats.hamToAllSpam() < FROM_ADDR_RATIO1)
            || (stats.allSpam() > FROM_ADDR_LIM2
                && stats.hamToAllSpam() < FROM_ADDR_RATIO2)
            || stats.allSpam() > FROM_ADDR_SPAM_LIM;
    }

    private static boolean checkFromAddrToSuid(final MassShingleStats stats) {
        return
            (stats.allSpam() > FROM_ADDR_TO_SUID_LIM1
                && stats.hamToAllSpam() < FROM_ADDR_RATIO1)
            || (stats.allSpam() > FROM_ADDR_TO_SUID_LIM2
                && stats.hamToAllSpam() < FROM_ADDR_RATIO2)
            || stats.allSpam() > FROM_ADDR_TO_SUID_SPAM_LIM1;
    }

    private static boolean checkFromAddrToSuidAlone(final MassShingleStats stats) {
        return stats.allSpam() > FROM_ADDR_TO_SUID_SPAM_LIM2
            && stats.ham() < 2;
    }

    private static boolean checkFromAddrToSuidMalic(final MassShingleStats stats) {
        return stats.malic() > FROM_ADDR_TO_SUID_MALIC_LIM
            && stats.ham() < 2
            && stats.spam() < 1;
    }

    private static class Callback
        extends AbstractFilterFutureCallback<MassShinglerResult, SoResolution>
    {
        private final Logger logger;

        Callback(
            final FutureCallback<SoResolution> callback,
            final Logger logger)
        {
            super(callback);
            this.logger = logger;
        }

        @Override
        public void completed(final MassShinglerResult result) {
            logger.fine(result.toString());
            boolean fromAddrFull = false;
            boolean fromAddrToSuidFull = false;
            boolean fromAddrToSuidAlone = false;
            boolean fromAddrToSuidMalic = false;
            List<MassShingleResult> shingles =
                result.shinglesFor(ShingleType.FROM_ADDR_SHINGLE);
            if (!shingles.isEmpty()) {
                MassShingleResult shingle = shingles.get(0);
                MassShingleStats stats = shingle.shingleInfo().todayStats();
                if (stats != null) {
                    fromAddrFull = checkFromAddr(stats);
                    logger.fine(
                        AFTER_SHINGLE + shingle
                        + ", from addr full is " + fromAddrFull);
                }
            }
            shingles =
                result.shinglesFor(ShingleType.FROM_ADDR_TO_SUID_SHINGLE);
            for (MassShingleResult shingle: shingles) {
                MassShingleStats stats = shingle.shingleInfo().todayStats();
                if (!fromAddrToSuidFull) {
                    fromAddrToSuidFull = checkFromAddrToSuid(stats);
                    logger.fine(
                        AFTER_SHINGLE + shingle
                        + ", from addr to suid full is " + fromAddrToSuidFull);
                }
                if (!fromAddrToSuidAlone) {
                    fromAddrToSuidAlone = checkFromAddrToSuidAlone(stats);
                    logger.fine(
                        AFTER_SHINGLE + shingle
                        + ", from addr to suid alone is "
                        + fromAddrToSuidAlone);
                }
                if (!fromAddrToSuidMalic) {
                    fromAddrToSuidMalic = checkFromAddrToSuidMalic(stats);
                    logger.fine(
                        AFTER_SHINGLE + shingle
                        + ", from addr to suid malic is "
                        + fromAddrToSuidMalic);
                }
            }
            logger.info(
                "Total shingles result: " + fromAddrFull
                + ' ' + fromAddrToSuidFull
                + ' ' + fromAddrToSuidAlone
                + ' ' + fromAddrToSuidMalic);
            SoResolution resolution;
            if (fromAddrToSuidMalic) {
                resolution = SoResolution.REJECT;
            } else if (fromAddrToSuidAlone
                || (fromAddrToSuidFull && fromAddrFull))
            {
                resolution = SoResolution.SPAM;
            } else {
                resolution = SoResolution.UNKNOWN;
            }
            callback.completed(resolution);
        }
    }
}

