package ru.yandex.mail.so.spampkin;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.function.Consumer;
import java.util.logging.Logger;

import org.apache.http.HttpException;
import org.apache.http.HttpRequest;
import org.apache.http.HttpStatus;
import org.apache.http.entity.ContentType;
import org.apache.http.nio.entity.NStringEntity;
import org.apache.http.nio.protocol.HttpAsyncExchange;
import org.apache.http.nio.protocol.HttpAsyncRequestHandler;
import org.apache.http.protocol.HttpContext;

import ru.yandex.client.so.shingler.MassShinglerClient;
import ru.yandex.collection.Pattern;
import ru.yandex.concurrent.TimeFrameQueue;
import ru.yandex.http.config.ImmutableHttpHostConfig;
import ru.yandex.http.proxy.AbstractProxySessionCallback;
import ru.yandex.http.proxy.BasicProxySession;
import ru.yandex.http.proxy.HttpProxy;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.nio.FakeAsyncConsumer;
import ru.yandex.parser.mail.envelope.SmtpEnvelopeHolder;
import ru.yandex.parser.mail.errors.ErrorInfo;
import ru.yandex.stater.EnumStaterFactory;
import ru.yandex.stater.PassiveStaterAdapter;
import ru.yandex.util.ip.CidrSet;

public class Spampkin
    extends HttpProxy<ImmutableSpampkinConfig>
    implements HttpAsyncRequestHandler<HttpRequest>
{
    private static final String RESPONSE_PREFIX = "OK ";
    private static final String AMMM = "_ammm";

    private final SpampkinFastChecker checker;
    private final MassShinglerClient shinglerClient;

    public Spampkin(final ImmutableSpampkinConfig config)
        throws IOException
    {
        super(config);
        shinglerClient = registerClient(
            "MassShinglerClient",
            new MassShinglerClient(reactor, config.shinglerConfig()),
            config.shinglerConfig());
        List<SpampkinFastChecker> checkers = new ArrayList<>();
        TimeFrameQueue<SoResolution> resolutions =
            new TimeFrameQueue<>(config.metricsTimeFrame());
        if (!config.outMail()) {
            SpampkinFastChecker connectChecker =
                new ConnectFastChecker(
                    CidrSet.INSTANCE.load(config.intranetIps().file()));
            TimeFrameQueue<SoResolution> connectResolutions =
                new TimeFrameQueue<>(config.metricsTimeFrame());
            checkers.add(
                new StatingFastChecker(connectChecker, connectResolutions));
            registerStater(
                new PassiveStaterAdapter<>(
                    connectResolutions,
                    new EnumStaterFactory<>(
                        type -> "connect-resolution-" + type + AMMM,
                        SoResolution.values())));
        }

        SpampkinFastChecker shinglesChecker =
            new ShinglesFastChecker(shinglerClient);
        TimeFrameQueue<SoResolution> shinglesResolutions =
            new TimeFrameQueue<>(config.metricsTimeFrame());
        checkers.add(
            new StatingFastChecker(shinglesChecker, shinglesResolutions));
        registerStater(
            new PassiveStaterAdapter<>(
                shinglesResolutions,
                new EnumStaterFactory<>(
                    type -> "shingles-resolution-" + type + AMMM,
                    SoResolution.values())));

        ImmutableHttpHostConfig jrbldConfig = config.jrbldConfig();
        SpampkinFastChecker jrbldChecker =
            new JrbldFastChecker(
                client("JRBLD", jrbldConfig),
                jrbldConfig.host(),
                config);
        TimeFrameQueue<SoResolution> jrbldResolutions =
            new TimeFrameQueue<>(config.metricsTimeFrame());
        checkers.add(
            new StatingFastChecker(jrbldChecker, jrbldResolutions));
        registerStater(
            new PassiveStaterAdapter<>(
                jrbldResolutions,
                new EnumStaterFactory<>(
                    type -> "jrbld-resolution-" + type + AMMM,
                    SoResolution.values())));

        checker = new StatingFastChecker(
            new MultiFastChecker(checkers),
            resolutions);
        register(new Pattern<>("/antispam", false), this);
        registerStater(
            new PassiveStaterAdapter<>(
                resolutions,
                new EnumStaterFactory<>(
                    type -> "resolution-" + type + AMMM,
                    SoResolution.values())));
    }

    @Override
    public FakeAsyncConsumer<HttpRequest> processRequest(
        final HttpRequest request,
        final HttpContext context)
    {
        return new FakeAsyncConsumer<>(request);
    }

    @Override
    public void handle(
        final HttpRequest request,
        final HttpAsyncExchange exchange,
        final HttpContext context)
        throws HttpException, IOException
    {
        ProxySession session = new BasicProxySession(this, exchange, context);
        SmtpEnvelopeHolder envelope = new SmtpEnvelopeHolder(
            new ErrorsLogger(session.logger()),
            session.params());
        checker.check(envelope, session, new Callback(session));
    }

    private static class ErrorsLogger implements Consumer<ErrorInfo> {
        private final Logger logger;

        ErrorsLogger(final Logger logger) {
            this.logger = logger;
        }

        @Override
        public void accept(final ErrorInfo errorInfo) {
            logger.warning(errorInfo.toString());
        }
    }

    private static class Callback
        extends AbstractProxySessionCallback<SoResolution>
    {
        Callback(final ProxySession session) {
            super(session);
        }

        @Override
        public void completed(final SoResolution resolution) {
            session.connection().setSessionInfo(
                SpampkinConfigDefaults.SO_RESOLUTION,
                resolution.toString());
            session.response(
                HttpStatus.SC_OK,
                new NStringEntity(
                    RESPONSE_PREFIX + resolution,
                    ContentType.TEXT_PLAIN.withCharset(
                        session.acceptedCharset())));
        }
    }
}

