package ru.yandex.tma;

import java.util.ArrayList;
import java.util.List;
import java.util.logging.Level;

import io.vertx.sqlclient.Row;
import io.vertx.sqlclient.RowSet;
import io.vertx.sqlclient.Tuple;
import org.apache.http.HttpEntity;
import org.apache.http.HttpEntityEnclosingRequest;
import org.apache.http.HttpException;
import org.apache.http.HttpRequest;
import org.apache.http.HttpStatus;
import org.apache.http.nio.protocol.HttpAsyncExchange;
import org.apache.http.nio.protocol.HttpAsyncRequestConsumer;
import org.apache.http.nio.protocol.HttpAsyncRequestHandler;
import org.apache.http.protocol.HttpContext;

import ru.yandex.client.pg.SqlQuery;
import ru.yandex.http.proxy.AbstractProxySessionCallback;
import ru.yandex.http.proxy.BasicProxySession;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.ServiceUnavailableException;
import ru.yandex.http.util.nio.EmptyAsyncConsumer;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumer;
import ru.yandex.json.dom.BasicContainerFactory;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonNull;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.parser.StringCollectorsFactory;
import ru.yandex.json.writer.JsonType;

public class DeliveryReportsHandler
    implements HttpAsyncRequestHandler<JsonObject>
{
    private static final SqlQuery UPDATE_MESSAGE_STATE =
        new SqlQuery("update-message-state.sql", QueueProcessor.class);
    private static final String[] ACCEPTED_SENT =
        new String[]{"accepted", "sent"};
    private static final String[] ACCEPTED_SENT_DELIVERED =
        new String[]{"accepted", "sent", "delivered"};

    private final TmaServer tmaServer;

    public DeliveryReportsHandler(final TmaServer tmaServer) {
        this.tmaServer = tmaServer;
    }

    @Override
    public HttpAsyncRequestConsumer<JsonObject> processRequest(
        final HttpRequest request,
        final HttpContext context)
        throws HttpException
    {
        if (request instanceof HttpEntityEnclosingRequest) {
            HttpEntity entity =
                ((HttpEntityEnclosingRequest) request).getEntity();
            if (entity.getContentLength() != 0) {
                return new JsonAsyncTypesafeDomConsumer(
                    entity,
                    StringCollectorsFactory.INSTANCE,
                    BasicContainerFactory.INSTANCE);
            }
        }
        return new EmptyAsyncConsumer<>(JsonNull.INSTANCE);
    }

    @Override
    public void handle(
        final JsonObject payload,
        final HttpAsyncExchange exchange,
        final HttpContext context)
        throws HttpException
    {
        ProxySession session =
            new BasicProxySession(tmaServer, exchange, context);
        session.logger().info(
            "Got delivery reports: " + JsonType.NORMAL.toString(payload));
        List<Tuple> tuples = new ArrayList<>();
        try {
            JsonList reports = payload.get("reports").asList();
            int size = reports.size();
            for (int i = 0; i < size; ++i) {
                JsonMap report = reports.get(i).asMap();
                String[] updateableStates;
                String state = report.getString("status");
                switch (state) {
                    case "delivered":
                    case "error":
                        updateableStates = ACCEPTED_SENT;
                        break;
                    case "seen":
                        updateableStates = ACCEPTED_SENT_DELIVERED;
                        break;
                    default:
                        updateableStates = null;
                }
                if (updateableStates == null) {
                    session.logger().info(
                        "Skipping report in state " + state
                        + ':' + ' ' + JsonType.NORMAL.toString(report));
                } else {
                    JsonMap serviceData =
                        report.get("service_data").asMapOrNull();
                    if (serviceData == null) {
                        session.logger().info(
                            "Skipping report without service_data: "
                            + JsonType.NORMAL.toString(report));
                    } else {
                        Tuple tuple = Tuple.tuple();
                        tuple.addLong(serviceData.get("message_id").asLong());
                        tuple.addString(state);
                        tuple.addArrayOfString(updateableStates);
                        tuples.add(tuple);
                    }
                }
            }
        } catch (JsonException e) {
            session.logger().log(
                Level.WARNING,
                "Failed to process request: "
                + JsonType.NORMAL.toString(payload),
                e);
            throw new BadRequestException(e);
        }

        if (tuples.isEmpty()) {
            session.response(HttpStatus.SC_OK);
        } else {
            tmaServer.pgClient().executeBatchOnMaster(
                UPDATE_MESSAGE_STATE,
                tuples,
                session.listener(),
                new Callback(session, tmaServer));
        }
    }

    private static class Callback
        extends AbstractProxySessionCallback<RowSet<Row>>
    {
        private final TmaServer tmaServer;

        Callback(final ProxySession session, final TmaServer tmaServer) {
            super(session);
            this.tmaServer = tmaServer;
        }

        @Override
        public void completed(RowSet<Row> rowSet) {
            List<Long> updatedIds = new ArrayList<>();
            try {
                while (rowSet != null) {
                    for (Row row: rowSet) {
                        updatedIds.add(row.getLong("id"));
                    }
                    rowSet = rowSet.next();
                }
            } catch (RuntimeException e) {
                session.handleException(new ServiceUnavailableException(e));
                return;
            }
            tmaServer.wakeupQueueProcessor();
            session.logger().info("Messages updated: " + updatedIds);
            session.response(HttpStatus.SC_OK);
        }
    }
}

