package ru.yandex.tma;

import java.io.IOException;
import java.time.LocalDateTime;
import java.time.ZoneOffset;
import java.util.Date;
import java.util.concurrent.ExecutionException;
import java.util.logging.Level;

import io.vertx.sqlclient.Row;
import io.vertx.sqlclient.RowSet;
import io.vertx.sqlclient.Tuple;
import org.apache.http.entity.ContentType;
import org.jsmpp.util.DeliveryReceiptState;

import ru.yandex.client.pg.SqlQuery;
import ru.yandex.function.GenericAutoCloseable;
import ru.yandex.http.util.EmptyFutureCallback;
import ru.yandex.http.util.RequestErrorType;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.EmptyAsyncConsumerFactory;
import ru.yandex.http.util.nio.client.BasicRequestsListener;
import ru.yandex.http.util.nio.client.RequestsListener;
import ru.yandex.http.util.server.SessionContext;
import ru.yandex.io.StringBuilderWriter;
import ru.yandex.json.writer.JsonWriter;
import ru.yandex.logger.PrefixedLogger;
import ru.yandex.tma.config.ImmutableTmaConfig;

public class QueueProcessor
    implements GenericAutoCloseable<IOException>, Runnable
{
    private static final SqlQuery FIND_NEXT_TASK =
        new SqlQuery("find-next-task.sql", QueueProcessor.class);
    private static final SqlQuery MARK_MESSAGE_SENT =
        new SqlQuery("mark-message-sent.sql", QueueProcessor.class);
    private static final SqlQuery MARK_MESSAGE_FAILED =
        new SqlQuery("mark-message-failed.sql", QueueProcessor.class);
    private static final SqlQuery MARK_MESSAGE_DONE =
        new SqlQuery("mark-message-done.sql", QueueProcessor.class);
    private static final String[] DELIVERED_ARRAY =
        new String[]{"delivered", "seen"};
    private static final String[] ERROR_ARRAY = new String[]{"error"};
    private static final String[] FAILED_ARRAY = new String[]{"failed"};

    private final TmaServer tmaServer;
    private final Thread thread;
    private final PrefixedLogger logger;
    private volatile boolean stopped = false;

    public QueueProcessor(
        final TmaServer tmaServer,
        final ThreadGroup threadGroup)
    {
        this.tmaServer = tmaServer;
        ImmutableTmaConfig config = tmaServer.config();
        String name = config.name() + "-QueueProcessor";
        thread = new Thread(threadGroup, this, name);
        thread.setDaemon(true);
        logger = tmaServer.logger().addPrefix(name);
    }

    public synchronized void wakeup() {
        notify();
    }

    private synchronized void wakeMeUpWhenSeptemberEnds() {
        try {
            wait(tmaServer.config().dbScanInterval());
        } catch (InterruptedException e) {
        }
    }

    public void start() {
        thread.start();
    }

    @Override
    public void close() throws IOException {
        logger.info("Termination started");
        try {
            stopped = true;
            thread.interrupt();
            try {
                thread.join();
            } catch (InterruptedException e) {
            }
        } finally {
            logger.info("Termination completed");
        }
    }

    @Override
    public void run() {
        while (!stopped) {
            logger.fine("Looking for pending tasks");
            RequestsListener listener = new BasicRequestsListener();
            int rowCount = 0;
            Long id = null;
            String state = null;
            boolean failed = true;
            PrefixedLogger logger = this.logger;
            try {
                Tuple tuple = Tuple.tuple();
                tuple.addString(SessionContext.HOSTNAME);
                tuple.addString(tmaServer.workerId());
                tuple.addLong(tmaServer.config().deliveryTtl());
                RowSet<Row> rowSet =
                    tmaServer.pgClient().executeOnMaster(
                        FIND_NEXT_TASK,
                        tuple,
                        listener,
                        EmptyFutureCallback.INSTANCE)
                        .get();
                rowCount = rowSet.rowCount();
                logger.fine("Tasks received: " + rowCount);
                if (rowCount == 0) {
                    wakeMeUpWhenSeptemberEnds();
                } else {
                    Row row = rowSet.iterator().next();
                    id = row.getLong("id");
                    logger.addPrefix("msg_" + id);
                    state = row.getString("state");
                    logger.info("Processing message in state " + state);
                    switch (state) {
                        case "accepted":
                            processInitialMessage(id, row, listener, logger);
                            break;
                        case "delivered":
                        case "seen":
                            processDeliveredMessage(
                                id,
                                state,
                                row,
                                listener,
                                logger);
                            break;
                        case "error":
                            processErrorMessage(
                                id,
                                row,
                                listener,
                                logger);
                            break;
                        case "failed":
                            processFailedMessage(
                                id,
                                row,
                                listener,
                                logger);
                            break;
                        default:
                            // TODO: stater
                            logger.warning(
                                "Message " + id
                                + " has unexpected state " +  state);
                            break;
                    }
                }
                failed = false;
            } catch (RuntimeException e) {
                // TODO: stater
                logger.log(Level.WARNING, "Task processing failed", e);
            } catch (ExecutionException e) {
                // TODO: stater
                logger.log(Level.WARNING, "Database query failed", e);
            } catch (IOException e) {
                // TODO: stater
                logger.log(Level.WARNING, "SMPP request failed", e);
            } catch (InterruptedException e) {
                // TODO: stater
                logger.log(Level.WARNING, "Task interrupted", e);
            }
            if (rowCount > 0) {
                logger.info(
                    "For message " + id + " in state " + state
                    + " upstream status: " + listener);
            }
            if (failed) {
                logger.info("Failed requests details:\n" + listener.details());
                try {
                    Thread.sleep(1000L);
                } catch (InterruptedException e) {
                }
            }
        }
    }

    private void processInitialMessage(
        final Long id,
        final Row row,
        final RequestsListener listener,
        final PrefixedLogger logger)
        throws ExecutionException, IOException, InterruptedException
    {
        String phoneNumber = tmaServer.decrypt(row.getString("phone_number"));
        String templateId = row.getString("message_template_id");
        String templateLocale = row.getString("message_template_locale");
        String[] templateParameters =
            row.getArrayOfStrings("message_template_parameters");
        for (int i = 0; i < templateParameters.length; ++i) {
            templateParameters[i] = tmaServer.decrypt(templateParameters[i]);
        }
        logger.fine(
            "Message template id: " + templateId
            + ", locale: " + templateLocale);
        StringBuilderWriter sbw = new StringBuilderWriter();
        try (JsonWriter writer = new JsonWriter(sbw)) {
            writer.startObject();
            writer.key("service");
            writer.value(tmaServer.config().messengerService());
            writer.key("account");
            writer.value(tmaServer.config().messengerAccount());
            writer.key("service_message_id");
            writer.value(id.toString());

            writer.key("service_data");
            writer.startObject();
            writer.key("message_id");
            writer.value(id.longValue());
            writer.key("worker_id");
            writer.value(tmaServer.workerId());
            writer.endObject();

            writer.key("client");
            writer.startObject();
            writer.key("phone");
            writer.value(phoneNumber);
            writer.endObject();

            writer.key("payload");
            writer.startObject();
            writer.key("type");
            writer.value("template");
            writer.key("locale");
            writer.value(templateLocale);
            writer.key("template");
            writer.startObject();
            writer.key("name");
            writer.value(templateId);
            writer.key("params");
            writer.startArray();
            for (String parameter: templateParameters) {
                writer.value(parameter);
            }
            writer.endArray();
            writer.endObject();
            writer.endObject();

            writer.key("ttl");
            writer.value(tmaServer.config().deliveryTtl() / 1000L);

            writer.endObject();
        }

        BasicAsyncRequestProducerGenerator producerGenerator =
            new BasicAsyncRequestProducerGenerator(
                "/v1/send",
                sbw.toString(),
                ContentType.APPLICATION_JSON);
        producerGenerator.addHeader("X-Idempotency-Token", id.toString());
        try {
            tmaServer.messengerGatewayClient().execute(
                tmaServer.config().messengerGatewayConfig().host(),
                producerGenerator,
                EmptyAsyncConsumerFactory.ANY_GOOD,
                listener.createContextGeneratorFor(
                    tmaServer.messengerGatewayClient()),
                EmptyFutureCallback.INSTANCE)
                .get();
        } catch (ExecutionException e) {
            logger.log(Level.WARNING, "Message send failed", e);
            Throwable cause = e.getCause();
            if (e instanceof Exception
                && RequestErrorType.ERROR_CLASSIFIER.apply((Exception) cause)
                    == RequestErrorType.NON_RETRIABLE)
            {
                logger.info(
                    "Non retriable error encountered, mark message as failed");
                try {
                    RowSet<Row> rowSet =
                        tmaServer.pgClient().executeOnMaster(
                            MARK_MESSAGE_FAILED,
                            Tuple.of(id),
                            listener,
                            EmptyFutureCallback.INSTANCE)
                            .get();
                    logger.info(
                        "Message marked as failed, row count = "
                        + rowSet.rowCount());
                    return;
                } catch (Throwable t) {
                    e.addSuppressed(t);
                    throw e;
                }
            } else {
                throw e;
            }
        }

        logger.info("Message sent");

        RowSet<Row> rowSet =
            tmaServer.pgClient().executeOnMaster(
                MARK_MESSAGE_SENT,
                Tuple.of(id),
                listener,
                EmptyFutureCallback.INSTANCE)
                .get();
        logger.info(
            "Message marked as sent, row count = " + rowSet.rowCount());
    }

    private void processDeliveredMessage(
        final Long id,
        final String state,
        final Row row,
        final RequestsListener listener,
        final PrefixedLogger logger)
        throws ExecutionException, IOException, InterruptedException
    {
        sendDeliveryReport(
            id,
            state,
            DELIVERED_ARRAY,
            DeliveryReceiptState.DELIVRD,
            row,
            listener,
            logger);
    }

    private void processErrorMessage(
        final Long id,
        final Row row,
        final RequestsListener listener,
        final PrefixedLogger logger)
        throws ExecutionException, IOException, InterruptedException
    {
        sendDeliveryReport(
            id,
            "error",
            ERROR_ARRAY,
            DeliveryReceiptState.UNDELIV,
            row,
            listener,
            logger);
    }

    private void processFailedMessage(
        final Long id,
        final Row row,
        final RequestsListener listener,
        final PrefixedLogger logger)
        throws ExecutionException, IOException, InterruptedException
    {
        sendDeliveryReport(
            id,
            "failed",
            FAILED_ARRAY,
            DeliveryReceiptState.UNDELIV,
            row,
            listener,
            logger);
    }

    private void sendDeliveryReport(
        final Long id,
        final String state,
        final String[] updateableStates,
        final DeliveryReceiptState deliveryStatus,
        final Row row,
        final RequestsListener listener,
        final PrefixedLogger logger)
        throws ExecutionException, IOException, InterruptedException
    {
        if (!tmaServer.smppServer().sendDeliveryReport(
            id,
            deliveryStatus,
            Date.from(
                LocalDateTime.from(
                    row.getTemporal("accepted_timestamp"))
                    .toInstant(ZoneOffset.of("Z"))),
            Date.from(
                LocalDateTime.from(
                    row.getTemporal("delivered_timestamp"))
                    .toInstant(ZoneOffset.of("Z"))),
            logger))
        {
            logger.warning(
                "Message session lost in state " + state
                + ", update it anyway");
        }
        Tuple tuple = Tuple.tuple();
        tuple.addLong(id);
        tuple.addArrayOfString(updateableStates);
        RowSet<Row> rowSet =
            tmaServer.pgClient().executeOnMaster(
                MARK_MESSAGE_DONE,
                tuple,
                listener,
                EmptyFutureCallback.INSTANCE)
                .get();
        logger.info(
            "Message marked as done, row count = " + rowSet.rowCount());
    }
}

