package ru.yandex.tma;

import java.io.IOException;
import java.net.InetAddress;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.security.SecureRandom;
import java.time.Instant;
import java.time.temporal.Temporal;
import java.util.Arrays;
import java.util.Date;
import java.util.EnumMap;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.net.ServerSocketFactory;
import javax.net.ssl.SSLContext;

import io.vertx.sqlclient.Row;
import io.vertx.sqlclient.RowSet;
import io.vertx.sqlclient.Tuple;
import org.jsmpp.InvalidResponseException;
import org.jsmpp.PDUException;
import org.jsmpp.PDUStringException;
import org.jsmpp.SMPPConstant;
import org.jsmpp.bean.Alphabet;
import org.jsmpp.bean.BindType;
import org.jsmpp.bean.BroadcastSm;
import org.jsmpp.bean.CancelBroadcastSm;
import org.jsmpp.bean.CancelSm;
import org.jsmpp.bean.DataCodings;
import org.jsmpp.bean.DataSm;
import org.jsmpp.bean.DeliveryReceipt;
import org.jsmpp.bean.ESMClass;
import org.jsmpp.bean.GSMSpecificFeature;
import org.jsmpp.bean.InterfaceVersion;
import org.jsmpp.bean.MessageMode;
import org.jsmpp.bean.MessageState;
import org.jsmpp.bean.MessageType;
import org.jsmpp.bean.NumberingPlanIndicator;
import org.jsmpp.bean.OptionalParameter;
import org.jsmpp.bean.QueryBroadcastSm;
import org.jsmpp.bean.QuerySm;
import org.jsmpp.bean.RegisteredDelivery;
import org.jsmpp.bean.ReplaceSm;
import org.jsmpp.bean.SubmitMulti;
import org.jsmpp.bean.SubmitSm;
import org.jsmpp.bean.TypeOfNumber;
import org.jsmpp.extra.NegativeResponseException;
import org.jsmpp.extra.ProcessRequestException;
import org.jsmpp.extra.ResponseTimeoutException;
import org.jsmpp.session.BindRequest;
import org.jsmpp.session.BroadcastSmResult;
import org.jsmpp.session.DataSmResult;
import org.jsmpp.session.QueryBroadcastSmResult;
import org.jsmpp.session.QuerySmResult;
import org.jsmpp.session.SMPPServerSession;
import org.jsmpp.session.SMPPServerSessionListener;
import org.jsmpp.session.ServerMessageReceiverListener;
import org.jsmpp.session.Session;
import org.jsmpp.session.SubmitMultiResult;
import org.jsmpp.session.SubmitSmResult;
import org.jsmpp.util.AbsoluteTimeFormatter;
import org.jsmpp.util.DeliveryReceiptState;
import org.jsmpp.util.MessageId;

import ru.yandex.client.pg.SqlQuery;
import ru.yandex.concurrent.ExecutorServiceCloser;
import ru.yandex.concurrent.LifoWaitBlockingQueue;
import ru.yandex.concurrent.NamedThreadFactory;
import ru.yandex.function.GenericAutoCloseable;
import ru.yandex.function.GenericAutoCloseableChain;
import ru.yandex.http.util.EmptyFutureCallback;
import ru.yandex.http.util.nio.client.BasicRequestsListener;
import ru.yandex.http.util.nio.client.EmptyRequestsListener;
import ru.yandex.http.util.nio.client.RequestsListener;
import ru.yandex.http.util.server.SessionContext;
import ru.yandex.logger.HashMapLookup;
import ru.yandex.logger.IdGenerator;
import ru.yandex.logger.PrefixedLogger;
import ru.yandex.tma.config.ImmutableTemplateConfig;
import ru.yandex.tma.config.ImmutableTmaConfig;
import ru.yandex.util.string.StringUtils;
import ru.yandex.util.timesource.TimeSource;

public class SmppServer
    implements
        GenericAutoCloseable<IOException>,
        Runnable,
        ServerMessageReceiverListener
{
    private static final SqlQuery ADD_MESSAGE =
        new SqlQuery("add-message.sql", SmppServer.class);
    private static final SqlQuery MESSAGE_STATE =
        new SqlQuery("message-state.sql", SmppServer.class);
    private static final OptionalParameter[] EMPTY_PARAMETERS =
        new OptionalParameter[0];
    private static final Map<Alphabet, Charset> ALPHABET_TO_CHARSET =
        new EnumMap<>(
            Map.of(
                Alphabet.ALPHA_DEFAULT, Charset.forName("GSM7"),
                Alphabet.ALPHA_LATIN1, StandardCharsets.ISO_8859_1,
                Alphabet.ALPHA_UCS2, StandardCharsets.UTF_16BE));

    private final GenericAutoCloseableChain<IOException> closeChain =
        new GenericAutoCloseableChain<>();
    private final IdGenerator idGenerator = new IdGenerator(
        new SecureRandom().nextLong(),
        SessionContext.SESSION_ID_LENGTH,
        IdGenerator.MAX_RADIX);
    private final Map<Long, SubmitSession> sessions =
        new ConcurrentHashMap<>();
    private final AbsoluteTimeFormatter timeFormatter =
        new AbsoluteTimeFormatter();
    private final TmaServer tmaServer;
    private final String name;
    private final PrefixedLogger logger;
    private final Logger smppAccessLogger;
    private final Thread thread;
    private final ThreadPoolExecutor threadPool;
    private final QueueProcessor queueProcessor;
    private final PortSavingServerSocketFactory portSavingServerSocketFactory;
    private volatile SMPPServerSessionListener listener = null;
    private volatile boolean stopped = false;
    private volatile IOException e = null;

    public SmppServer(final TmaServer tmaServer) {
        this.tmaServer = tmaServer;
        ImmutableTmaConfig config = tmaServer.config();
        name = config.name() + "-SMPP";
        logger = tmaServer.logger().addPrefix(name);
        smppAccessLogger = tmaServer.smppAccessLogger();
        ThreadGroup threadGroup =
            new ThreadGroup(tmaServer.getThreadGroup(), name);
        thread = new Thread(threadGroup, this, name);
        thread.setDaemon(true);

        threadPool = new ThreadPoolExecutor(
            config.workers(),
            config.workers(),
            1,
            TimeUnit.DAYS,
            new LifoWaitBlockingQueue<>(config.connections()),
            new NamedThreadFactory(threadGroup, name + '-', true));
        closeChain.add(
            new ExecutorServiceCloser(threadPool, config.timeout()));

        queueProcessor = new QueueProcessor(tmaServer, threadGroup);
        closeChain.add(queueProcessor);

        SSLContext sslContext = tmaServer.sslContext();
        if (sslContext == null) {
            portSavingServerSocketFactory =
                new PortSavingServerSocketFactory(
                    ServerSocketFactory.getDefault());
        } else {
            portSavingServerSocketFactory =
                new PortSavingServerSocketFactory(
                    sslContext.getServerSocketFactory());
        }
    }

    public int port() {
        return portSavingServerSocketFactory.lastPort();
    }

    public void start() {
        threadPool.prestartAllCoreThreads();
        queueProcessor.start();
        thread.start();
    }

    private String generateSessionId() {
        return SessionContext.INSTANCE_ID + idGenerator.next();
    }

    @Override
    @SuppressWarnings("try")
    public void close() throws IOException {
        logger.info("Termination started");
        try (GenericAutoCloseableChain<IOException> guard = closeChain) {
            stopped = true;
            thread.interrupt();
            SMPPServerSessionListener listener = this.listener;
            if (listener != null) {
                listener.close();
            }
            try {
                thread.join();
            } catch (InterruptedException e) {
            }
            if (e != null) {
                throw e;
            }
        } finally {
            logger.info("Termination completed");
        }
    }

    @Override
    public void run() {
        boolean accepting = false;
        try (SMPPServerSessionListener listener =
                new SMPPServerSessionListener(
                    tmaServer.config().smppPort(),
                    new SimpleServerConnectionFactory(
                        portSavingServerSocketFactory)))
        {
            this.listener = listener;
            listener.setInitiationTimer(tmaServer.config().timeout());
            logger.info("SMPP server bound to port " + port());
            while (!stopped) {
                accepting = true;
                SMPPServerSession session = listener.accept();
                accepting = false;
                logger.info(
                    "Accepted SMPP session " + session.getSessionId()
                    + " from " + session.getInetAddress()
                    + ':' + session.getPort());
                session.setMessageReceiverListener(this);
                threadPool.execute(
                    new WaitBindTask(
                        session,
                        tmaServer.config(),
                        logger.addPrefix(session.getSessionId())));
            }
        } catch (IOException e) {
            if (!accepting) {
                this.e = e;
            }
        } finally {
            logger.info("SMPP server stopped");
        }
    }

    public boolean sendDeliveryReport(
        final Long id,
        final DeliveryReceiptState status,
        final Date submitDate,
        final Date doneDate,
        final PrefixedLogger logger)
        throws IOException
    {
        SubmitSession session = sessions.remove(id);
        if (session == null) {
            logger.info("No session found");
            return false;
        } else {
            try {
                DeliveryReceipt receipt =
                    new DeliveryReceipt(
                        id.toString(),
                        1,
                        1,
                        submitDate,
                        doneDate,
                        status,
                        "000",
                        id.toString());
                SubmitSm submitSm = session.submitSm;
                session.smppSession.deliverShortMessage(
                    "mc",
                    TypeOfNumber.valueOf(submitSm.getDestAddrTon()),
                    NumberingPlanIndicator.valueOf(submitSm.getDestAddrNpi()),
                    submitSm.getDestAddress(),
                    TypeOfNumber.valueOf(submitSm.getSourceAddrTon()),
                    NumberingPlanIndicator.valueOf(
                        submitSm.getSourceAddrNpi()),
                    submitSm.getSourceAddr(),
                    new ESMClass(
                        MessageMode.DEFAULT,
                        MessageType.SMSC_DEL_RECEIPT,
                        GSMSpecificFeature.DEFAULT),
                    (byte) 0,
                    (byte) 0,
                    new RegisteredDelivery(0),
                    DataCodings.ZERO,
                    receipt.toString().getBytes(StandardCharsets.US_ASCII));
                logger.info("Delivery receipt sent");
                return true;
            } catch (InvalidResponseException
                | NegativeResponseException
                | PDUException
                | ResponseTimeoutException e)
            {
                throw new IOException(e);
            }
        }
    }

    public void wakeupQueueProcessor() {
        queueProcessor.wakeup();
    }

    @Override
    public SubmitSmResult onAcceptSubmitSm(
        final SubmitSm submitSm,
        final SMPPServerSession session)
        throws ProcessRequestException
    {
        RequestsListener listener = new BasicRequestsListener();
        String destAddress = submitSm.getDestAddress();
        String request =
            "submit_sm " + destAddress
            + " ton:" + submitSm.getDestAddrTon()
            + "/npi:" + submitSm.getDestAddrNpi();
        try (Loggers loggers = new Loggers(request, listener, session)) {
            try {
                PrefixedLogger logger = loggers.logger;
                byte[] shortMessage = submitSm.getShortMessage();
                logger.fine(
                    "Message from " + submitSm.getSourceAddr()
                    + " ton:" + submitSm.getSourceAddrTon()
                    + "/npi:" + submitSm.getSourceAddrNpi()
                    + ", message length: " + shortMessage.length);
                if (destAddress.length() > 0 && destAddress.charAt(0) != '+') {
                    destAddress = '+' + destAddress;
                }
                if (submitSm.isUdhi()) {
                    shortMessage =
                        Arrays.copyOfRange(
                            shortMessage,
                            (shortMessage[0] & 0xff) + 1,
                            shortMessage.length);
                    logger.fine(
                        "UDH trimmed, message length become: "
                        + shortMessage.length);
                }
                Charset charset;
                try {
                    Alphabet alphabet =
                        Alphabet.parseDataCoding(submitSm.getDataCoding());
                    logger.fine("Message alphabet: " + alphabet);
                    charset = ALPHABET_TO_CHARSET.get(alphabet);
                    if (charset == null) {
                        throw new ProcessRequestException(
                            "Unsupported alphabet " + alphabet,
                            SMPPConstant.STAT_ESME_RSUBMITFAIL);
                    }
                } catch (RuntimeException e) {
                    throw new ProcessRequestException(
                        "Bad data coding " + (submitSm.getDataCoding() & 0xff),
                        SMPPConstant.STAT_ESME_RSUBMITFAIL,
                        e);
                }
                logger.fine("Message charset: " + charset);
                String text = new String(shortMessage, charset);
                String templateName = null;
                String templateLocale = null;
                String[] templateParameters = null;
                for (Map.Entry<String, ImmutableTemplateConfig> entry
                    : tmaServer.config().templates().entrySet())
                {
                    ImmutableTemplateConfig config = entry.getValue();
                    Pattern pattern = config.pattern();
                    Matcher matcher = pattern.matcher(text);
                    if (matcher.matches()) {
                        logger.fine("Template matched: " + entry.getKey());
                        templateName = config.name();
                        templateLocale = config.locale();
                        int groups = matcher.groupCount();
                        templateParameters = new String[groups];
                        for (int i = 1; i <= groups; ++i) {
                            templateParameters[i - 1] =
                                tmaServer.encrypt(matcher.group(i));
                        }
                        break;
                    }
                }

                if (templateName == null) {
                    throw new ProcessRequestException(
                        "No matching template found for message <"
                        + text.replaceAll("[0-9]", "*") + '>',
                        SMPPConstant.STAT_ESME_RSUBMITFAIL);
                }

                Tuple tuple = Tuple.tuple();
                tuple.addString(SessionContext.HOSTNAME);
                tuple.addString(tmaServer.workerId());
                tuple.addString(tmaServer.encrypt(destAddress));
                tuple.addString(templateName);
                tuple.addString(templateLocale);
                tuple.addArrayOfString(templateParameters);
                try {
                    RowSet<Row> rowSet =
                        tmaServer.pgClient().executeOnMaster(
                            ADD_MESSAGE,
                            tuple,
                            listener,
                            EmptyFutureCallback.INSTANCE)
                            .get();
                    Long id = rowSet.iterator().next().getLong("id");
                    logger.info("Created message id: " + id);
                    sessions.put(id, new SubmitSession(session, submitSm));
                    tmaServer.wakeupQueueProcessor();
                    loggers.status = "OK";
                    return new SubmitSmResult(
                        new MessageId(id.toString()),
                        EMPTY_PARAMETERS);
                } catch (Exception e) {
                    throw new ProcessRequestException(
                        "Database transaction failed",
                        SMPPConstant.STAT_ESME_RSUBMITFAIL,
                        e);
                }
            } catch (Throwable t) {
                loggers.error(t);
                throw t;
            }
        } catch (IOException | RuntimeException e) {
            logger.log(
                Level.WARNING,
                "onAcceptSubmitSm internal error:\n" + listener.details(),
                e);
            throw new ProcessRequestException(
                "Interal error",
                SMPPConstant.STAT_ESME_RSYSERR,
                e);
        } catch (Throwable t) {
            logger.log(
                Level.WARNING,
                "onAcceptSubmitSm failed:\n" + listener.details(),
                t);
            throw t;
        } finally {
            logger.info("onAcceptSubmitSm upstreams stats: " + listener);
        }
    }

    @Override
    public SubmitMultiResult onAcceptSubmitMulti(
        final SubmitMulti submitMulti,
        final SMPPServerSession session)
        throws ProcessRequestException
    {
        logger.warning("Rejecting submit_multi request");
        String request =
            StringUtils.join(
                Arrays.asList(submitMulti.getDestAddresses()),
                StringBuilder::append,
                ',',
                "submit_multi [",
                "] ?");
        try (Loggers loggers =
                new Loggers(request, EmptyRequestsListener.INSTANCE, session))
        {
            ProcessRequestException e = new ProcessRequestException(
                "submit_multi not implemented",
                SMPPConstant.STAT_ESME_RSYSERR);
            loggers.error(e);
            throw e;
        }
    }

    @Override
    public QuerySmResult onAcceptQuerySm(
        final QuerySm querySm,
        final SMPPServerSession session)
        throws ProcessRequestException
    {
        RequestsListener listener = new BasicRequestsListener();
        String request = "query_sm " + querySm.getMessageId() + " ?";
        try (Loggers loggers = new Loggers(request, listener, session)) {
            try {
                Long id = Long.valueOf(querySm.getMessageId());
                logger.info("Checking status for message id " + id);
                RowSet<Row> rowSet =
                    tmaServer.pgClient().executeOnMaster(
                        MESSAGE_STATE,
                        Tuple.of(id),
                        listener,
                        EmptyFutureCallback.INSTANCE)
                        .get();
                int rowCount = rowSet.rowCount();
                logger.info(
                    "For message id " + id
                    + ", message state count = " + rowCount);
                if (rowCount == 0) {
                    return new QuerySmResult(
                        timeFormatter.format(new Date()),
                        MessageState.UNKNOWN,
                        (byte) 0);
                } else {
                    Row row = rowSet.iterator().next();
                    Date finalDate;
                    Temporal deliveredTimestamp =
                        row.getTemporal("delivered_timestamp");
                    if (deliveredTimestamp == null) {
                        finalDate = new Date();
                    } else {
                        finalDate =
                            Date.from(Instant.from(deliveredTimestamp));
                    }
                    String state = row.getString("state");
                    logger.info(
                        "For message id " + id + " got state " + state
                        + " with delivered timestamp " + deliveredTimestamp);
                    MessageState messageState;
                    switch (state) {
                        case "accepted":
                            messageState = MessageState.SCHEDULED;
                            break;
                        case "sent":
                            messageState = MessageState.ENROUTE;
                            break;
                        case "delivered":
                        case "seen":
                        case "delivered_receipt_sent":
                        case "seen_receipt_sent":
                            messageState = MessageState.DELIVERED;
                            break;
                        case "error":
                        case "error_receipt_sent":
                            messageState = MessageState.UNDELIVERABLE;
                            break;
                        default:
                            throw new ProcessRequestException(
                                "Invalid state: " + state,
                                SMPPConstant.STAT_ESME_RSYSERR);
                    }
                    loggers.status = "OK";
                    return new QuerySmResult(
                        timeFormatter.format(finalDate),
                        messageState,
                        (byte) 0);
                }
            } catch (Throwable t) {
                loggers.error(t);
                throw new ProcessRequestException(
                    "Database transaction failed",
                    SMPPConstant.STAT_ESME_RSUBMITFAIL,
                    t);
            }
        } catch (RuntimeException e) {
            logger.log(
                Level.WARNING,
                "onAcceptQuerySm internal error:\n" + listener.details(),
                e);
            throw new ProcessRequestException(
                "Interal error",
                SMPPConstant.STAT_ESME_RSYSERR,
                e);
        } catch (Throwable t) {
            logger.log(
                Level.WARNING,
                "onAcceptQuerySm failed:\n" + listener.details(),
                t);
            throw t;
        } finally {
            logger.info("onAcceptQuerySm upstreams stats: " + listener);
        }
    }

    @Override
    public void onAcceptReplaceSm(
        final ReplaceSm replaceSm,
        final SMPPServerSession session)
        throws ProcessRequestException
    {
        logger.warning("Rejecting replace_sm request");
        try (Loggers loggers =
                new Loggers(
                    "replace_sm ? ?",
                    EmptyRequestsListener.INSTANCE,
                    session))
        {
            ProcessRequestException e = new ProcessRequestException(
                "replace_sm not implemented",
                SMPPConstant.STAT_ESME_RREPLACEFAIL);
            loggers.error(e);
            throw e;
        }
    }

    @Override
    public void onAcceptCancelSm(
        final CancelSm cancelSm,
        final SMPPServerSession session)
        throws ProcessRequestException
    {
        logger.warning("Rejecting cancel_sm request");
        try (Loggers loggers =
                new Loggers(
                    "cancel_sm ? ?",
                    EmptyRequestsListener.INSTANCE,
                    session))
        {
            ProcessRequestException e = new ProcessRequestException(
                "cancel_sm not implemented",
                SMPPConstant.STAT_ESME_RCANCELFAIL);
            loggers.error(e);
            throw e;
        }
    }

    @Override
    public BroadcastSmResult onAcceptBroadcastSm(
        final BroadcastSm broadcastSm,
        final SMPPServerSession session)
        throws ProcessRequestException
    {
        logger.warning("Rejecting broadcast_sm request");
        try (Loggers loggers =
                new Loggers(
                    "broadcast_sm ? ?",
                    EmptyRequestsListener.INSTANCE,
                    session))
        {
            ProcessRequestException e = new ProcessRequestException(
                "broadcast_sm not implemented",
                SMPPConstant.STAT_ESME_RBCASTFAIL);
            loggers.error(e);
            throw e;
        }
    }

    @Override
    public void onAcceptCancelBroadcastSm(
        final CancelBroadcastSm cancelBroadcastSm,
        final SMPPServerSession session)
        throws ProcessRequestException
    {
        logger.warning("Rejecting cancel_broadcast_sm request");
        try (Loggers loggers =
                new Loggers(
                    "cancel_broadcast_sm ? ?",
                    EmptyRequestsListener.INSTANCE,
                    session))
        {
            ProcessRequestException e = new ProcessRequestException(
                "cancel_broadcast_sm not implemented",
                SMPPConstant.STAT_ESME_RBCASTCANCELFAIL);
            loggers.error(e);
            throw e;
        }
    }

    @Override
    public QueryBroadcastSmResult onAcceptQueryBroadcastSm(
        final QueryBroadcastSm queryBroadcastSm,
        final SMPPServerSession session)
        throws ProcessRequestException
    {
        logger.warning("Rejecting query_broadcast_sm request");
        try (Loggers loggers =
                new Loggers(
                    "query_broadcast_sm ? ?",
                    EmptyRequestsListener.INSTANCE,
                    session))
        {
            ProcessRequestException e = new ProcessRequestException(
                "query_broadcast_sm not implemented",
                SMPPConstant.STAT_ESME_RBCASTQUERYFAIL);
            loggers.error(e);
            throw e;
        }
    }

    @Override
    public DataSmResult onAcceptDataSm(
        final DataSm dataSm,
        final Session session)
        throws ProcessRequestException
    {
        logger.warning("Rejecting data_sm request");
        try (Loggers loggers =
                new Loggers(
                    "data_sm ? ?",
                    EmptyRequestsListener.INSTANCE,
                    session))
        {
            ProcessRequestException e = new ProcessRequestException(
                "data_sm not implemented",
                SMPPConstant.STAT_ESME_RSYSERR);
            loggers.error(e);
            throw e;
        }
    }

    private static class WaitBindTask implements Runnable {
        private final SMPPServerSession session;
        private final ImmutableTmaConfig config;
        private final PrefixedLogger logger;

        WaitBindTask(
            final SMPPServerSession session,
            final ImmutableTmaConfig config,
            final PrefixedLogger logger)
        {
            this.session = session;
            this.config = config;
            this.logger = logger;
        }

        private static InterfaceVersion minVersion(
            final InterfaceVersion lhs,
            final InterfaceVersion rhs)
        {
            if (rhs == null || rhs.ordinal() > lhs.ordinal()) {
                return lhs;
            } else {
                return rhs;
            }
        }

        @Override
        public void run() {
            int timeout = config.timeout();
            logger.info("Waiting for bind for " + timeout + " ms");
            try {
                BindRequest bindRequest = session.waitForBind(timeout);
                logger.info("Got bind type " + bindRequest.getBindType());
                try {
                    if (BindType.BIND_TRX.equals(bindRequest.getBindType())) {
                        String systemId = bindRequest.getSystemId();
                        if (config.systemId().equals(systemId)) {
                            String password = bindRequest.getPassword();
                            if (config.password().equals(password)) {
                                logger.info("Accepting bind");
                                session.setInterfaceVersion(
                                    minVersion(
                                        InterfaceVersion.IF_50,
                                        bindRequest.getInterfaceVersion()));
                                bindRequest.accept(
                                    "sys",
                                    InterfaceVersion.IF_50);
                            } else {
                                logger.warning(
                                    "Rejecting bind: invalid password");
                                bindRequest.reject(
                                    SMPPConstant.STAT_ESME_RINVPASWD);
                            }
                        } else {
                            logger.warning(
                                "Rejecting bind: invalid system id: "
                                + systemId);
                            bindRequest.reject(
                                SMPPConstant.STAT_ESME_RINVSYSID);
                        }
                    } else {
                        logger.warning(
                            "Rejecting bind: "
                            + "expected transciever bind type, got: "
                            + bindRequest.getBindType());
                        bindRequest.reject(SMPPConstant.STAT_ESME_RBINDFAIL);
                    }
                } catch (PDUStringException e) {
                    logger.log(
                        Level.WARNING,
                        "Rejecting bind: bind accept failed",
                        e);
                    bindRequest.reject(SMPPConstant.STAT_ESME_RSYSERR);
                }
            } catch (TimeoutException e) {
                logger.log(Level.WARNING, "Bind timeout", e);
            } catch (IOException e) {
                logger.log(Level.WARNING, "Bind failed", e);
            }
        }
    }

    private static class SubmitSession {
        private final SMPPServerSession smppSession;
        private final SubmitSm submitSm;

        SubmitSession(
            final SMPPServerSession smppSession,
            final SubmitSm submitSm)
        {
            this.smppSession = smppSession;
            this.submitSm = submitSm;
        }
    }

    private class Loggers implements AutoCloseable {
        private final long start = TimeSource.INSTANCE.currentTimeMillis();
        private final String smppRequest;
        private final RequestsListener listener;
        private final String requestId;
        private final String sessionId;
        private final String address;
        private final PrefixedLogger logger;
        private final Logger smppAccessLogger;
        private String status = null;
        private Throwable t = null;

        Loggers(
            final String smppRequest,
            final RequestsListener listener,
            final Session session)
        {
            this.smppRequest = smppRequest;
            this.listener = listener;
            requestId = generateSessionId();
            sessionId = session.getSessionId();
            int port;
            if (session instanceof SMPPServerSession) {
                SMPPServerSession serverSession = (SMPPServerSession) session;
                InetAddress inetAddress = serverSession.getInetAddress();
                if (inetAddress == null) {
                    address = null;
                    port = -1;
                } else {
                    address = inetAddress.getHostAddress();
                    port = serverSession.getPort();
                }
            } else {
                address = null;
                port = -1;
            }
            logger = SmppServer.this.logger.addPrefix(requestId);
            smppAccessLogger = SmppServer.this.smppAccessLogger;
            logger.info(
                "Processing request \"" + smppRequest
                + "\" on session " + sessionId
                + " from " + address + ':' + port);
        }

        @Override
        public void close() {
            logger.info("Request processed");
            HashMapLookup sessionInfo = new HashMapLookup();
            if (t == null) {
                sessionInfo.put("status", status);
            } else {
                sessionInfo.put("status", t.getClass().getName());
            }
            sessionInfo.put("remote_addr", address);
            sessionInfo.put("session_id", requestId);
            sessionInfo.put("smpp_session_id", sessionId);
            sessionInfo.put("request", smppRequest);
            long end = TimeSource.INSTANCE.currentTimeMillis();
            sessionInfo.put("request_time", Long.toString(end - start));
            if (listener != null) {
                sessionInfo.put("upstream_stats", listener.toString());
            }
            if (t != null) {
                StringBuilder sb = new StringBuilder("Request failed");
                if (listener != null && !listener.isEmpty()) {
                    sb.append(':');
                    sb.append('\n');
                    sb.append(listener.details());
                }
                logger.log(
                    Level.WARNING,
                    new String(sb),
                    t);
            }
            smppAccessLogger.log(Level.INFO, "", sessionInfo);
        }

        public void error(final Throwable t) {
            if (this.t == null) {
                this.t = t;
            } else {
                this.t.addSuppressed(t);
            }
        }
    }
}

