package ru.yandex.tma;

import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.security.GeneralSecurityException;
import java.security.Key;
import java.security.SecureRandom;
import java.security.spec.AlgorithmParameterSpec;
import java.util.logging.Logger;

import javax.crypto.Cipher;
import javax.crypto.spec.IvParameterSpec;
import javax.crypto.spec.SecretKeySpec;

import io.vertx.core.Vertx;

import ru.yandex.base64.Base64Decoder;
import ru.yandex.base64.Base64Encoder;
import ru.yandex.client.pg.PgClient;
import ru.yandex.collection.Pattern;
import ru.yandex.function.ByteArrayCopier;
import ru.yandex.http.proxy.HttpProxy;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.http.util.server.SessionContext;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.tma.config.ImmutableTmaConfig;
import ru.yandex.tskv.BasicPrefixedTskvLogger;
import ru.yandex.util.string.StringUtils;
import ru.yandex.util.timesource.TimeSource;

public class TmaServer extends HttpProxy<ImmutableTmaConfig> {
    private static final String ENCRYPTION_ALGORITHM = "AES/CBC/PKCS5Padding";

    private final String workerId =
        SessionContext.HOSTNAME
        + '_' + TimeSource.INSTANCE.currentTimeMillis()
        + '_' + Long.toHexString(new SecureRandom().nextLong() | (1L << 63));

    private final PgClient pgClient;
    private final AsyncClient messengerGatewayClient;
    private final Logger smppAccessLogger;
    private final BasicPrefixedTskvLogger tskvLogger;
    private final SmppServer smppServer;
    private final Key aesKey;
    private final AlgorithmParameterSpec aesIv;

    public TmaServer(final ImmutableTmaConfig config)
        throws ConfigException, IOException, InterruptedException
    {
        super(config);

        Vertx vertx = Vertx.vertx();
        closeChain.add(vertx::close);

        pgClient = new PgClient(vertx, config.pgClientConfig(), logger);
        closeChain.add(pgClient);

        messengerGatewayClient =
            client("MessengerGateway", config.messengerGatewayConfig())
                .adjustAsteriskStater();

        smppAccessLogger =
            config.smppAccessLoggerConfig().build(
                config.loggers().handlersManager());
        registerLoggerForLogrotate(smppAccessLogger);

        tskvLogger =
            new BasicPrefixedTskvLogger(
                config.tskvLoggerConfig().build(
                    config.loggers().handlersManager()));
        registerLoggerForLogrotate(tskvLogger.logger());

        smppServer = new SmppServer(this);
        closeChain.add(smppServer);

        aesKey = new SecretKeySpec(config.aesKey(), "AES");
        aesIv = new IvParameterSpec(config.aesIv());

        register(
            new Pattern<>("/delivery-reports", false),
            new DeliveryReportsHandler(this));
    }

    public String workerId() {
        return workerId;
    }

    public PgClient pgClient() {
        return pgClient;
    }

    public AsyncClient messengerGatewayClient() {
        return messengerGatewayClient;
    }

    public Logger smppAccessLogger() {
        return smppAccessLogger;
    }

    public BasicPrefixedTskvLogger tskvLogger() {
        return tskvLogger;
    }

    public SmppServer smppServer() {
        return smppServer;
    }

    @Override
    public void start() throws IOException {
        smppServer.start();
        super.start();
    }

    public String encrypt(final String str) throws IOException {
        try {
            Cipher cipher = Cipher.getInstance(ENCRYPTION_ALGORITHM);
            cipher.init(Cipher.ENCRYPT_MODE, aesKey, aesIv);
            Base64Encoder encoder = new Base64Encoder();
            encoder.process(cipher.doFinal(StringUtils.getUtf8Bytes(str)));
            return encoder.toString();
        } catch (GeneralSecurityException e) {
            throw new IOException(e);
        }
    }

    public String decrypt(final String str) throws IOException {
        try {
            Cipher cipher = Cipher.getInstance(ENCRYPTION_ALGORITHM);
            cipher.init(Cipher.DECRYPT_MODE, aesKey, aesIv);
            Base64Decoder decoder = new Base64Decoder();
            decoder.process(str.toCharArray());
            return
                new String(
                    cipher.doFinal(
                        decoder.processWith(ByteArrayCopier.INSTANCE)),
                    StandardCharsets.UTF_8);
        } catch (GeneralSecurityException e) {
            throw new IOException(e);
        }
    }

    public void wakeupQueueProcessor() {
        smppServer.wakeupQueueProcessor();
    }
}

