package ru.yandex.tma.config;

import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Set;

import ru.yandex.client.pg.config.PgClientConfig;
import ru.yandex.client.pg.config.PgClientConfigBuilder;
import ru.yandex.http.config.HttpHostConfig;
import ru.yandex.http.config.HttpHostConfigBuilder;
import ru.yandex.http.util.server.AbstractHttpProxyConfigBuilder;
import ru.yandex.logger.LoggerConfig;
import ru.yandex.logger.LoggerConfigBuilder;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.config.IniConfig;

public abstract class AbstractTmaConfigBuilder
    <T extends AbstractTmaConfigBuilder<T>>
    extends AbstractHttpProxyConfigBuilder<T>
    implements TmaConfig
{
    private int smppPort;
    private String messengerService;
    private String messengerAccount;
    private long deliveryTtl;
    private long dbScanInterval;
    private String systemId;
    private String password;
    private byte[] aesKey;
    private byte[] aesIv;
    private Map<String, TemplateConfigBuilder> templates;
    private PgClientConfigBuilder pgClientConfig;
    private HttpHostConfigBuilder messengerGatewayConfig;
    private LoggerConfigBuilder smppAccessLoggerConfig;
    private LoggerConfigBuilder tskvLoggerConfig;

    protected AbstractTmaConfigBuilder(final TmaConfig config) {
        super(config);
        smppPort(config.smppPort());
        messengerService(config.messengerService());
        messengerAccount(config.messengerAccount());
        deliveryTtl(config.deliveryTtl());
        dbScanInterval(config.dbScanInterval());
        systemId(config.systemId());
        password(config.password());
        aesKey(config.aesKey());
        aesIv(config.aesIv());
        templates(config.templates());
        pgClientConfig(config.pgClientConfig());
        messengerGatewayConfig(config.messengerGatewayConfig());
        smppAccessLoggerConfig(config.smppAccessLoggerConfig());
        tskvLoggerConfig(config.tskvLoggerConfig());
    }

    protected AbstractTmaConfigBuilder(
        final IniConfig config,
        final TmaConfig defaults)
        throws ConfigException
    {
        super(config, defaults);
        smppPort = SMPP_PORT.extract(config, defaults.smppPort());
        messengerService =
            MESSENGER_SERVICE.extract(config, defaults.messengerService());
        messengerAccount =
            MESSENGER_ACCOUNT.extract(config, defaults.messengerAccount());
        deliveryTtl = DELIVERY_TTL.extract(config, defaults.deliveryTtl());
        dbScanInterval =
            DB_SCAN_INTERVAL.extract(config, defaults.dbScanInterval());
        systemId = SYSTEM_ID.extract(config, defaults.systemId());
        password = PASSWORD.extract(config, defaults.password());
        aesKey = AES_KEY.extract(config, defaults.aesKey());
        aesIv = AES_IV.extract(config, defaults.aesIv());
        IniConfig templatesSection = config.section("templates");
        Set<String> templatesNames = templatesSection.sections().keySet();
        templates = new LinkedHashMap<>(templatesNames.size() << 1);
        for (String templateName: templatesNames) {
            templates.put(
                templateName,
                TmaConfig.templateParser(templateName)
                    .extract(
                        templatesSection,
                        TemplateConfigDefaults.INSTANCE));
        }
        pgClientConfig =
            PG_CLIENT_CONFIG.extract(config, defaults.pgClientConfig());
        messengerGatewayConfig =
            MESSENGER_GATEWAY_CONFIG.extract(
                config,
                defaults.messengerGatewayConfig());
        smppAccessLoggerConfig =
            SMPP_ACCESS_LOGGER_CONFIG.extract(
                config,
                defaults.smppAccessLoggerConfig());
        tskvLoggerConfig =
            TSKV_LOGGER_CONFIG.extract(config, defaults.tskvLoggerConfig());
    }

    @Override
    public int smppPort() {
        return smppPort;
    }

    public T smppPort(final int smppPort) {
        this.smppPort = smppPort;
        return self();
    }

    @Override
    public String messengerService() {
        return messengerService;
    }

    public T messengerService(final String messengerService) {
        this.messengerService = messengerService;
        return self();
    }

    @Override
    public String messengerAccount() {
        return messengerAccount;
    }

    public T messengerAccount(final String messengerAccount) {
        this.messengerAccount = messengerAccount;
        return self();
    }

    @Override
    public long deliveryTtl() {
        return deliveryTtl;
    }

    public T deliveryTtl(final long deliveryTtl) {
        this.deliveryTtl = deliveryTtl;
        return self();
    }

    @Override
    public long dbScanInterval() {
        return dbScanInterval;
    }

    public T dbScanInterval(final long dbScanInterval) {
        this.dbScanInterval = dbScanInterval;
        return self();
    }

    @Override
    public String systemId() {
        return systemId;
    }

    public T systemId(final String systemId) {
        this.systemId = systemId;
        return self();
    }

    @Override
    public String password() {
        return password;
    }

    public T password(final String password) {
        this.password = password;
        return self();
    }

    @Override
    public byte[] aesKey() {
        return aesKey;
    }

    public T aesKey(final byte[] aesKey) {
        this.aesKey = aesKey;
        return self();
    }

    @Override
    public byte[] aesIv() {
        return aesIv;
    }

    public T aesIv(final byte[] aesIv) {
        this.aesIv = aesIv;
        return self();
    }

    @Override
    public Map<String, TemplateConfigBuilder> templates() {
        return templates;
    }

    public T templates(final Map<String, ? extends TemplateConfig> templates) {
        this.templates = new LinkedHashMap<>(templates.size() << 1);
        for (Map.Entry<String, ? extends TemplateConfig> entry
            : templates.entrySet())
        {
            this.templates.put(
                entry.getKey(),
                new TemplateConfigBuilder(entry.getValue()));
        }
        return self();
    }

    @Override
    public PgClientConfigBuilder pgClientConfig() {
        return pgClientConfig;
    }

    public T pgClientConfig(final PgClientConfig pgClientConfig) {
        this.pgClientConfig = new PgClientConfigBuilder(pgClientConfig);
        return self();
    }

    @Override
    public HttpHostConfigBuilder messengerGatewayConfig() {
        return messengerGatewayConfig;
    }

    public T messengerGatewayConfig(
        final HttpHostConfig messengerGatewayConfig)
    {
        this.messengerGatewayConfig =
            new HttpHostConfigBuilder(messengerGatewayConfig);
        return self();
    }

    @Override
    public LoggerConfigBuilder smppAccessLoggerConfig() {
        return smppAccessLoggerConfig;
    }

    public T smppAccessLoggerConfig(final LoggerConfig smppAccessLoggerConfig) {
        this.smppAccessLoggerConfig = new LoggerConfigBuilder(smppAccessLoggerConfig);
        return self();
    }

    @Override
    public LoggerConfigBuilder tskvLoggerConfig() {
        return tskvLoggerConfig;
    }

    public T tskvLoggerConfig(final LoggerConfig tskvLoggerConfig) {
        this.tskvLoggerConfig = new LoggerConfigBuilder(tskvLoggerConfig);
        return self();
    }
}

