package ru.yandex.passport.familypay.backend;

import java.io.IOException;
import java.util.logging.Level;

import org.apache.http.HttpStatus;
import org.apache.http.entity.ContentType;
import org.apache.http.nio.entity.NStringEntity;

import ru.yandex.http.util.YandexHeaders;
import ru.yandex.io.StringBuilderWriter;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonWriter;

public abstract class AbstractFamilypayCallback<T>
    implements FamilypayCallback<T>
{
    protected final RequestContext context;

    protected AbstractFamilypayCallback(final RequestContext context) {
        this.context = context;
    }

    @Override
    public RequestContext context() {
        return context;
    }

    @Override
    public void cancelled() {
        context.session().logger().warning("Request cancelled");
    }

    @Override
    public void failed(final Exception e) {
        failed(ErrorType.INTERNAL_ERROR, null, e);
    }

    @Override
    public void failed(
        final ErrorType errorType,
        final String message,
        final Throwable cause)
    {
        failed(context, errorType, message, cause);
    }

    public static void failed(
        final RequestContext context,
        final ErrorType errorType,
        final String message,
        final Throwable cause)
    {
        String description = errorType.description();
        if (message != null) {
            description = description + ": " + message;
        }
        String logMessage =
            "Error code " + errorType.errorCode() + ": " + description;
        context.session().logger().log(Level.WARNING, logMessage, cause);
        if (!context.quiet()) {
            context.tskvLogger().log(
                context.tskvRecord(
                    TskvFields.Stage.FAILED,
                    "Request failed: " + logMessage)
                    .append(TskvFields.ERROR_CODE, errorType.errorCode()));
        }
        StringBuilderWriter sbw = new StringBuilderWriter();
        try (JsonWriter writer = context.jsonType().create(sbw)) {
            writer.startObject();
            writer.key("error");
            writer.value(errorType.errorCode());
            writer.key("description");
            writer.value(description);
            if (cause != null) {
                writer.key("stacktrace");
                writer.value(cause);
            }
            writer.endObject();
        } catch (IOException e) {
            // impossible
        }

        String requestId =
            context.session().headers().getOrNull(YandexHeaders.X_REQUEST_ID);
        if (requestId != null) {
            context.session().getResponse().addHeader(
                YandexHeaders.X_REQUEST_ID,
                requestId);
        }
        context.session().response(
            errorType.httpCode(),
            new NStringEntity(
                sbw.toString(),
                ContentType.APPLICATION_JSON.withCharset(
                    context.session().acceptedCharset())));
    }

    @Override
    public void sendResponse(final JsonObject result) {
        if (!context.quiet()) {
            context.tskvLogger().log(
                context.tskvRecord(
                    TskvFields.Stage.COMPLETED,
                    JsonType.NORMAL.toString(result)));
        }
        context.session().response(
            HttpStatus.SC_OK,
            new NStringEntity(
                context.jsonType().toString(result),
                ContentType.APPLICATION_JSON.withCharset(
                    context.session().acceptedCharset())));
    }

    @Override
    public void emptyResponse(final String logMessage) {
        emptyResponse(context, logMessage);
    }

    public static void emptyResponse(
        final RequestContext context,
        final String logMessage)
    {
        if (!context.quiet()) {
            context.tskvLogger().log(
                context.tskvRecord(
                    TskvFields.Stage.COMPLETED,
                    logMessage));
        }
        context.session().response(HttpStatus.SC_NO_CONTENT);
    }
}

