package ru.yandex.passport.familypay.backend;

import java.util.Objects;

import io.vertx.sqlclient.Row;
import io.vertx.sqlclient.Tuple;

import ru.yandex.function.AbstractStringBuilderable;
import ru.yandex.json.dom.ContainerFactory;
import ru.yandex.json.dom.JsonBoolean;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonString;
import ru.yandex.json.parser.JsonException;
import ru.yandex.parser.string.NonEmptyValidator;

public class CardInfo extends AbstractStringBuilderable {
    private final String cardId;
    private final String cardMask;
    private final String currency;
    private final String paySystem;
    private final Boolean bound;

    public CardInfo(
        final String cardId,
        final String cardMask,
        final String currency,
        final String paySystem,
        final Boolean bound)
    {
        this.cardId = cardId;
        this.cardMask = cardMask;
        this.currency = currency;
        this.paySystem = paySystem;
        this.bound = bound;
    }

    public static CardInfo fromRow(final Row row) {
        return new CardInfo(
            row.getString("card_id"),
            row.getString("card_mask"),
            row.getString("card_currency"),
            row.getString("card_pay_system"),
            row.getBoolean("card_bound"));
    }

    public void toTuple(final Tuple tuple) {
        tuple.addString(cardId);
        tuple.addString(cardMask);
        tuple.addString(currency);
        tuple.addString(paySystem);
        tuple.addBoolean(bound);
    }

    public static CardInfo fromJson(final JsonMap map) throws JsonException {
        return fromJson(map, map.getBoolean("bound", null));
    }

    public static CardInfo fromJson(
        final JsonMap map,
        final Boolean bound)
        throws JsonException
    {
        return new CardInfo(
            map.get("cardId", null, CardIdParser.INSTANCE),
            map.get("maskedNumber", null, NonEmptyValidator.TRIMMED),
            map.get("currency", null, NonEmptyValidator.TRIMMED),
            map.get("paySystem", null, NonEmptyValidator.TRIMMED),
            bound);
    }

    public JsonMap toJson(final ContainerFactory containerFactory) {
        JsonMap result = new JsonMap(containerFactory, 7);
        result.put("cardId", JsonString.valueOf(cardId));
        result.put("maskedNumber", JsonString.valueOf(cardMask));
        result.put("currency", JsonString.valueOf(currency));
        result.put("paySystem", JsonString.valueOf(paySystem));
        result.put("bound", JsonBoolean.valueOf(bound));
        return result;
    }

    public String cardId() {
        return cardId;
    }

    public String cardMask() {
        return cardMask;
    }

    public String currency() {
        return currency;
    }

    public String paySystem() {
        return paySystem;
    }

    public Boolean bound() {
        return bound;
    }

    @Override
    public int hashCode() {
        return Objects.hash(cardId, cardMask, currency, paySystem, bound);
    }

    @Override
    public boolean equals(final Object o) {
        if (o instanceof CardInfo) {
            CardInfo other = (CardInfo) o;
            return Objects.equals(cardId, other.cardId)
                && Objects.equals(cardMask, other.cardMask)
                && Objects.equals(currency, other.currency)
                && Objects.equals(paySystem, other.paySystem)
                && Objects.equals(bound, other.bound);
        }
        return false;
    }

    @Override
    public void toStringBuilder(final StringBuilder sb) {
        sb.append("CardInfo(");
        sb.append(cardId);
        sb.append(',');
        sb.append(cardMask);
        sb.append(',');
        sb.append(currency);
        sb.append(',');
        sb.append(paySystem);
        sb.append(',');
        sb.append(bound);
        sb.append(')');
    }
}

