package ru.yandex.passport.familypay.backend;

import io.vertx.sqlclient.Row;
import io.vertx.sqlclient.RowSet;
import io.vertx.sqlclient.Tuple;
import org.apache.http.HttpEntity;
import org.apache.http.HttpEntityEnclosingRequest;
import org.apache.http.HttpException;
import org.apache.http.HttpRequest;
import org.apache.http.nio.protocol.HttpAsyncExchange;
import org.apache.http.nio.protocol.HttpAsyncRequestConsumer;
import org.apache.http.nio.protocol.HttpAsyncRequestHandler;
import org.apache.http.protocol.HttpContext;

import ru.yandex.client.pg.SqlQuery;
import ru.yandex.http.proxy.BasicProxySession;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.nio.EmptyAsyncConsumer;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumer;
import ru.yandex.json.dom.BasicContainerFactory;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonNull;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.parser.StringCollectorsFactory;
import ru.yandex.json.writer.JsonType;
import ru.yandex.parser.string.PositiveLongValidator;

public class CardInfoChangeHandler
    implements HttpAsyncRequestHandler<JsonObject>
{
    private static final SqlQuery UNBIND_CARD =
        new SqlQuery("unbind-card.sql", CardInfoChangeHandler.class);

    protected final FamilypayBackend server;

    public CardInfoChangeHandler(final FamilypayBackend server) {
        this.server = server;
    }

    @Override
    public HttpAsyncRequestConsumer<JsonObject> processRequest(
        final HttpRequest request,
        final HttpContext context)
        throws HttpException
    {
        if (request instanceof HttpEntityEnclosingRequest) {
            HttpEntity entity =
                ((HttpEntityEnclosingRequest) request).getEntity();
            if (entity.getContentLength() != 0) {
                return new JsonAsyncTypesafeDomConsumer(
                    entity,
                    StringCollectorsFactory.INSTANCE,
                    BasicContainerFactory.INSTANCE);
            }
        }
        return new EmptyAsyncConsumer<>(JsonNull.INSTANCE);
    }

    @Override
    public void handle(
        final JsonObject payload,
        final HttpAsyncExchange exchange,
        final HttpContext context)
        throws HttpException
    {
        ProxySession session =
            new BasicProxySession(server, exchange, context);
        RequestContext requestContext =
            new RequestContext(
                server,
                session,
                server.tskvLogger().record()
                    .append(TskvFields.HANDLER, getClass().getSimpleName()),
                JsonType.NORMAL.toString(payload),
                false);
        try {
            JsonMap map = payload.asMap();
            String kind = map.getString("kind", null);
            if ("unbind".equals(kind)) {
                long uid = map.get("uid", PositiveLongValidator.INSTANCE);
                String cardId = map.get("card_id", CardIdParser.INSTANCE);
                String shortCardId = map.getString("short_card_id", "");
                if (shortCardId.isEmpty()) {
                    shortCardId = cardId;
                } else {
                    try {
                        shortCardId = CardIdParser.INSTANCE.apply(shortCardId);
                    } catch (Exception e) {
                        throw new BadRequestException(
                            "Failed to parse short card id <"
                            + shortCardId + '>',
                            e);
                    }
                }
                String message =
                    "Unbounding uid " + uid
                    + " card " + cardId + '/' + shortCardId;
                session.logger().info(message);
                requestContext.tskvLogger().log(
                    requestContext.tskvRecord(
                        TskvFields.Stage.INTERMEDIATE,
                        message));
                Tuple tuple = Tuple.tuple();
                tuple.addLong(uid);
                tuple.addString(cardId);
                tuple.addString(shortCardId);
                FamilyCallback callback =
                    new FamilyCallback(requestContext, tuple);
                FamilyHandler.findFamilyByAdminUid(uid, callback);
            } else {
                session.logger().info("Skipping event kind: " + kind);
                AbstractFamilypayCallback.emptyResponse(
                    requestContext,
                    "Event skipped");
            }
        } catch (JsonException e) {
            AbstractFamilypayCallback.failed(
                requestContext,
                ErrorType.MALFORMED_REQUEST,
                JsonType.NORMAL.toString(payload),
                e);
        }
    }

    private static class FamilyCallback
        extends AbstractFamilypayCallback<Family>
    {
        private final Tuple tuple;

        FamilyCallback(
            final RequestContext context,
            final Tuple tuple)
        {
            super(context);
            this.tuple = tuple;
        }

        @Override
        public void failed(
            final ErrorType errorType,
            final String message,
            final Throwable cause)
        {
            if (errorType == ErrorType.FAMILY_NOT_FOUND) {
                completed(null);
            } else {
                super.failed(errorType, message, cause);
            }
        }

        @Override
        public void completed(final Family family) {
            if (family == null) {
                context.session().logger().info("Family not found");
                emptyResponse();
            } else {
                context.server().pgClient().executeOnMaster(
                    UNBIND_CARD,
                    tuple,
                    context.session().listener(),
                    new Callback(context, family));
            }
        }
    }

    private static class Callback
        extends AbstractFamilypayCallback<RowSet<Row>>
    {
        private final Family oldFamily;

        Callback(final RequestContext context, final Family oldFamily) {
            super(context);
            this.oldFamily = oldFamily;
        }

        @Override
        public void completed(final RowSet<Row> rowSet) {
            int rows = rowSet.rowCount();
            context.session().logger().info("Rows affected: " + rows);
            emptyResponse();

            FamilyHandler.findFamily(
                oldFamily.familyInfo().familyId(),
                new FamilyChangeNotifyCallback(context, oldFamily));
        }
    }
}

