package ru.yandex.passport.familypay.backend;

import java.util.Locale;

import io.vertx.sqlclient.Row;
import io.vertx.sqlclient.RowSet;
import io.vertx.sqlclient.Tuple;
import org.apache.http.HttpException;

import ru.yandex.client.pg.SqlQuery;
import ru.yandex.json.dom.BasicContainerFactory;
import ru.yandex.json.dom.JsonLong;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonString;
import ru.yandex.json.writer.JsonType;
import ru.yandex.passport.familypay.backend.config.ImmutableServiceConfig;

public class CompletePaymentHandler extends PaymentHandlerBase {
    private static final SqlQuery COMPLETE_PAYMENT =
        new SqlQuery("complete-payment.sql", CompletePaymentHandler.class);

    public CompletePaymentHandler(final FamilypayBackend server) {
        super(server, false);
    }

    @Override
    protected void handle(
        final String paymentId,
        final RequestContext context)
        throws HttpException
    {
        Tuple tuple = Tuple.tuple();
        tuple.addString(paymentId);
        Status status =
            context.session().params().getEnum(Status.class, "status");
        tuple.addString(status.toString().toLowerCase(Locale.ROOT));
        server.pgClient().executeOnMaster(
            COMPLETE_PAYMENT,
            tuple,
            context.session().listener(),
            new Callback(context, paymentId, status));
    }

    private static class Callback
        extends AbstractFamilypayCallback<RowSet<Row>>
    {
        private final String paymentId;
        private final Status status;

        Callback(
            final RequestContext context,
            final String paymentId,
            final Status status)
        {
            super(context);
            this.paymentId = paymentId;
            this.status = status;
        }

        @Override
        public void completed(final RowSet<Row> rowSet) {
            try {
                if (rowSet.rowCount() <= 0) {
                    failed(ErrorType.PAYMENT_NOT_FOUND, null, null);
                } else {
                    Row row = rowSet.iterator().next();
                    String status = row.getString("status");
                    long uid = row.getLong("initiator_uid");
                    long amount = row.getLong("amount");
                    long serviceId = row.getLong("service_id");
                    String message =
                        "Payment status become '" + status
                        + "' for uid " + uid;
                    context.session().logger().info(message);

                    if (this.status.toString().toLowerCase(Locale.ROOT).equals(
                            status))
                    {
                        UserInfoHandler.findUser(
                            uid,
                            serviceId,
                            new UserCallback(
                                context,
                                paymentId,
                                this.status,
                                amount,
                                serviceId));
                    } else {
                        emptyResponse();
                    }
                }
            } catch (RuntimeException e) {
                failed(
                    ErrorType.INTERNAL_ERROR,
                    "Failed to parse database response",
                    e);
            }
        }
    }

    private static class UserCallback
        extends AbstractFamilypayCallback<UserInfo>
    {
        private final String paymentId;
        private final Status status;
        private final long amount;
        private final long serviceId;

        UserCallback(
            final RequestContext context,
            final String paymentId,
            final Status status,
            final long amount,
            final long serviceId)
        {
            super(context);
            this.paymentId = paymentId;
            this.status = status;
            this.amount = amount;
            this.serviceId = serviceId;
        }

        @Override
        public void completed(final UserInfo userInfo) {
            String origin = userInfo.familyOrigin();
            ImmutableServiceConfig serviceConfig =
                context.server().servicesConfigs().get(origin);
            boolean sendPush = true;
            if (serviceConfig != null) {
                sendPush = serviceConfig.sendPushOnPayment();
            }
            context.session().logger()
                .info("For origin " + origin + ", sendPush = " + sendPush);
            JsonMap info =
                userInfo.toJson(BasicContainerFactory.INSTANCE, true);
            info.put("amount", new JsonLong(amount));
            info.put("serviceId", new JsonLong(serviceId));
            info.put("status", new JsonString(status.toString()));
            info.put("paymentId", new JsonString(paymentId));
            emptyResponse(JsonType.NORMAL.toString(info));
            if (!sendPush) {
                return;
            }
            status.sendPush(
                context,
                userInfo,
                paymentId,
                amount,
                serviceId);
        }
    }

    private enum Status {
        COMPLETED {
            @Override
            public void sendPush(
                final RequestContext context,
                final UserInfo userInfo,
                final String paymentId,
                final long amount,
                final long serviceId)
            {
                context.server().pusher().paymentCompleted(
                    context,
                    userInfo,
                    paymentId,
                    amount,
                    serviceId);
            }
        },
        REJECTED {
            @Override
            public void sendPush(
                final RequestContext context,
                final UserInfo userInfo,
                final String paymentId,
                final long amount,
                final long serviceId)
            {
                context.server().pusher().paymentRejected(
                    context,
                    userInfo,
                    paymentId,
                    amount,
                    serviceId);
            }
        },
        FAILED {
            @Override
            public void sendPush(
                final RequestContext context,
                final UserInfo userInfo,
                final String paymentId,
                final long amount,
                final long serviceId)
            {
                context.server().pusher().paymentFailed(
                    context,
                    userInfo,
                    paymentId,
                    amount,
                    serviceId);
            }
        };

        public abstract void sendPush(
            RequestContext context,
            UserInfo userInfo,
            String paymentId,
            long amount,
            long serviceId);
    }
}

