package ru.yandex.passport.familypay.backend;

import io.vertx.sqlclient.Row;
import io.vertx.sqlclient.RowSet;
import io.vertx.sqlclient.Tuple;
import org.apache.http.HttpException;

import ru.yandex.client.pg.SqlQuery;
import ru.yandex.json.dom.BasicContainerFactory;
import ru.yandex.json.dom.JsonBoolean;
import ru.yandex.json.dom.JsonLong;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.writer.JsonType;
import ru.yandex.parser.string.PositiveLongValidator;

public class CreatePaymentHandler extends PaymentHandlerBase {
    private static final SqlQuery CREATE_PAYMENT =
        new SqlQuery("create-payment.sql", CreatePaymentHandler.class);
    private static final SqlQuery EXPENSES =
        new SqlQuery("calculate-expenses.sql", CreatePaymentHandler.class);

    public CreatePaymentHandler(final FamilypayBackend server) {
        super(server, false);
    }

    @Override
    protected void handle(
        final String paymentId,
        final RequestContext context)
        throws HttpException
    {
        long uid = context.session().params().get(
            "uid",
            PositiveLongValidator.INSTANCE);
        long serviceId = context.session().params().getLong("serviceId");
        String service = server.services().serviceName(serviceId, uid);
        context.session().logger().info(
            "Service resolved from service id " + serviceId
            + ':' + ' ' + service);
        if (service == null) {
            AbstractFamilypayCallback.failed(
                context,
                ErrorType.SERVICE_ID_NOT_FOUND,
                Long.toString(serviceId),
                null);
            return;
        }
        String cardId = context.session().params().get(
            "cardId",
            CardIdParser.INSTANCE);
        long amount = context.session().params().get(
            "amount",
            PositiveLongValidator.INSTANCE);
        Tuple tuple = Tuple.tuple();
        tuple.addString(paymentId);
        tuple.addLong(uid);
        tuple.addString(cardId);
        tuple.addLong(amount);
        tuple.addString(service);
        tuple.addLong(serviceId);
        tuple.addString(context.session().params().getString("currency"));
        server.pgClient().executeOnMaster(
            CREATE_PAYMENT,
            tuple,
            context.session().listener(),
            new CreateCallback(context, uid));
    }

    private static class CreateCallback
        extends AbstractFamilypayCallback<RowSet<Row>>
    {
        private final long uid;

        CreateCallback(final RequestContext context, final long uid) {
            super(context);
            this.uid = uid;
        }

        @Override
        public void completed(final RowSet<Row> rowSet) {
            if (rowSet.rowCount() <= 0) {
                failed(ErrorType.PAYMENT_NOT_ACCEPTABLE, null, null);
            } else {
                context.server().pgClient().executeOnMaster(
                    EXPENSES,
                    Tuple.of(uid),
                    context.session().listener(),
                    new ExpensesCallback(context));
            }
        }
    }

    private static class ExpensesCallback
        extends AbstractFamilypayCallback<RowSet<Row>>
    {
        ExpensesCallback(final RequestContext context) {
            super(context);
        }

        @Override
        public void completed(final RowSet<Row> rowSet) {
            try {
                if (rowSet.rowCount() <= 0) {
                    failed(ErrorType.INTERNAL_ERROR, null, null);
                } else {
                    Row row = rowSet.iterator().next();
                    JsonMap result =
                        new JsonMap(BasicContainerFactory.INSTANCE, 4);
                    result.put(
                        "familyAdminUid",
                        new JsonLong(row.getLong("admin_uid")));
                    result.put(
                        "unlim",
                        JsonBoolean.valueOf(row.getBoolean("unlim")));
                    result.put(
                        "limits",
                        LimitsInfo.fromRow(row, "", "_limit")
                            .toJson(
                                BasicContainerFactory.INSTANCE,
                                false,
                                ""));
                    result.put(
                        "expenses",
                        LimitsInfo.fromRow(row, "", "_amount")
                            .toJson(
                                BasicContainerFactory.INSTANCE,
                                false,
                                ""));
                    context.tskvLogger().log(
                        context.tskvRecord(
                            TskvFields.Stage.INTERMEDIATE,
                            "User state become: "
                            + JsonType.NORMAL.toString(result)));
                    sendResponse(result);
                }
            } catch (RuntimeException e) {
                failed(
                    ErrorType.INTERNAL_ERROR,
                    "Failed to parse database response",
                    e);
            }
        }
    }
}

