package ru.yandex.passport.familypay.backend;

import io.vertx.sqlclient.Row;
import io.vertx.sqlclient.RowSet;
import io.vertx.sqlclient.Tuple;

import ru.yandex.client.pg.SqlQuery;

public class DeleteFamilyEvent extends FamilyChangeEventBase {
    private static final SqlQuery DELETE_FAMILY =
        new SqlQuery("delete-family.sql", FamilyChangeHandler.class);

    public DeleteFamilyEvent(final String familyId, final long adminUid) {
        super(familyId, adminUid);
    }

    @Override
    protected void familyNotFound(
        final FamilypayCallback<? super Boolean> callback,
        final String eventId)
    {
        callback.context().server().pusher().notifyAdminOnFamilyDelete(
            callback.context(),
            uid,
            eventId);
        callback.completed(Boolean.FALSE);
    }

    @Override
    protected void familyFound(
        final FamilypayCallback<? super Boolean> callback,
        final String eventId,
        final Family family)
    {
        Tuple tuple = Tuple.tuple();
        tuple.addString(familyId);
        tuple.addLong(uid);
        callback.context().server().pgClient().executeOnMaster(
            DELETE_FAMILY,
            tuple,
            callback.context().session().listener(),
            new DeleteCallback(callback, eventId, family));
    }

    private static class DeleteCallback
        extends AbstractFilterFamilypayCallback<RowSet<Row>, Boolean>
    {
        private final String eventId;
        private final Family family;

        DeleteCallback(
            final FamilypayCallback<? super Boolean> callback,
            final String eventId,
            final Family family)
        {
            super(callback);
            this.eventId = eventId;
            this.family = family;
        }

        @Override
        public void completed(final RowSet<Row> rowSet) {
            int rows = rowSet.rowCount();
            callback.context().session().logger()
                .info("Rows affected by families delete: " + rows);
            if (rows == 0) {
                // There was either race condition or admin uid mismatch
                callback.completed(Boolean.FALSE);
            } else {
                context().server().pusher().notifyOnFamilyChange(
                    context(),
                    family,
                    null,
                    eventId);
                callback.completed(Boolean.TRUE);
            }
        }
    }
}

