package ru.yandex.passport.familypay.backend;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.logging.Level;

import org.apache.http.HttpEntity;
import org.apache.http.HttpEntityEnclosingRequest;
import org.apache.http.HttpException;
import org.apache.http.HttpRequest;
import org.apache.http.nio.protocol.HttpAsyncExchange;
import org.apache.http.nio.protocol.HttpAsyncRequestHandler;
import org.apache.http.protocol.HttpContext;

import ru.yandex.http.proxy.BasicProxySession;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.json.writer.JsonType;
import ru.yandex.parser.string.NonEmptyValidator;
import ru.yandex.tskv.TskvAsyncConsumer;
import ru.yandex.tskv.TskvException;
import ru.yandex.tskv.TskvRecord;

public class FamilyChangeHandler
    implements HttpAsyncRequestHandler<List<TskvRecord>>
{
    protected final FamilypayBackend server;

    public FamilyChangeHandler(final FamilypayBackend server) {
        this.server = server;
    }

    @Override
    public TskvAsyncConsumer processRequest(
        final HttpRequest request,
        final HttpContext context)
        throws HttpException
    {
        if (request instanceof HttpEntityEnclosingRequest) {
            HttpEntity entity =
                ((HttpEntityEnclosingRequest) request).getEntity();
            if (entity.getContentLength() != 0) {
                return new TskvAsyncConsumer(entity);
            }
        }
        throw new BadRequestException("Payload expected");
    }

    @Override
    public void handle(
        final List<TskvRecord> records,
        final HttpAsyncExchange exchange,
        final HttpContext context)
        throws HttpException
    {
        ProxySession session =
            new BasicProxySession(server, exchange, context);
        RequestContext requestContext =
            new RequestContext(
                server,
                session,
                server.tskvLogger().record()
                    .append(TskvFields.HANDLER, getClass().getSimpleName()),
                JsonType.NORMAL.toString(records),
                false);
        int i = 0;
        try {
            int size = records.size();
            List<FamilyChangeEvent> events = new ArrayList<>(size);
            for (; i < size; ++i) {
                TskvRecord record = records.get(i);
                String eventType = record.getString("event", "");
                if (!eventType.equals("family_info_modification")) {
                    session.logger().info(
                        "Skipping unsupported event type '" + eventType
                        + "': " + record);
                    continue;
                }
                OperationType operationType =
                    record.getEnum(OperationType.class, "operation");
                EntityType entityType =
                    record.getEnum(EntityType.class, "entity");
                String familyId = record.get(
                    "family_id",
                    NonEmptyValidator.INSTANCE);
                if (operationType == OperationType.DELETED) {
                    long uid = record.getLong("old");
                    if (entityType == EntityType.ADMIN_UID) {
                        events.add(new DeleteFamilyEvent(familyId, uid));
                    } else {
                        events.add(new DeleteFamilyMemberEvent(familyId, uid));
                    }
                } else {    // operationType == OperationType.CREATED
                    if (entityType == EntityType.ADMIN_UID) {
                        session.logger().info(
                            "Skipping family created event: " + record);
                    } else {
                        events.add(
                            new AddFamilyMemberEvent(
                                familyId,
                                record.getLong("new")));
                    }
                }
            }
            String message = "Family events: " + events;
            session.logger().info(message);
            requestContext.tskvLogger().log(
                requestContext.tskvRecord(
                    TskvFields.Stage.INTERMEDIATE,
                    message));
            new EventProcessor(
                requestContext,
                events,
                session.params().getString("topic", "") + '-'
                + session.params().getString("partition", "") + '-'
                + session.params().getString("offset", "") + '-')
                .process(0);
        } catch (TskvException e) {
            session.logger().log(
                Level.WARNING,
                "Failed to parse record #" + i + " in records: " + records,
                e);
            AbstractFamilypayCallback.failed(
                requestContext,
                ErrorType.MALFORMED_URI,
                null,
                e);
        }
    }

    private enum OperationType {
        CREATED,
        DELETED
    }

    private enum EntityType {
        ADMIN_UID,
        MEMBERS,
        KID
    }

    private static class EventProcessor {
        private final Map<String, String> processedFamilies =
            new ConcurrentHashMap<>();
        private final RequestContext context;
        private final List<FamilyChangeEvent> events;
        private final String eventId;

        EventProcessor(
            final RequestContext context,
            final List<FamilyChangeEvent> events,
            final String eventId)
        {
            this.context = context;
            this.events = events;
            this.eventId = eventId;
        }

        public void process(int n) {
            while (true) {
                if (n >= events.size()) {
                    AbstractFamilypayCallback.emptyResponse(
                        context,
                        "All events processed");
                    return;
                }
                FamilyChangeEvent event = events.get(n);
                if (processedFamilies.get(event.familyId()) == null) {
                    context.session().logger()
                        .info("Processing event " + event);
                    event.process(
                        new EventProcessingCallback(
                            context.addPrefix(
                                TskvFields.CHANGE_EVENT,
                                event.toString()),
                            this,
                            event,
                            n + 1),
                        eventId + n);
                    return;
                } else {
                    context.session().logger()
                        .info("Skipping event " + event);
                    ++n;
                }
            }
        }
    }

    private static class EventProcessingCallback
        extends AbstractFamilypayCallback<Boolean>
    {
        private final EventProcessor eventProcessor;
        private final FamilyChangeEvent event;
        private final int nextEvent;

        EventProcessingCallback(
            final RequestContext context,
            final EventProcessor eventProcessor,
            final FamilyChangeEvent event,
            final int nextEvent)
        {
            super(context);
            this.eventProcessor = eventProcessor;
            this.event = event;
            this.nextEvent = nextEvent;
        }

        @Override
        public void completed(final Boolean familyProcessed) {
            context.session().logger().info(
                "Event processed. Family processing status = "
                + familyProcessed);
            if (familyProcessed.booleanValue()) {
                eventProcessor.processedFamilies.put(event.familyId(), "");
            }
            eventProcessor.process(nextEvent);
        }
    }
}

