package ru.yandex.passport.familypay.backend;

import java.util.Map;
import java.util.TreeMap;

import io.vertx.sqlclient.Row;
import io.vertx.sqlclient.RowSet;
import io.vertx.sqlclient.Tuple;

import ru.yandex.client.pg.SqlQuery;
import ru.yandex.json.dom.BasicContainerFactory;
import ru.yandex.json.dom.JsonObject;

public class FamilyHandler extends FamilyHandlerBase {
    private static final SqlQuery FAMILY_INFO =
        new SqlQuery("family-info.sql", FamilyHandler.class);
    private static final SqlQuery FAMILY_INFO_BY_ADMIN_UID =
        new SqlQuery("family-info-by-admin-uid.sql", FamilyHandler.class);
    private static final SqlQuery FAMILY_INFO_BY_MEMBER_UID =
        new SqlQuery("family-info-by-member-uid.sql", FamilyHandler.class);

    public FamilyHandler(final FamilypayBackend server) {
        super(server, true);
    }

    @Override
    protected void handle(
        final String familyId,
        final JsonObject payload,
        final RequestContext context)
    {
        findFamily(familyId, new FamilyCallback(context, false));
    }

    public static void findFamily(
        final String familyId,
        final FamilypayCallback<Family> callback)
    {
        callback.context().server().pgClient().executeOnMaster(
            FAMILY_INFO,
            Tuple.of(familyId),
            callback.context().session().listener(),
            new Callback(callback));
    }

    public static void findFamilyByAdminUid(
        final Long adminUid,
        final FamilypayCallback<Family> callback)
    {
        Tuple tuple = Tuple.tuple();
        tuple.addLong(adminUid);
        callback.context().server().pgClient().executeOnMaster(
            FAMILY_INFO_BY_ADMIN_UID,
            tuple,
            callback.context().session().listener(),
            new Callback(callback));
    }

    public static void findFamilyByMemberUid(
        final Long memberUid,
        final FamilypayCallback<Family> callback)
    {
        Tuple tuple = Tuple.tuple();
        tuple.addLong(memberUid);
        callback.context().server().pgClient().executeOnMaster(
            FAMILY_INFO_BY_MEMBER_UID,
            tuple,
            callback.context().session().listener(),
            new Callback(callback));
    }

    private static class Callback
        extends AbstractFilterFamilypayCallback<RowSet<Row>, Family>
    {
        Callback(final FamilypayCallback<? super Family> callback) {
            super(callback);
        }

        @Override
        public void completed(final RowSet<Row> rowSet) {
            try {
                FamilyInfo familyInfo = null;
                Map<Long, FamilyMember> familyMembers = new TreeMap<>();
                for (Row row: rowSet) {
                    if (familyInfo == null) {
                        familyInfo = FamilyInfo.fromRow(row);
                    }
                    FamilyMember familyMember =
                        FamilyMember.fromRow(row, true);
                    long uid = familyMember.uid();
                    familyMembers.put(uid, familyMember);
                }
                if (familyInfo == null) {
                    failed(ErrorType.FAMILY_NOT_FOUND, null, null);
                } else {
                    callback.completed(new Family(familyInfo, familyMembers));
                }
            } catch (RuntimeException e) {
                failed(e);
            }
        }
    }

    public static class FamilyCallback
        extends AbstractFamilypayCallback<Family>
    {
        private final boolean singleLimit;

        FamilyCallback(
            final RequestContext context,
            final boolean singleLimit)
        {
            super(context);
            this.singleLimit = singleLimit;
        }

        @Override
        public void completed(final Family family) {
            sendResponse(
                family.toJson(BasicContainerFactory.INSTANCE, singleLimit));
        }
    }
}

