package ru.yandex.passport.familypay.backend;

import java.util.Set;
import java.util.TreeSet;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.apache.http.HttpEntity;
import org.apache.http.HttpEntityEnclosingRequest;
import org.apache.http.HttpException;
import org.apache.http.HttpRequest;
import org.apache.http.nio.protocol.HttpAsyncExchange;
import org.apache.http.nio.protocol.HttpAsyncRequestConsumer;
import org.apache.http.nio.protocol.HttpAsyncRequestHandler;
import org.apache.http.protocol.HttpContext;

import ru.yandex.http.proxy.BasicProxySession;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.nio.EmptyAsyncConsumer;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumer;
import ru.yandex.json.dom.BasicContainerFactory;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonNull;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.parser.StringCollectorsFactory;
import ru.yandex.json.writer.JsonType;

public abstract class FamilyHandlerBase
    implements HttpAsyncRequestHandler<JsonObject>
{
    private static final Pattern FAMILY_ID =
        Pattern.compile("/family/(f?[0-9]+).*");

    protected final FamilypayBackend server;
    private final boolean quiet;

    protected FamilyHandlerBase(
        final FamilypayBackend server,
        final boolean quiet)
    {
        this.server = server;
        this.quiet = quiet;
    }

    public static Set<String> convertAllowedServices(
        final JsonList allowedServicesList)
        throws JsonException
    {
        Set<String> allowedServices = new TreeSet<>();
        int size = allowedServicesList.size();
        for (int i = 0; i < size; ++i) {
            allowedServices.add(allowedServicesList.get(i).asString());
        }
        allowedServices.add("passport");
        allowedServices.add("payment_sdk");
        return allowedServices;
    }

    @Override
    public HttpAsyncRequestConsumer<JsonObject> processRequest(
        final HttpRequest request,
        final HttpContext context)
        throws HttpException
    {
        if (request instanceof HttpEntityEnclosingRequest) {
            HttpEntity entity =
                ((HttpEntityEnclosingRequest) request).getEntity();
            if (entity.getContentLength() != 0) {
                return new JsonAsyncTypesafeDomConsumer(
                    entity,
                    StringCollectorsFactory.INSTANCE,
                    BasicContainerFactory.INSTANCE);
            }
        }
        return new EmptyAsyncConsumer<>(JsonNull.INSTANCE);
    }

    @Override
    public void handle(
        final JsonObject payload,
        final HttpAsyncExchange exchange,
        final HttpContext context)
        throws HttpException
    {
        ProxySession session =
            new BasicProxySession(server, exchange, context);
        String uri = session.request().getRequestLine().getUri();
        Matcher matcher = FAMILY_ID.matcher(uri);
        String familyId;
        if (matcher.matches()) {
            familyId = matcher.group(1);
            if (Character.isDigit(familyId.charAt(0))) {
                familyId = 'f' + familyId;
            }
        } else {
            throw new BadRequestException(
                "No family_id found in uri <" + uri + '>');
        }
        RequestContext requestContext =
            new RequestContext(
                server,
                session,
                server.tskvLogger().record()
                    .append(TskvFields.HANDLER, getClass().getSimpleName())
                    .append(TskvFields.FAMILY_ID, familyId),
                JsonType.NORMAL.toString(payload),
                quiet);
        try {
            handle(familyId, payload, requestContext);
        } catch (JsonException e) {
            AbstractFamilypayCallback.failed(
                requestContext,
                ErrorType.MALFORMED_REQUEST,
                JsonType.NORMAL.toString(payload),
                e);
        }
    }

    protected abstract void handle(
        String familyId,
        JsonObject payload,
        RequestContext context)
        throws HttpException, JsonException;
}

