package ru.yandex.passport.familypay.backend;

import java.util.Arrays;
import java.util.Objects;
import java.util.Set;
import java.util.TreeSet;

import io.vertx.sqlclient.Row;
import io.vertx.sqlclient.Tuple;

import ru.yandex.function.AbstractStringBuilderable;
import ru.yandex.json.dom.ContainerFactory;
import ru.yandex.json.dom.JsonBoolean;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonLong;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonString;

public class FamilyInfo extends AbstractStringBuilderable {
    private final String familyId;
    private final long adminUid;
    private final CardInfo cardInfo;
    private final String origin;
    private final boolean unlim;
    private final LimitsInfo limitsInfo;
    private final Set<String> allowedServices;

    public FamilyInfo(
        final String familyId,
        final long adminUid,
        final CardInfo cardInfo,
        final String origin,
        final boolean unlim,
        final LimitsInfo limitsInfo,
        final Set<String> allowedServices)
    {
        this.familyId = familyId;
        this.adminUid = adminUid;
        this.cardInfo = cardInfo;
        this.origin = origin;
        this.unlim = unlim;
        this.limitsInfo = limitsInfo;
        this.allowedServices = allowedServices;
    }

    public static FamilyInfo fromRow(final Row row) {
        String[] allowedServicesList =
            row.getArrayOfStrings("default_allowed_services");
        Set<String> allowedServices;
        if (allowedServicesList == null) {
            allowedServices = null;
        } else {
            allowedServices =
                new TreeSet<>(Arrays.asList(allowedServicesList));
        }
        return new FamilyInfo(
            row.getString("family_id"),
            row.getLong("admin_uid"),
            CardInfo.fromRow(row),
            row.getString("origin"),
            row.getBoolean("default_unlim"),
            LimitsInfo.fromRow(row, "default_", "_limit"),
            allowedServices);
    }

    public void toTuple(final Tuple tuple) {
        tuple.addString(familyId);
        tuple.addLong(adminUid);
        cardInfo.toTuple(tuple);
        tuple.addString(origin);
        tuple.addBoolean(unlim);
        limitsInfo.toTuple(tuple);
        String[] allowedServices;
        if (this.allowedServices == null) {
            allowedServices = null;
        } else {
            allowedServices =
                this.allowedServices.toArray(
                    new String[this.allowedServices.size()]);
        }
        tuple.addArrayOfString(allowedServices);
    }

    public static JsonList setToJsonList(
        final ContainerFactory containerFactory,
        final Set<String> set)
    {
        JsonList result = new JsonList(containerFactory, set.size());
        for (String value: set) {
            result.add(new JsonString(value));
        }
        return result;
    }

    public JsonMap toJson(
        final ContainerFactory containerFactory,
        final boolean singleLimit)
    {
        JsonMap result = new JsonMap(containerFactory, 3);
        // familyId intentionally skipped
        result.put("adminUid", new JsonLong(adminUid));
        JsonMap properties = new JsonMap(containerFactory, 4);
        result.put("properties", properties);
        properties.put("cardInfo", cardInfo.toJson(containerFactory));
        properties.put("origin", new JsonString(origin));
        properties.put("defaultUnlim", JsonBoolean.valueOf(unlim));
        properties.put(
            singleLimit ? "limit" : "defaultLimits",
            limitsInfo.toJson(containerFactory, singleLimit, ""));
        if (allowedServices == null) {
            properties.put("defaultAllowAllServices", JsonBoolean.TRUE);
        } else {
            properties.put(
                "defaultAllowedServices",
                setToJsonList(containerFactory, allowedServices));
        }
        return result;
    }

    public String familyId() {
        return familyId;
    }

    public long adminUid() {
        return adminUid;
    }

    public CardInfo cardInfo() {
        return cardInfo;
    }

    public String origin() {
        return origin;
    }

    public boolean unlim() {
        return unlim;
    }

    public LimitsInfo limitsInfo() {
        return limitsInfo;
    }

    public Set<String> allowedServices() {
        return allowedServices;
    }

    @Override
    public int hashCode() {
        return familyId.hashCode()
            ^ Long.hashCode(adminUid)
            ^ cardInfo.hashCode()
            ^ origin.hashCode()
            ^ Boolean.hashCode(unlim)
            ^ limitsInfo.hashCode()
            ^ Objects.hashCode(allowedServices);
    }

    @Override
    public boolean equals(final Object o) {
        if (o instanceof FamilyInfo) {
            FamilyInfo other = (FamilyInfo) o;
            return familyId.equals(other.familyId)
                && adminUid == other.adminUid
                && cardInfo.equals(other.cardInfo)
                && origin.equals(other.origin)
                && unlim == other.unlim
                && limitsInfo.equals(other.limitsInfo)
                && Objects.equals(allowedServices, other.allowedServices);
        }
        return false;
    }

    @Override
    public void toStringBuilder(final StringBuilder sb) {
        sb.append("FamilyInfo(");
        sb.append(familyId);
        sb.append(',');
        sb.append(adminUid);
        sb.append(',');
        cardInfo.toStringBuilder(sb);
        sb.append(",origin=");
        sb.append(origin);
        sb.append(",unlim=");
        sb.append(unlim);
        sb.append(',');
        limitsInfo.toStringBuilder(sb);
        sb.append(",allowedServices=");
        sb.append(allowedServices);
        sb.append(')');
    }
}

