package ru.yandex.passport.familypay.backend;

import java.util.Arrays;
import java.util.Set;
import java.util.TreeSet;

import io.vertx.sqlclient.Row;

import ru.yandex.function.AbstractStringBuilderable;
import ru.yandex.json.dom.ContainerFactory;
import ru.yandex.json.dom.JsonBoolean;
import ru.yandex.json.dom.JsonLong;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonString;

public class FamilyMember extends AbstractStringBuilderable {
    private final long uid;
    private final boolean enabled;
    private final boolean unlim;
    private final boolean hasSecurePhone;
    private final LimitsInfo limits;
    private final Set<String> allowedServices;
    private final LimitsInfo expenses;
    private final String limitCurrency;

    private FamilyMember(
        final long uid,
        final boolean enabled,
        final boolean unlim,
        final boolean hasSecurePhone,
        final String limitCurrency,
        final LimitsInfo limits,
        final Set<String> allowedServices,
        final LimitsInfo expenses)
    {
        this.uid = uid;
        this.enabled = enabled;
        this.unlim = unlim;
        this.hasSecurePhone = hasSecurePhone;
        this.limitCurrency = limitCurrency;
        this.limits = limits;
        this.allowedServices = allowedServices;
        this.expenses = expenses;
    }

    public static FamilyMember fromRow(
        final Row row,
        final boolean parseExpenses)
    {
        return fromRow(row, row.getLong("uid"), parseExpenses);
    }

    public static FamilyMember fromRow(
        final Row row,
        final long uid,
        final boolean parseExpenses)
    {
        LimitsInfo expenses;
        if (parseExpenses) {
            expenses = LimitsInfo.fromRow(row, "", "_amount");
        } else {
            expenses = null;
        }
        String[] allowedServicesList =
            row.getArrayOfStrings("allowed_services");
        Set<String> allowedServices;
        if (allowedServicesList == null) {
            allowedServices = null;
        } else {
            allowedServices =
                new TreeSet<>(Arrays.asList(allowedServicesList));
        }
        return new FamilyMember(
            uid,
            row.getBoolean("enabled"),
            row.getBoolean("unlim"),
            row.getBoolean("has_secure_phone"),
            row.getString("limit_currency"),
            LimitsInfo.fromRow(row, "", "_limit"),
            allowedServices,
            expenses);
    }

    public static boolean familypayEnabled(final FamilyMember familyMember) {
        return familyMember != null
            && familyMember.enabled
            && familyMember.hasSecurePhone
            && familyMember.limits.day() > 0L
            && familyMember.limits.week() > 0L
            && familyMember.limits.month() > 0L
            && familyMember.limits.total() > 0L;
    }

    public JsonMap toJson(
        final ContainerFactory containerFactory,
        final boolean singleLimit)
    {
        JsonMap result = new JsonMap(containerFactory, 7);
        result.put("uid", new JsonLong(uid));
        result.put("enabled", JsonBoolean.valueOf(enabled));
        result.put("unlim", JsonBoolean.valueOf(unlim));
        result.put("hasSecurePhone", JsonBoolean.valueOf(hasSecurePhone));
        result.put("limitCurrency", new JsonString(limitCurrency));
        result.put(
            singleLimit ? "limit" : "limits",
            limits.toJson(containerFactory, singleLimit, ""));
        if (allowedServices == null) {
            result.put("allowAllServices", JsonBoolean.TRUE);
        } else {
            result.put(
                "allowedServices",
                FamilyInfo.setToJsonList(containerFactory, allowedServices));
        }
        if (expenses != null) {
            JsonMap expensesMap;
            if (singleLimit) {
                LimitMode limitMode = LimitMode.detectLimitMode(limits);
                expensesMap = new JsonMap(containerFactory, 2);
                expensesMap.put(
                    "amount",
                    new JsonLong(limitMode.limitValue(expenses)));
            } else {
                expensesMap =
                    expenses.toJson(containerFactory, false, "Amount");
            }
            result.put("expenses", expensesMap);
        }
        return result;
    }

    public long uid() {
        return uid;
    }

    public boolean enabled() {
        return enabled;
    }

    public boolean unlim() {
        return unlim;
    }

    public boolean hasSecurePhone() {
        return hasSecurePhone;
    }

    public String limitCurrency() {
        return limitCurrency;
    }

    public LimitsInfo limits() {
        return limits;
    }

    public Set<String> allowedServices() {
        return allowedServices;
    }

    public LimitsInfo expenses() {
        return expenses;
    }

    @Override
    public void toStringBuilder(final StringBuilder sb) {
        sb.append("FamilyMember(");
        sb.append(uid);
        sb.append(",enabled=");
        sb.append(enabled);
        sb.append(",unlim=");
        sb.append(unlim);
        sb.append(",hasSecurePhone=");
        sb.append(hasSecurePhone);
        sb.append(",limitCurrency=");
        sb.append(limitCurrency);
        sb.append(",limits=");
        limits.toStringBuilder(sb);
        sb.append(",allowedServices=");
        sb.append(allowedServices);
        sb.append(",expenses=");
        if (expenses == null) {
            sb.append("null");
        } else {
            expenses.toStringBuilder(sb);
        }
        sb.append(')');
    }
}

